// Copyright 2018 Adam S Levy
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

package jsonrpc2

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

var requestTests = []struct {
	Name   string
	Req    *Request
	Params interface{}
	Data   string
	Err    string
}{{
	Name:   "Request",
	Req:    &Request{ID: 5, Method: "method", Params: &[]int{0, 1}},
	Params: &[]int{},
	Data:   `{"jsonrpc":"2.0","method":"method","params":[0,1],"id":5}`,
}, {
	Name:   "Notification",
	Req:    &Request{Method: "method", Params: &struct{ D string }{D: "hi"}},
	Params: &struct{ D string }{},
	Data:   `{"jsonrpc":"2.0","method":"method","params":{"D":"hi"}}`,
}, {
	Name: "bad JSON-RPC version",
	Data: `{"method":"method","params":{"D":"hi"}}`,
	Err:  "invalid JSON-RPC 2.0 version",
}}

func TestRequest(t *testing.T) {
	t.Run("MarshalJSON", func(t *testing.T) {
		for _, test := range requestTests {
			if test.Req == nil {
				continue
			}
			t.Run(test.Name, func(t *testing.T) {
				assert := assert.New(t)
				data, err := test.Req.MarshalJSON()
				assert.NoError(err)
				assert.Equal(test.Data, string(data))
			})
		}
	})
}
