// Unless explicitly stated otherwise all files in this repository are licensed
// under the Apache License Version 2.0.
// This product includes software developed at Datadog (https://www.datadoghq.com/).
// Copyright 2016-present Datadog, Inc.

package stats

import (
	"fmt"
	"testing"
	"time"

	"github.com/DataDog/datadog-agent/pkg/trace/traceutil"

	pb "github.com/DataDog/datadog-agent/pkg/proto/pbgo/trace"

	"github.com/stretchr/testify/assert"
)

func TestGrain(t *testing.T) {
	assert := assert.New(t)
	s := StatSpan{service: "thing", name: "other", resource: "yo"}
	aggr := NewAggregationFromSpan(&s, "", PayloadAggregationKey{
		Env:         "default",
		Hostname:    "default",
		ContainerID: "cid",
	})
	assert.Equal(Aggregation{
		PayloadAggregationKey: PayloadAggregationKey{
			Env:         "default",
			Hostname:    "default",
			ContainerID: "cid",
		},
		BucketsAggregationKey: BucketsAggregationKey{
			Service:     "thing",
			Name:        "other",
			Resource:    "yo",
			IsTraceRoot: pb.Trilean_TRUE,
		},
	}, aggr)
}

func TestGrainWithPeerTags(t *testing.T) {
	sc := &SpanConcentrator{}
	t.Run("none present", func(t *testing.T) {
		assert := assert.New(t)
		s, _ := sc.NewStatSpanWithConfig(StatSpanConfig{"thing", "yo", "other", "", 0, 0, 0, 0, map[string]string{"span.kind": "client"}, map[string]float64{"_dd.measured": 1}, []string{"aws.s3.bucket", "db.instance", "db.system", "peer.service"}, "", ""})
		aggr := NewAggregationFromSpan(s, "", PayloadAggregationKey{
			Env:         "default",
			Hostname:    "default",
			ContainerID: "cid",
		})

		assert.Equal(Aggregation{
			PayloadAggregationKey: PayloadAggregationKey{
				Env:         "default",
				Hostname:    "default",
				ContainerID: "cid",
			},
			BucketsAggregationKey: BucketsAggregationKey{
				Service:     "thing",
				SpanKind:    "client",
				Name:        "other",
				Resource:    "yo",
				IsTraceRoot: pb.Trilean_TRUE,
			},
		}, aggr)
	})
	t.Run("computeBySpanKind config", func(t *testing.T) {
		for _, spanKindEnabled := range []bool{true, false} {
			t.Run(fmt.Sprintf("%t", spanKindEnabled), func(t *testing.T) {
				assert := assert.New(t)
				sci := NewSpanConcentrator(&SpanConcentratorConfig{
					ComputeStatsBySpanKind: spanKindEnabled,
					BucketInterval:         (time.Duration(10) * time.Second).Nanoseconds(),
				}, time.Now().Add(-time.Minute))
				s, _ := sci.NewStatSpanWithConfig(StatSpanConfig{"thing", "yo", "other", "", 0, 0, 0, 0, map[string]string{"span.kind": "client", "server.address": "foo"}, nil, []string{"_dd.base_service", "server.address"}, "", ""})
				if spanKindEnabled {
					assert.Equal([]string{"server.address:foo"}, s.matchingPeerTags)
				} else {
					assert.Nil(s)
				}
			})
		}
	})
	t.Run("service override", func(t *testing.T) {
		for _, spanKind := range []string{"client", "internal"} {
			t.Run(spanKind, func(t *testing.T) {
				assert := assert.New(t)
				s, _ := sc.NewStatSpanWithConfig(StatSpanConfig{"thing", "yo", "other", "", 0, 0, 0, 0, map[string]string{"span.kind": spanKind, "_dd.base_service": "the-real-base", "server.address": "foo"}, map[string]float64{"_dd.measured": 1}, []string{"_dd.base_service", "server.address"}, "", ""})
				if spanKind == "client" {
					assert.Equal([]string{"_dd.base_service:the-real-base", "server.address:foo"}, s.matchingPeerTags)
				} else {
					assert.Equal([]string{"_dd.base_service:the-real-base"}, s.matchingPeerTags)
				}
			})
		}
	})
	t.Run("partially present", func(t *testing.T) {
		assert := assert.New(t)
		meta := map[string]string{"span.kind": "client", "peer.service": "aws-s3", "aws.s3.bucket": "bucket-a"}
		s, _ := sc.NewStatSpanWithConfig(StatSpanConfig{"thing", "yo", "other", "", 0, 0, 0, 0, meta, map[string]float64{"_dd.measured": 1}, []string{"aws.s3.bucket", "db.instance", "db.system", "peer.service"}, "", ""})

		aggr := NewAggregationFromSpan(s, "", PayloadAggregationKey{
			Env:         "default",
			Hostname:    "default",
			ContainerID: "cid",
		})

		assert.Equal(Aggregation{
			PayloadAggregationKey: PayloadAggregationKey{
				Env:         "default",
				Hostname:    "default",
				ContainerID: "cid",
			},
			BucketsAggregationKey: BucketsAggregationKey{
				Service:      "thing",
				SpanKind:     "client",
				Name:         "other",
				Resource:     "yo",
				PeerTagsHash: 13698082192712149795,
				IsTraceRoot:  pb.Trilean_TRUE,
			},
		}, aggr)
	})
	t.Run("peer ip quantization", func(t *testing.T) {
		assert := assert.New(t)
		meta := map[string]string{"span.kind": "client", "server.address": "129.49.218.65"}
		s, _ := sc.NewStatSpanWithConfig(StatSpanConfig{"thing", "yo", "other", "", 0, 0, 0, 0, meta, map[string]float64{"_dd.measured": 1}, []string{"server.address"}, "", ""})

		aggr := NewAggregationFromSpan(s, "", PayloadAggregationKey{
			Env:         "default",
			Hostname:    "default",
			ContainerID: "cid",
		})

		assert.Equal(Aggregation{
			PayloadAggregationKey: PayloadAggregationKey{
				Env:         "default",
				Hostname:    "default",
				ContainerID: "cid",
			},
			BucketsAggregationKey: BucketsAggregationKey{
				Service:      "thing",
				SpanKind:     "client",
				Name:         "other",
				Resource:     "yo",
				PeerTagsHash: 0xad02dc568e7330c5,
				IsTraceRoot:  pb.Trilean_TRUE,
			},
		}, aggr)
		assert.Equal([]string{"server.address:blocked-ip-address"}, s.matchingPeerTags)
	})
	t.Run("all present", func(t *testing.T) {
		assert := assert.New(t)
		meta := map[string]string{"span.kind": "client", "peer.service": "aws-dynamodb", "db.instance": "dynamo.test.us1", "db.system": "dynamodb"}
		s, _ := sc.NewStatSpanWithConfig(StatSpanConfig{"thing", "yo", "other", "", 0, 0, 0, 0, meta, map[string]float64{"_dd.measured": 1}, []string{"aws.s3.bucket", "db.instance", "db.system", "peer.service"}, "", ""})

		aggr := NewAggregationFromSpan(s, "", PayloadAggregationKey{
			Env:         "default",
			Hostname:    "default",
			ContainerID: "cid",
		})

		assert.Equal(Aggregation{
			PayloadAggregationKey: PayloadAggregationKey{
				Env:         "default",
				Hostname:    "default",
				ContainerID: "cid",
			},
			BucketsAggregationKey: BucketsAggregationKey{
				Service:      "thing",
				SpanKind:     "client",
				Name:         "other",
				Resource:     "yo",
				PeerTagsHash: 5537613849774405073,
				IsTraceRoot:  pb.Trilean_TRUE,
			},
		}, aggr)
		assert.Equal([]string{"db.instance:dynamo.test.us1", "db.system:dynamodb", "peer.service:aws-dynamodb"}, s.matchingPeerTags)
	})
}

func TestGrainWithSynthetics(t *testing.T) {
	assert := assert.New(t)
	sc := &SpanConcentrator{}
	meta := map[string]string{traceutil.TagStatusCode: "418"}
	s, _ := sc.NewStatSpanWithConfig(StatSpanConfig{"thing", "yo", "other", "", 0, 0, 0, 0, meta, map[string]float64{"_dd.measured": 1}, nil, "", ""})

	aggr := NewAggregationFromSpan(s, "synthetics-browser", PayloadAggregationKey{
		Hostname:    "host-id",
		Version:     "v0",
		Env:         "default",
		ContainerID: "cid",
	})

	assert.Equal(Aggregation{
		PayloadAggregationKey: PayloadAggregationKey{
			Hostname:    "host-id",
			Version:     "v0",
			Env:         "default",
			ContainerID: "cid",
		},
		BucketsAggregationKey: BucketsAggregationKey{
			Service:     "thing",
			Resource:    "yo",
			Name:        "other",
			StatusCode:  418,
			Synthetics:  true,
			IsTraceRoot: pb.Trilean_TRUE,
		},
	}, aggr)
}

func BenchmarkHandleSpanRandom(b *testing.B) {
	sc := NewSpanConcentrator(&SpanConcentratorConfig{}, time.Now())
	b.Run("no_peer_tags", func(b *testing.B) {
		sb := NewRawBucket(0, 1e9)
		var benchStatSpans []*StatSpan
		for _, s := range benchSpans {
			statSpan, ok := sc.NewStatSpanFromPB(s, nil)
			assert.True(b, ok, "Statically defined benchmark spans should require stats")
			benchStatSpans = append(benchStatSpans, statSpan)
		}
		b.ResetTimer()
		b.ReportAllocs()
		for i := 0; i < b.N; i++ {
			for _, span := range benchStatSpans {
				sb.HandleSpan(span, 1, "", PayloadAggregationKey{Env: "a", Hostname: "b", Version: "c", ContainerID: "d"})
			}
		}
	})
	// This is copied from comp/trace/config/peer_tags_test.go
	// The actual values are not necessarily relevant for the benchmark
	// but we should update them periodically.
	peerTags := []string{
		"_dd.base_service",
		"amqp.destination",
		"amqp.exchange",
		"amqp.queue",
		"aws.queue.name",
		"aws.s3.bucket",
		"bucketname",
		"cassandra.keyspace",
		"db.cassandra.contact.points",
		"db.couchbase.seed.nodes",
		"db.hostname",
		"db.instance",
		"db.name",
		"db.namespace",
		"db.system",
		"grpc.host",
		"hostname",
		"http.host",
		"http.server_name",
		"messaging.destination",
		"messaging.destination.name",
		"messaging.kafka.bootstrap.servers",
		"messaging.rabbitmq.exchange",
		"messaging.system",
		"mongodb.db",
		"msmq.queue.path",
		"net.peer.name",
		"network.destination.name",
		"peer.hostname",
		"peer.service",
		"queuename",
		"rpc.service",
		"rpc.system",
		"server.address",
		"streamname",
		"tablename",
		"topicname",
	}
	b.Run("peer_tags", func(b *testing.B) {
		sb := NewRawBucket(0, 1e9)
		var benchStatSpans []*StatSpan
		for _, s := range benchSpans {
			statSpan, ok := sc.NewStatSpanFromPB(s, peerTags)
			assert.True(b, ok, "Statically defined benchmark spans should require stats")
			benchStatSpans = append(benchStatSpans, statSpan)
		}
		b.ResetTimer()
		b.ReportAllocs()
		for i := 0; i < b.N; i++ {
			for _, span := range benchStatSpans {
				sb.HandleSpan(span, 1, "", PayloadAggregationKey{Env: "a", Hostname: "b", Version: "c", ContainerID: "d"})
			}
		}
	})
}

var benchSpans = []*pb.Span{
	{
		Service:  "rails",
		Name:     "web.template",
		Resource: "SELECT user.handle AS user_handle, user.id AS user_id, user.org_id AS user_org_id, user.password AS user_password, user.email AS user_email, user.name AS user_name, user.role AS user_role, user.team AS user_team, user.support AS user_support, user.is_admin AS user_is_admin, user.github_username AS user_github_username, user.github_token AS user_github_token, user.disabled AS user_disabled, user.verified AS user_verified, user.bot AS user_bot, user.created AS user_created, user.modified AS user_modified, user.time_zone AS user_time_zone, user.password_modified AS user_password_modified FROM user WHERE user.id = ? AND user.org_id = ? LIMIT ?",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x3fd1ce2fbc1dde9e,
		ParentID: 0x55acf95eafb06955,
		Start:    1548931840954169000,
		Duration: 100000000,
		Error:    403,
		Meta:     map[string]string{"query": "SELECT id\n                 FROM ddsuperuser\n                WHERE id = %(id)s", "db.hostname": "db.host.us1.prod", "db.name": "postgres"},
		Metrics:  map[string]float64{"rowcount": 0.5066325669281033},
		Type:     "",
	},
	{
		Service:  "pg-master",
		Name:     "postgres.query",
		Resource: "データの犬",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x57be126d97c3eed2,
		ParentID: 0x3fd1ce2fbc1dde9e,
		Start:    1548931841019932928,
		Duration: 19844796,
		Error:    400,
		Meta:     map[string]string{"user": "leo", "db.hostname:": "db.host.us1.prod", "db.name": "postgres"},
		Metrics:  map[string]float64{"size": 0.47564235466940796, "rowcount": 0.12453347154800333},
		Type:     "lamar",
	},
	{
		Service:  "rails",
		Name:     "sqlalchemy",
		Resource: "GET cache|xxx",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x61973c4d43bd8f04,
		ParentID: 0x3fd1ce2fbc1dde9e,
		Start:    1548931840963747104,
		Duration: 3566171,
		Error:    0,
		Meta:     map[string]string{"in.host": "8.8.8.8", "query": "GET beaker:c76db4c3af90410197cf88b0afba4942:session", "db.hostname:": "db.host.us1.prod", "db.name": "postgres"},
		Metrics:  map[string]float64{"rowcount": 0.276209049435507, "size": 0.18889910131880996},
		Type:     "redis",
	},
	{
		Service:  "pylons",
		Name:     "postgres.query",
		Resource: "events.buckets",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x4541e015c8c62f79,
		ParentID: 0x3fd1ce2fbc1dde9e,
		Start:    1548931840954371301,
		Duration: 259245,
		Error:    502,
		Meta:     map[string]string{"db.hostname:": "db.host.us1.prod", "db.name": "postgres", "query": "\n        -- get_contexts_sub_query[[org:9543 query_id:a135e15e7d batch:1]]\n        WITH sub_contexts as (\n            \n        -- \n        --\n        SELECT key,\n            host_name,\n            device_name,\n            tags,\n            org_id\n        FROM vs9543.dim_context c\n        WHERE key = ANY(%(key)s)\n        \n        \n        \n        \n    \n        )\n        \n        -- \n        --\n        SELECT key,\n            host_name,\n            device_name,\n            tags\n        FROM sub_contexts c\n        WHERE (c.org_id = %(org_id)s AND c.tags @> %(yes_tags0)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags1)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags2)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags3)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags4)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags5)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags6)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags7)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags8)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags9)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags10)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags11)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags12)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags13)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags14)s)\n        OR (c.org_id = %(org_id)s AND c.tags @> %(yes_tags15)s)\n        \n        \n        \n        \n    \n        "},
		Metrics:  map[string]float64{"rowcount": 0.5543063276573277, "size": 0.6196504333337066, "payloads": 0.9689311094466356},
		Type:     "lamar",
	},
	{
		Service:  "rails",
		Name:     "postgres.query",
		Resource: "データの犬",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x273710f0da9967a7,
		ParentID: 0x3fd1ce2fbc1dde9e,
		Start:    1548931840954749862,
		Duration: 161372,
		Error:    0,
		Meta:     map[string]string{"out.section": "-", "db.hostname:": "db.host.us1.prod", "db.name": "postgres"},
		Metrics:  map[string]float64{"rowcount": 0.2646545763337349},
		Type:     "lamar",
	},
	{
		Service:  "web-billing",
		Name:     "web.query",
		Resource: "GET /url/test/fixture/resource/42",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x69ff3ac466831715,
		ParentID: 0x3fd1ce2fbc1dde9e,
		Start:    1548931840954191909,
		Duration: 9908,
		Error:    0,
		Meta:     map[string]string{"peer.service": "foo", "net.peer.name": "foo.us1", "network.destination.name": "foo.us1.12345"},
		Metrics:  map[string]float64{"rowcount": 0.7800384694533715, "payloads": 0.24585482170573683, "loops": 0.3119738365111953, "size": 0.6693070719377765},
		Type:     "sql",
	},
	{
		Service:  "pg-master",
		Name:     "sqlalchemy",
		Resource: "データの犬",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x27dea5ee886c9fbb,
		ParentID: 0x3fd1ce2fbc1dde9e,
		Start:    1548931840954175872,
		Duration: 2635,
		Error:    400,
		Meta:     map[string]string{"user": "benjamin", "query": "GET beaker:c76db4c3af90410197cf88b0afba4942:session", "db.hostname:": "db.host.us1.prod", "db.name": "postgres"},
		Metrics:  map[string]float64{"payloads": 0.5207323287655542, "loops": 0.4731462684058845, "heap_allocated": 0.5386526456622786, "size": 0.9438291624690298, "rowcount": 0.14536182482282964},
		Type:     "lamar",
	},
	{
		Service:  "django",
		Name:     "pylons.controller",
		Resource: "データの犬",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x3d34aa36af4e081f,
		ParentID: 0x3fd1ce2fbc1dde9e,
		Start:    1548931840954169013,
		Duration: 370,
		Error:    400,
		Meta:     map[string]string{"db.hostname:": "db.host.us1.prod", "db.name": "postgres", "user": "leo", "query": "SELECT id\n                 FROM ddsuperuser\n                WHERE id = %(id)s"},
		Metrics:  map[string]float64{},
		Type:     "lamar",
	},
	{
		Service:  "django",
		Name:     "grpc.client.request",
		Resource: "events.buckets",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x3a51491c82d0b322,
		ParentID: 0x69ff3ac466831715,
		Start:    1548931840954198336,
		Duration: 2474,
		Error:    1,
		Meta:     map[string]string{"rpc.service": "buckets", "out.host": "baz", "net.peer.name": "baz.us1", "network.destination.name": "baz.us1.12345"},
		Metrics:  map[string]float64{"rowcount": 0.9895177718616301},
		Type:     "lamar",
	},
	{
		Service:  "django",
		Name:     "postgres.query",
		Resource: "SELECT id FROM table;",
		TraceID:  0x5df0afd382d351de,
		SpanID:   0x3fd1ce2fbc1dde9e,
		ParentID: 0x3a51491c82d0b322,
		Start:    1548931840954169000,
		Duration: 100000000,
		Error:    403,
		Meta:     map[string]string{"query": "SELECT id\n                 FROM ddsuperuser\n                WHERE id = %(id)s", "db.hostname": "db.host.us1.prod", "db.name": "postgres"},
		Metrics:  map[string]float64{"rowcount": 0.5066325669281033},
		Type:     "db",
	},
}

const roundMask int64 = 1 << 10

func oldNSTimestampToFloat(ns int64) float64 {
	var shift uint
	for ns > roundMask {
		ns = ns >> 1
		shift++
	}
	return float64(ns << shift)
}

func TestNSTimestampToFloat(t *testing.T) {
	ns := []int64{
		int64(1066789584153112 - 1066789583298779), // kernel boot time values
		int64(0),
		int64(1),
		int64(1066789584153112),
		int64(time.Hour * 24 * 3650), // 10 year
		int64(time.Now().UnixNano()),
		int64(0x000000000000ffff),
		int64(1023),
		int64(1024),
		int64(1025),
		//^int64(0), this can't be used here because float64 have only 52 bits of mantissa
		// and filter(float(int64)) will difference due to roundup than float(filter(int64))
		int64(0x001fffffffffffff),
		^int64(0x001fffffffffffff), // ~584 years
	}

	for _, n := range ns {
		assert.Equal(t, oldNSTimestampToFloat(n), nsTimestampToFloat(n), "uint64 10 bits mantissa truncation failed "+fmt.Sprintf("%d 0x%x", n, n))
	}
}
