// Copyright (c) HashiCorp, Inc.
// SPDX-License-Identifier: MPL-2.0

package tfe

import (
	"context"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestStackDeploymentGroupsList(t *testing.T) {
	skipUnlessBeta(t)

	client := testClient(t)
	ctx := context.Background()

	orgTest, orgTestCleanup := createOrganization(t, client)
	t.Cleanup(orgTestCleanup)

	oauthClient, cleanup := createOAuthClient(t, client, orgTest, nil)
	t.Cleanup(cleanup)

	stack, err := client.Stacks.Create(ctx, StackCreateOptions{
		Name: "test-stack",
		VCSRepo: &StackVCSRepoOptions{
			Identifier:   "hashicorp-guides/pet-nulls-stack",
			OAuthTokenID: oauthClient.OAuthTokens[0].ID,
		},
		Project: &Project{
			ID: orgTest.DefaultProject.ID,
		},
	})

	require.NoError(t, err)
	require.NotNil(t, stack)

	stackUpdated, err := client.Stacks.FetchLatestFromVcs(ctx, stack.ID)
	require.NoError(t, err)
	require.NotNil(t, stackUpdated)

	stackUpdated = pollStackDeploymentGroups(t, ctx, client, stackUpdated.ID)
	require.NotEmpty(t, stackUpdated.LatestStackConfiguration.ID)

	t.Run("List with valid stack configuration ID", func(t *testing.T) {
		sdgl, err := client.StackDeploymentGroups.List(ctx, stackUpdated.LatestStackConfiguration.ID, nil)
		require.NoError(t, err)
		require.NotNil(t, sdgl)
		for _, item := range sdgl.Items {
			assert.NotNil(t, item.ID)
			assert.NotEmpty(t, item.Name)
			assert.NotEmpty(t, item.Status)
			assert.NotNil(t, item.CreatedAt)
			assert.NotNil(t, item.UpdatedAt)
		}
		require.Len(t, sdgl.Items, 2)
	})

	t.Run("List with invalid stack configuration ID", func(t *testing.T) {
		_, err := client.StackDeploymentGroups.List(ctx, "", nil)
		require.Error(t, err)
	})

	t.Run("List with pagination", func(t *testing.T) {
		options := &StackDeploymentGroupListOptions{
			ListOptions: ListOptions{
				PageNumber: 1,
				PageSize:   1,
			},
		}
		sdgl, err := client.StackDeploymentGroups.List(ctx, stackUpdated.LatestStackConfiguration.ID, options)
		require.NoError(t, err)
		require.NotNil(t, sdgl)
		require.Len(t, sdgl.Items, 1)
	})
}

func TestStackDeploymentGroupsRead(t *testing.T) {
	skipUnlessBeta(t)

	client := testClient(t)
	ctx := context.Background()

	orgTest, orgTestCleanup := createOrganization(t, client)
	t.Cleanup(orgTestCleanup)

	oauthClient, cleanup := createOAuthClient(t, client, orgTest, nil)
	t.Cleanup(cleanup)

	stack, err := client.Stacks.Create(ctx, StackCreateOptions{
		Name: "test-stack",
		VCSRepo: &StackVCSRepoOptions{
			Identifier:   "hashicorp-guides/pet-nulls-stack",
			OAuthTokenID: oauthClient.OAuthTokens[0].ID,
		},
		Project: &Project{
			ID: orgTest.DefaultProject.ID,
		},
	})

	require.NoError(t, err)
	require.NotNil(t, stack)

	stackUpdated, err := client.Stacks.FetchLatestFromVcs(ctx, stack.ID)
	require.NoError(t, err)
	require.NotNil(t, stackUpdated)

	stackUpdated = pollStackDeploymentGroups(t, ctx, client, stackUpdated.ID)
	require.NotNil(t, stackUpdated.LatestStackConfiguration)

	sdgl, err := client.StackDeploymentGroups.List(ctx, stackUpdated.LatestStackConfiguration.ID, nil)
	require.NoError(t, err)
	require.NotNil(t, sdgl)
	require.Len(t, sdgl.Items, 2)

	t.Run("Read with valid ID", func(t *testing.T) {
		sdgRead, err := client.StackDeploymentGroups.Read(ctx, sdgl.Items[0].ID)
		require.NoError(t, err)
		assert.Equal(t, sdgl.Items[0].ID, sdgRead.ID)
		assert.Equal(t, sdgl.Items[0].Name, sdgRead.Name)
		assert.Equal(t, sdgl.Items[0].Status, sdgRead.Status)
	})

	t.Run("Read with invalid ID", func(t *testing.T) {
		_, err := client.StackDeploymentGroups.Read(ctx, "")
		require.Error(t, err)
	})
}

func TestStackDeploymentGroupsApproveAllPlans(t *testing.T) {
	skipUnlessBeta(t)

	client := testClient(t)
	ctx := context.Background()

	orgTest, orgTestCleanup := createOrganization(t, client)
	t.Cleanup(orgTestCleanup)

	oauthClient, cleanup := createOAuthClient(t, client, orgTest, nil)
	t.Cleanup(cleanup)

	stack, err := client.Stacks.Create(ctx, StackCreateOptions{
		Name: "test-stack",
		VCSRepo: &StackVCSRepoOptions{
			Identifier:   "hashicorp-guides/pet-nulls-stack",
			OAuthTokenID: oauthClient.OAuthTokens[0].ID,
			Branch:       "main",
		},
		Project: &Project{
			ID: orgTest.DefaultProject.ID,
		},
	})
	require.NoError(t, err)
	require.NotNil(t, stack)

	stackUpdated, err := client.Stacks.FetchLatestFromVcs(ctx, stack.ID)
	require.NoError(t, err)
	require.NotNil(t, stackUpdated)

	stackUpdated = pollStackDeploymentGroups(t, ctx, client, stackUpdated.ID)
	require.NotNil(t, stackUpdated.LatestStackConfiguration)

	// Get the deployment group ID from the stack configuration
	deploymentGroups, err := client.StackDeploymentGroups.List(ctx, stackUpdated.LatestStackConfiguration.ID, nil)
	require.NoError(t, err)
	require.NotNil(t, deploymentGroups)
	require.NotEmpty(t, deploymentGroups.Items)

	deploymentGroupID := deploymentGroups.Items[0].ID

	t.Run("Approving all plans", func(t *testing.T) {
		err := client.StackDeploymentGroups.ApproveAllPlans(ctx, deploymentGroupID)
		require.NoError(t, err)
	})
}

func TestStackDeploymentGroupsRerun(t *testing.T) {
	skipUnlessBeta(t)

	client := testClient(t)
	ctx := context.Background()

	orgTest, orgTestCleanup := createOrganization(t, client)
	t.Cleanup(orgTestCleanup)

	oauthClient, cleanup := createOAuthClient(t, client, orgTest, nil)
	t.Cleanup(cleanup)

	stack, err := client.Stacks.Create(ctx, StackCreateOptions{
		Name: "test-stack",
		VCSRepo: &StackVCSRepoOptions{
			Identifier:   "hashicorp-guides/pet-nulls-stack",
			OAuthTokenID: oauthClient.OAuthTokens[0].ID,
			Branch:       "main",
		},
		Project: &Project{
			ID: orgTest.DefaultProject.ID,
		},
	})
	require.NoError(t, err)
	require.NotNil(t, stack)

	stackUpdated, err := client.Stacks.FetchLatestFromVcs(ctx, stack.ID)
	require.NoError(t, err)
	require.NotNil(t, stackUpdated)

	stackUpdated = pollStackDeploymentGroups(t, ctx, client, stackUpdated.ID)
	require.NotNil(t, stackUpdated.LatestStackConfiguration)

	deploymentGroups, err := client.StackDeploymentGroups.List(ctx, stackUpdated.LatestStackConfiguration.ID, nil)
	require.NoError(t, err)
	require.NotNil(t, deploymentGroups)
	require.NotEmpty(t, deploymentGroups.Items)

	deploymentGroupID := deploymentGroups.Items[0].ID

	deploymentRuns, err := client.StackDeploymentRuns.List(ctx, deploymentGroupID, nil)
	require.NoError(t, err)
	require.NotNil(t, deploymentRuns)
	require.NotEmpty(t, deploymentRuns.Items)

	err = client.StackDeploymentGroups.ApproveAllPlans(ctx, deploymentGroupID)
	require.NoError(t, err)

	pollStackDeploymentRunStatus(t, ctx, client, deploymentRuns.Items[0].ID, "deploying")

	deploymentRunIds := []string{deploymentRuns.Items[0].ID}
	for _, dr := range deploymentRuns.Items {
		deploymentRunIds = append(deploymentRunIds, dr.ID)
	}

	t.Run("No deployments specified for rerun", func(t *testing.T) {
		err := client.StackDeploymentGroups.Rerun(ctx, deploymentGroupID, nil)
		require.Error(t, err)
		assert.Contains(t, err.Error(), "no deployments specified for rerun")
	})

	t.Run("Rerun with invalid ID", func(t *testing.T) {
		err := client.StackDeploymentGroups.Rerun(ctx, "", &StackDeploymentGroupRerunOptions{
			Deployments: deploymentRunIds,
		})
		require.Error(t, err)
		assert.Contains(t, err.Error(), "invalid stack deployment group ID")
	})
}
