/*
 * tmain.c - ProIcon version
 *	     main control of translation, linking, and execution.
 */

#include <WindowMgr.h>
#include <MenuMgr.h>
#include <EventMgr.h>
#include <StdFilePkg.h>
#include "compiler.h"
#include <setjmp.h>
#include <ctype.h>
#include "config.h"
#include "general.h"
#include "header.h"
#include "iproto.h"
#include "pedit.h"
#include "rproto.h"
#include "rt.h"
#include "tproto.h"

/********************************** GLOBALS ****************************/

#define Global
#define Init(v) = v
#include "globals.h"

typedef struct resinfo {
   ResType type;
   int	   id;
   } resinfo;

/********************************* EXTERNALS ***************************/
extern CompResult	compResult;	/* return values from compiler */
extern Boolean 		AutoLinkOn, AutoRunOn;
extern WindInfoPtr	runWptr;
extern int		findline();	/* prototype problems */
extern char 		*findfile();	/* prototype problems */


/***************************** FORWARD REFERENCES **********************/
static novalue closerf(int myRefNum);
static int     docopy(ResType type, resinfo excres[], int fromRef, int toRef);
static novalue fetchopts(char *fname);
static novalue mfree(void);
static novalue saveopts(void);
static novalue unpackMem(void);
static novalue unpackFlags(void);
static novalue deleteucode(void);
static novalue parsecl(char *cp);
static novalue errcommon(int, char*, long, long, char*, char*, char*, char*);

/********************************* STATICS *****************************/
static CompOpt compOpt;
static Boolean ignorefile;
static Boolean upreserve;

static resinfo excludeu[] = {		/* Resources to exclude when copying	*/
	{fontType, fontID},		/* from icon source to ucode file	*/
	{tabType, tabID},
	{winType, winID},
	{cmpType, cmpID},
	{NULL, NULL}
	};

static resinfo excludei[] = {		/* Resources to exclude when linking	*/
	{ConfResType, ConfResID},	/* ucode files to create icode file	*/
	{NULL, NULL}
	};

#define excluden (&excludei[1])		/* Exclude none	*/

static char uname[256];

/* phases of tmain */
enum
   {
   translate = 0,
   link,
   execution,
   nphases
   };

static int phase = -1;
static int errorcodes[nphases] = {CompErr, LinkErr, ExeErr};

#define	MAX_ARGS 50

static Boolean u,s,t;
static int argc;
static char *argv[MAX_ARGS];
static jmp_buf	return_jump;		/* Used by exit(), etc. to get back */

novalue exit(i)
register int i;
   {
   /*
    * If rcode is not NoErr, we've been through one of our error routines,
    * and know what sort of error to return.  If it's still NoErr, this is
    * a direct call to exit() from the system, and we'll just go with the
    * caller's error number.
    */
   if (compResult.rcode == NoErr)
   	compResult.rcode = i;

   /* Try to be more specific if system just quit */
   if (compResult.rcode == Quit && compResult.msg[0] != '\0')
   	compResult.rcode = errorcodes[phase];
 
   fflush(stderr);
   _closeall();
   longjmp(return_jump,1);
   }


novalue abort()
   {
   compResult.rcode = ExeErr;
   longjmp(return_jump,3);
   }


/*
 * Here to Compile and/or Link and/or Run program.
 */

novalue Compile(pcompOpt,cp,whatItem,ignoreOpts)
pCompOpt  	pcompOpt;
char		*cp;			/* parameter string */
int		whatItem;		/* what to do */
Boolean		ignoreOpts;		/* ignore options found in file */
   {
   char icxname[256];
   register int err;
   StringHandle h;
   FILE *f;
   OSType UcodeFile = UcodeFileType;
   OSType IcodeFile = IcodeFileType;
   OSType IcodeCreator = RuntimeCreator;
   int ref;
   int myRefNum = CurResFile();	/* Record ProIcon's resource file number */

   ignorefile = ignoreOpts;		/* Make available to overrideOpt() */
   phase = translate;

   /* Create static copy of options, unpack into global variables */
   compOpt = *pcompOpt;
   unpackMem();
   unpackFlags();
   
   /* Preserve ucode files if not DELETEU or not Autolinking or explicit Link */
   upreserve = (compOpt.compflags & DELETEU) == 0 || !AutoLinkOn ||
		whatItem == LinkFileItem;
	
#ifdef DeBug
   Dflag = 1;
#endif					/* DeBug */	
	
   if (setjmp(return_jump)) {		/* for use by exit()	*/
   
#ifdef ExternalFunctions
      endxcmd();			/* release any XCMD/XFCN resources */
#endif 					/* ExternalFunctions */
      closerf(myRefNum);		/* close all resource forks */
      return;
      }
 
   /* Get input file name */
   argv[0] = compResult.file;
   argv[1] = (char *)0;
	
   makename(icxname, TargetDir, (char *) compResult.file, IcodeSuffix);
   makename(uname, TargetDir, (char *) compResult.file, U1Suffix);

   /*
    * Round hash table sizes to next power of two, and set masks for hashing.
    */
   chsize = round2(chsize);  cmask = chsize - 1;
   fhsize = round2(fhsize);  fmask = fhsize - 1;
   ghsize = round2(ghsize);  gmask = ghsize - 1;
   ihsize = round2(ihsize);  imask = ihsize - 1;
   lhsize = round2(lhsize);  lmask = lhsize - 1;

   switch (whatItem) {
#ifndef Runtime	
      case CompFileItem:		/* Translate */
      case RunItem: {
         SetState(CompState, runWptr);
         UpdateMenus(false);
         err = trans(argv) ? CompErr : NoErr;
         UnloadSeg(trans);

         ref = openres(uname,1);			/* Open .u1 resource fork */
         if (copyres(argv[0], 0) != noErr) 		/* copy any .icn resources to .u1 */
            comperr(NULL,0x7FFFFFFF,0,
             "Failure copying resources to intermediate file","","");
         saveopts();				/* write options to .u1 resource fork */
         CloseResFile(ref);

         SetFileType(uname,(char*)&UcodeFile,"");
         makename(uname, TargetDir, (char *) compResult.file, U2Suffix);
         SetFileType(uname,(char*)&UcodeFile,"");
         makename(uname, TargetDir, (char *) compResult.file, U1Suffix);
         
         if (!AutoLinkOn || err != NoErr)
            exit(NormalExit);
         }				/* (fall into Link case) */
	 
      case LinkFileItem: {		/* Link */
         SetState(LinkState, runWptr);
         phase = link;
         UpdateMenus(false);
         if (whatItem == LinkFileItem && !ignoreOpts)
            fetchopts(uname);	/* read options from .u1 resource */

         err = ilink(argv,icxname) ? LinkErr : NoErr;
         saveopts();			/* write options to .icx resource */

         UnloadSeg(ilink);
         SetFileType(icxname,(char*)&IcodeFile,
         	     compOpt.compflags & RUNTIME ? (char*)&IcodeCreator : "");
         if (!upreserve)
            deleteucode();
         if (!AutoRunOn || err != NoErr)
            exit(err);
         argv[0] = icxname;
       }				/* (fall into Execute case) */
         
#endif					/* Runtime */

      case RunFileItem:	{		/* Execute */
         phase = execution;
         if (OpenStdio())		/* Open stdin, stdout */
            return;
         if (whatItem == RunFileItem && !ignoreOpts)
            fetchopts(icxname);	/* read options from .icx resource */
         parsecl(cp);
         SetState(RunState, runWptr);
         UpdateMenus(false);
         openres(icxname,0); 		/* open resource fork to access any resources there */
         exec(argc,argv);		/* should not return */
         }
      }
      
   exit(NormalExit);			/* should never get here */
}


#ifdef MemMon
/*
 * get memmon file name
 */
char *MMname(buf, bufsize)
char *buf;
int bufsize;
   {
   struct fileparts *fp;
   char prompt[256];

   /* Fail if MemMon not requested */
   if ((compOpt.compflags & MEMMONF) == 0)
      return (char *)NULL;

   /* Make file name with .mem suffix */
   makename(buf, TargetDir, (char *) compResult.file, MemmonSuffix);

   /* Split name into directory and file name components */
   fp = fparse(buf);
   
   /* Put extension back onto name */
   strcat(fp->name, fp->ext);

   /* Get prompt string */
   GetIndString((StringPtr)prompt, MiscStr, MMFileID);
   PtoCstr(prompt);
   if (PutFile(prompt, fp->name, buf, bufsize, fp->dir) > 0)
      return buf;
   else
      return NULL;
   }
#endif


/*
 * parse command line string into argv[2...], set argc.
 */
static novalue parsecl(cp)
register char *cp;
   {
   char c;
	
   for (argc = 2; argc < MAX_ARGS;) {
      while (isspace(*cp++))
         ;
      if ( !*--cp )
         return;
      else {
         argv[argc] = cp;
         if (*cp == '"') {
            ++argv[argc];		/* remove opening quote */
            while (*++cp && (*cp != '"')) /* scan for EOS or closing quote */
               ;
            }
         else
            while (*++cp && !isspace(*cp))
               ;
         argc++;
         c = *cp;
         *cp++ = '\0';
         if (!c)
            return;
         }
      }
   }


/*
 * Delete ucode files if supposed to
 */
static novalue deleteucode()
   {     
   remove(uname);
   makename(uname, TargetDir, (char *) compResult.file, U2Suffix);
   remove(uname);
   }


/*
 * DisplayStmt - Display source statement where compiler is currently executing.
 *
 * Used during user-triggered Pause or Stop to show progress of program.
 */
novalue
DisplayStmt(wPtr)
WindInfoPtr wPtr;
   {
   char *fname;
   long   lineno;
   Str255   fnStr;
   WindInfoPtr wPtr2;

   if (phase != execution)
      return;

   fname = findfile(ipc.opnd);
   lineno = (long)findline(ipc.opnd);

   if (wPtr && lineno) {
      if (fname) {			/* If file name provided */
         strcpy((char *)fnStr, fname);
         CtoPstr((char *)fnStr);
         wPtr = DoLoadFile(fnStr);
         }

      if (wPtr) {
         /* If the source code window is no longer the frontmost, select it */
         if (FrontWindow () != (WindowPtr) wPtr)
            mySelectWindow((WindowPtr)wPtr);

         /* line = -1 is a flag to not change positioning on file */
         if (lineno > 0)
            XYSet(0L, lineno, wPtr); /* Position at line */
         }
      }
   }



/*
 * DoStopItem - While the compiler was running, the user has selected Stop
 *		from the Compile menu.
 */
novalue
DoStopItem(wPtr)
WindInfoPtr wPtr;
   {
#ifdef MemMon
   ETTerm(0,"stopped from menu");
#endif					/* MemMon */
   DisplayStmt(wPtr);
   compResult.line = 1;			/* line number ??	*/
   compResult.col = 0;
   mfree();
   exit(Stopped);
   }


novalue close_compiler()
   {
   /* close any file data structures within compiler */
   }


/*
 * Unload static memory sizes into globals
 */
novalue
unpackMem()
   {
   register int i;
   
   for (i = nCMemoryItems; i--; )
      *((unsigned int *)memadr(i,CMemoryID)) = compOpt.cmemsizes[i];
   for (i = nEMemoryItems; i--; )
      *((long *)memadr(i,EMemoryID)) = compOpt.ememsizes[i];
      
   /* The interpreter thinks of stksize and mstksize in words, not bytes. */
   stksize /= WordSize;
   mstksize /= WordSize;
   }

/*
 * Unload static flags into globals
 */
novalue
unpackFlags()
   {
  /* Unload global flags */
   silent = (compOpt.compflags & PROGRESS) == 0;
   uwarn = (compOpt.compflags & UWARN) != 0;
   compare = (compOpt.compflags & IUSTRG) != 0;
   dump = (compOpt.compflags & DUMP) != 0;
   if (compOpt.compflags & PTRACE)
      trace = -1;
   else
      trace = 0;
   if (compOpt.compflags & FTRACE)
      ftrace = -1;
   else
      ftrace = 0;
   }

/*
 * Read compiler options from resource fork of specified file,
 * set into compOpt if found.
 */
static novalue fetchopts(fname)
char *fname;
   {
   register int refNum;
   register CompOpt **hCompOpt;
   register StringHandle hparam;
   
   if ((refNum = openres(fname,0)) != -1) {
      if (hCompOpt = (CompOpt **) Get1Resource(ConfResType, ConfResID)) {
         compOpt = **hCompOpt;
         unpackMem();
         }
      CloseResFile(refNum);
      }
   }

   
static novalue saveopts()		/* write options to current resource file */
   {
   register CompOpt **hCompOpt;

   if ((hCompOpt = (CompOpt **)Get1Resource(ConfResType, ConfResID)) == nil) {
      hCompOpt = (CompOpt **)NewHandle(sizeof(CompOpt));  /* record options */
      AddResource(hCompOpt, ConfResType, ConfResID, "");
      }
   else
      ChangedResource(hCompOpt);
   **hCompOpt = compOpt;
   }

/*
 * overrideOpt - called by imain.c after it unpacks options from .icx header.
 *	      It allows the interface to override settings for dump, trace,
 *	      ftrace, and compare if the file is run with the option key.
 */
novalue overrideOpt(hdr)
struct header *hdr;
  {
   if (ignorefile) {
      hdr->trace   = trace;
      hdr->ftrace  = ftrace;
      hdr->dump    = dump;
      hdr->compare = compare;
      }
   }

/*
 * mfree - release memory used by translator, linker, or runtime
 */
static novalue mfree()
{
   switch (phase) {
#ifndef Runtime
      case translate:
         tmfree();			/* release translator's memory */
         break;
      case link:
         lmfree();			/* release linker's memory */
         break;
#endif					/* Runtime */
      case execution:
         xmfree();
         break;
      }
}

/*
 * errcommon returns an error to the editor, with file, line, and
 *  column information when available.  Up to four strings of messages may
 *  be included, and are used with sprintf to build a composite message.
 *  Icon proper can not do this into a temporary buffer because the stack
 *  is about to be clobbered by a longjmp..
 */
static novalue errcommon(type,file,line,col,s1,s2,s3,s4)
int type;
char *file;
long  line, col;
char *s1, *s2, *s3, *s4;
   {
   fflush(stderr);
   _closeall();				/* close ucode files before deleting */
   if (!upreserve)
      deleteucode();

   if (file)
      strcpy(compResult.file,file);
   else
      compResult.file[0] = '\0';

   compResult.line = line;
   if (col > 0)
      col--;				/* use zero-based column number */
   compResult.col  = col;
   sprintf(compResult.msg, s1, s2, s3, s4);
   mfree();				/* release any outstanding memory */
   compResult.rcode = type;		/* Set return code */
   exit(type);
   }

/*
 * comperr returns a compilation error to the editor, with file, line, and
 *  column information when available.
 *
 * s1 contains a %s string that must be evaluated in the context of s2, s3.
 */
novalue comperr(file,line,col,s1,s2,s3)
char *file;
long  line, col;
char *s1, *s2, *s3;
   {
   if (file)
      fprintf(stderr, "File %s\n", file);
   if (line != 0x7FFFFFFF)   		/* if not special case of end of file */
      fprintf(stderr, "Line %ld # ", line);
   fprintf(stderr, s1, s2, s3);
   fprintf(stderr, "\n");
   errcommon(CompErr,file,line,col,s1,s2,s3,"");
   }


/*
 * linkerr returns a link error to the editor, with file, and line
 *  information when available.
 *
 * s1 contains a %s string that must be evaluated in the context of s2, s3, s4.
 *
 * The line number is zero if not known.
 */
novalue linkerr(file,line,s1,s2,s3,s4)
char *file;
long  line;
char *s1, *s2, *s3, *s4;
   {
   if (file)
      fprintf(stderr, "File %s\n", file);
   if (line)
      fprintf(stderr, "Line %ld # ", line);
   fprintf(stderr, s1, s2, s3, s4);
   fprintf(stderr, "\n");
   errcommon(LinkErr,file,line == 0L ? -1L : line,0L,s1,s2,s3,s4);
   }


/*
 * quiterr services quitf(), and returns a serious compilation or linker error.
 *
 * s1 contains a %s string that must be evaluated in the context of s2.
 */
novalue quiterr(file,s1,s2)
char *file;
char *s1, *s2;
   {
   fprintf(stderr,s1,s2);
   fprintf(stderr,"\n");
   errcommon(Quit,file,-1L,0L,s1,s2,"","");
   }

/*
 * exeerr is used for runtime errors.
 *
 * Unlike the preceding error routines, it returns to the caller after storing
 * away the required information, allowing the caller to produce more
 * descriptive information to stderr.  Eventually, the caller calls C_Exit,
 * and we return the information saved here.
 *
 * s1 is a printf string to be evaluated in the context of errno and s2.
 * If errno is not used, pass 0x8000 instead.
 */
novalue exeerr(file,line,s1,errno,s2)
char *file;
long line;
char *s1,*s2;
int  errno;
{
   if (file) {
      if (line)
         fprintf(stderr, "File %s; Line %ld\n", file, line);
      else
         fprintf(stderr, "File %s\n", file);
      strcpy(compResult.file,file);
      }
   else
      compResult.file[0] = '\0';
 
   if (errno != 0x8000)
      sprintf(compResult.msg,s1,errno,s2);
   else
      sprintf(compResult.msg,s1,s2);

   if (compResult.msg[0])
      fprintf(stderr, "%s\n", compResult.msg);

   fflush(stderr);
   compResult.line = line;
   compResult.rcode = ExeErr;
}


/*
 * closerf - close resource forks of all files subsequent to myRefNum
 *	     on the open resource file chain.  Typically myRefNum refers
 *	     to the ProIcon application proper, and this closes the resource
 *	     fork of all files opened subsequently.
 */
static novalue closerf(myRefNum)
int myRefNum;
   {
   int refNum;
   
   while ((refNum = CurResFile()) != myRefNum)
      CloseResFile(refNum);
   }


/*
 * openres - open resource fork of file.
 *
 * file mode may be 0 for read, 1 for write and empty
 */
int openres(fname, mode)
char *fname;
int mode;
   {
   int ref;

   CtoPstr(fname);
   if (mode)
      CreateResFile(fname);			/* Create if doesn't exist */
   ref = sOpenResFile((StringPtr)fname);
   if (mode && ref != -1)			/* Empty existing file if write */
      SetEOF(ref, 0L);
   PtoCstr(fname);
   return ref;
   }

   
/*
 * copyres - copy all resources from one file to another.
 *
 * The source file is specified via a C-string file name.
 * The most recently opened resource file is the destination file.
 *
 * phase:  0 = icon source --> ucode
 *         1 = first ucode file --> icode
 *         2 = subsequent ucode files (if any) --> icode
 *
 * A check is made for duplicate resources.
 *
 * returns:  0 if all OK, or no resources to copy
 *           1 if could not open source file
 *           2 if failure reading resource
 *           3 if failure due to duplicate resource
 */
int copyres(fromfile, phase)
char *fromfile;
int phase;
   {
   int fromRef, toRef;
   int err;
   int nTypes, index;
   ResType theType;
   static resinfo *exc[3] = {excludeu, excluden, excludei};
   
   toRef = CurResFile();			/* Destination file */
   
   if ((fromRef = openres(fromfile,0)) == -1)   /* Source file */
      if ((err = ResError()) == resFNotFound || err == fnfErr || err == eofErr)
         return 0;
      else
         return 1;
   
   nTypes = Count1Types();			/* Number of resource types in source file */
   err = 0;
   for (index = 1; index <= nTypes && !err; index++) {
      Get1IndType(&theType, index);		/* Get next resource type in source file */
      err = docopy(theType, exc[phase], fromRef, toRef);	/* Copy this type of resource */
      }
   CloseResFile(fromRef);
   return err;
   }


  
/*
 * docopy - copy one type of resource between resource files
 *
 * excres is a NULL terminated array of resinfo structures that contain
 * specific resource types and IDs to be excluded from the copy.
 *
 * returns:  0 if all OK
 *           2 if failure reading resource
 *           3 if failure due to duplicate resource
 */
static int docopy(type, excres, fromRef, toRef)
ResType type;
int fromRef, toRef;
resinfo excres[];
   {
   int index, cnt;
   Str255 s;
   Handle resource, res2;
   ResType theType;
   int    theID, attrs, i;
   int err  = 0;
   char tbuf[5];
   
   if ((cnt = Count1Resources(type)) == 0)		/* no. of this type of resource in source */
      return 0;

   SetResPurge(true);					/* Make sure purged resources get written out */
   for (index = 1; index <= cnt && !err; index++) {
      UseResFile(fromRef);				/* read from source file */
      if ((resource = Get1IndResource(type, index)) == NULL)  {  /* get resource */
         err = 2;
         break;
         }
      attrs = GetResAttrs(resource);			/* record its attributes */
      GetResInfo(resource, &theID, &theType, s);	/* and type, ID, and name */

      for (i = 0; excres[i].type != NULL; i++)		/* Scan list of excluded types */
         if (theType == excres[i].type && theID == excres[i].id)
            break;					/* This resource is to be excluded */
      if (!excres[i].type) {
         DetachResource(resource);			/* Detach from old resource fork */
         UseResFile(toRef);				/* Select destination file */
         if ((res2 = Get1Resource(theType, theID)) != NULL ||  /* See if already in destination */
          (res2 = Get1NamedResource(theType, s)) != NULL) {
            PtoCstr((char *)s);				/* Duplicate.  Convert resource name */
            *((ResType *)tbuf) = theType;		/* Resource type as string */
            tbuf[4] = 0;
            fprintf(stderr, "Resource duplicates a previously copied resource:\n");
            fprintf(stderr, "Type %s, ID %d, name \"%s\"\n", tbuf, theID, s);   
            fflush(stderr);  
            ReleaseResource(res2);			/* Duplicate resource */
            err = 3;
            }
         else {
            AddResource(resource, theType, theID, s);	/* Add to destination file */
            SetResAttrs(resource, attrs | resChanged );	/* Copy attributes */
            }
         }
      HUnlock(resource);				/* Make sure resource can be removed */
      HPurge(resource);					/*   from memory if need to	*/
      }
   SetResPurge(false);
   UseResFile(fromRef);
   return err;
   }               
         
