package appsec

import (
	"context"
	"encoding/json"
	"errors"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/akamai/AkamaiOPEN-edgegrid-golang/v11/pkg/session"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAppSec_ListMalwarePolicyActions(t *testing.T) {

	result := GetMalwarePolicyActionsResponse{}

	respData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicyAction/MalwarePolicyActions.json"))
	err := json.Unmarshal([]byte(respData), &result)
	require.NoError(t, err)

	tests := map[string]struct {
		params           GetMalwarePolicyActionsRequest
		responseStatus   int
		responseBody     string
		expectedPath     string
		expectedResponse *GetMalwarePolicyActionsResponse
		withError        error
		headers          http.Header
	}{
		"200 OK": {
			params: GetMalwarePolicyActionsRequest{
				ConfigID: 43253,
				Version:  15,
				PolicyID: "AAAA_81230",
			},
			headers: http.Header{
				"Content-Type": []string{"application/json"},
			},
			responseStatus:   http.StatusOK,
			responseBody:     string(respData),
			expectedPath:     "/appsec/v1/configs/43253/versions/15/security-policies/AAAA_81230/malware-policies",
			expectedResponse: &result,
		},
		"500 internal server error": {
			params: GetMalwarePolicyActionsRequest{
				ConfigID: 43253,
				Version:  15,
				PolicyID: "AAAA_81230",
			},
			headers:        http.Header{},
			responseStatus: http.StatusInternalServerError,
			responseBody: `
{
    "type": "internal_error",
    "title": "Internal Server Error",
    "detail": "Error fetching propertys",
    "status": 500
}`,
			expectedPath: "/appsec/v1/configs/43253/versions/15/security-policies/AAAA_81230/malware-policies",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error fetching propertys",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, test.expectedPath, r.URL.String())
				assert.Equal(t, http.MethodGet, r.Method)
				w.WriteHeader(test.responseStatus)
				_, err := w.Write([]byte(test.responseBody))
				assert.NoError(t, err)
			}))
			client := mockAPIClient(t, mockServer)
			result, err := client.GetMalwarePolicyActions(
				session.ContextWithOptions(
					context.Background(),
					session.WithContextHeaders(test.headers),
				),
				test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, test.expectedResponse, result)
		})
	}
}

// Test Update MalwarePolicyAction.
func TestAppSec_UpdateMalwarePolicyAction(t *testing.T) {
	reqData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicyAction/MalwarePolicyAction.json"))
	request := UpdateMalwarePolicyActionRequest{}
	err := json.Unmarshal([]byte(reqData), &request)
	require.NoError(t, err)

	respData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicyAction/MalwarePolicyAction.json"))
	result := UpdateMalwarePolicyActionResponse{}
	err = json.Unmarshal([]byte(respData), &result)
	require.NoError(t, err)

	tests := map[string]struct {
		params           UpdateMalwarePolicyActionRequest
		responseStatus   int
		responseBody     string
		expectedPath     string
		expectedResponse *UpdateMalwarePolicyActionResponse
		withError        error
		headers          http.Header
	}{
		"200 Success": {
			params: UpdateMalwarePolicyActionRequest{
				ConfigID:        43253,
				Version:         15,
				PolicyID:        "AAAA_81230",
				MalwarePolicyID: 134644,
				Action:          "none",
				UnscannedAction: "none",
			},
			headers: http.Header{
				"Content-Type": []string{"application/json;charset=UTF-8"},
			},
			responseStatus:   http.StatusOK,
			responseBody:     respData,
			expectedResponse: &result,
			expectedPath:     "/appsec/v1/configs/43253/versions/15/security-policies/AAAA_81230/malware-policies/134644",
		},
		"500 internal server error": {
			params: UpdateMalwarePolicyActionRequest{
				ConfigID:        43253,
				Version:         15,
				PolicyID:        "AAAA_81230",
				MalwarePolicyID: 134644,
				Action:          "alert",
				UnscannedAction: "deny",
			},
			responseStatus: http.StatusInternalServerError,
			responseBody: `
			{
				"type": "internal_error",
				"title": "Internal Server Error",
				"detail": "Error creating zone"
			}`,
			expectedPath: "/appsec/v1/configs/43253/versions/15/security-policies/AAAA_81230/malware-policies/134644",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error creating zone",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, http.MethodPut, r.Method)
				w.WriteHeader(test.responseStatus)
				if len(test.responseBody) > 0 {
					_, err := w.Write([]byte(test.responseBody))
					assert.NoError(t, err)
				}
			}))
			client := mockAPIClient(t, mockServer)
			result, err := client.UpdateMalwarePolicyAction(
				session.ContextWithOptions(
					context.Background(),
					session.WithContextHeaders(test.headers)), test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, test.expectedResponse, result)
		})
	}
}
