/*
 * Copyright (c) 2022 Andrea Biscuola <a@abiscuola.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

package main

import (
	"io"
	"net/url"
	"strings"
	"time"

	"github.com/mmcdole/gofeed"
)

func feedifyrss(f *gofeed.Feed) (feeds []Feed, e error) {
	for _, v := range f.Items {
		url, e := url.Parse(v.Link)
		if e != nil {
			continue
		}

		var author, email string
		if len(v.Authors) > 0 {
			author = v.Authors[0].Name
			email = v.Authors[0].Email
		}

		if author == "" && email == "" {
			author = TildeUser(url.Path)
			if author == "" {
				author = url.Hostname()
			}
		}

		title := strings.ReplaceAll(strings.ReplaceAll(v.Title, "\r", ""), "\n", "")
		published := v.PublishedParsed
		if published == nil {
			now := time.Now()
			published = &now
		}

		feeds = append(feeds, Feed{
			Author:          author,
			Email:           email,
			Title:           title,
			PublishedParsed: *published,
			Published:       v.Published,
			Link:            v.Link,
			Description:     v.Description,
		})
	}

	return
}

func FetchRss(url string) ([]Feed, error) {
	f, e := gofeed.NewParser().ParseURL(url)
	if e != nil {
		return nil, e
	}

	return feedifyrss(f)
}

func RssFromReader(r io.Reader) ([]Feed, error) {
	f, e := gofeed.NewParser().Parse(r)
	if e != nil {
		return nil, e
	}
	return feedifyrss(f)
}

func TildeUser(path string) string {
	for _, v := range strings.Split(path, "/") {
		if len(v) > 0 && v[0] == '~' {
			return v[1:]
		}
	}

	return ""
}
