package gogen

import (
	"path/filepath"
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/openconfig/gnmi/errdiff"
	gpb "github.com/openconfig/gnmi/proto/gnmi"
	"github.com/openconfig/goyang/pkg/yang"
	"github.com/openconfig/ygot/genutil"
	"github.com/openconfig/ygot/ygen"
	"github.com/openconfig/ygot/ygot"
	"google.golang.org/protobuf/testing/protocmp"
)

// datapath is the path to common YANG test modules.
const datapath = "../testdata/modules"

type goLangMapper struct {
	*GoLangMapper
}

// PopulateFieldFlags populates extra information given a particular
// field of a ParsedDirectory and the corresponding AST node.
func (goLangMapper) PopulateFieldFlags(nd ygen.NodeDetails, field *yang.Entry) map[string]string {
	if field.Path() == "/openconfig-simple/parent" {
		return map[string]string{"foo": "bar"}
	} else {
		return nil
	}
}

// PopulateEnumFlags populates extra information given a particular
// enumerated type its corresponding AST representation.
func (goLangMapper) PopulateEnumFlags(et ygen.EnumeratedYANGType, yangtype *yang.YangType) map[string]string {
	return map[string]string{"typename": yangtype.Name}
}

func TestGenerateIR(t *testing.T) {
	tests := []struct {
		desc             string
		inYANGFiles      []string
		inIncludePaths   []string
		inExcludeModules []string
		inLangMapper     ygen.LangMapper
		inOpts           ygen.IROptions
		wantIR           *ygen.IR
		wantErrSubstring string
	}{{
		desc: "simple openconfig test with compression",
		inYANGFiles: []string{
			filepath.Join(datapath, "openconfig-simple.yang"),
			filepath.Join(datapath, "openconfig-simple-augment2.yang"),
		},
		inLangMapper: goLangMapper{GoLangMapper: NewGoLangMapper(true)},
		inOpts: ygen.IROptions{
			TransformationOptions: ygen.TransformationOpts{
				CompressBehaviour:                    genutil.PreferIntendedConfig,
				ShortenEnumLeafNames:                 true,
				EnumOrgPrefixesToTrim:                []string{"openconfig"},
				UseDefiningModuleForTypedefEnumNames: true,
				EnumerationsUseUnderscores:           true,
				GenerateFakeRoot:                     true,
			},
			AppendEnumSuffixForSimpleUnionEnums: true,
		},
		wantIR: &ygen.IR{
			Directories: map[string]*ygen.ParsedDirectory{
				"/device": {
					Name:       "Device",
					Type:       ygen.Container,
					Path:       "/device",
					SchemaPath: "",
					Fields: map[string]*ygen.NodeDetails{
						"parent": {
							Name: "Parent",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "parent",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent",
								SchemaPath:        "/parent",
								LeafrefTargetPath: "",
								Description:       "I am a parent container\nthat has 4 children.",
							},
							Type:                    ygen.ContainerNode,
							MappedPaths:             [][]string{{"parent"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
							Flags:                   map[string]string{"foo": "bar"},
						},
						"remote-container": {
							Name: "RemoteContainer",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "remote-container",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container",
								SchemaPath:        "/remote-container",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							MappedPaths:             [][]string{{"remote-container"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					IsFakeRoot: true,
				},
				"/openconfig-simple/parent": {
					Name:       "Parent",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent",
					SchemaPath: "/parent",
					Fields: map[string]*ygen.NodeDetails{
						"child": {
							Name: "Child",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "child",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child",
								SchemaPath:        "/parent/child",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"child"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
				},
				"/openconfig-simple/parent/child": {
					Name:       "Parent_Child",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent/child",
					SchemaPath: "/parent/child",
					Fields: map[string]*ygen.NodeDetails{
						"zero": {
							Name: "Zero",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "zero",
								BelongingModule:   "openconfig-simple-augment2",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple-grouping",
								Path:              "/openconfig-simple/parent/child/state/zero",
								SchemaPath:        "/parent/child/state/zero",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType: "string",
								ZeroValue:  `""`,
							},
							MappedPaths:             [][]string{{"state", "zero"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple-augment2"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"one": {
							Name: "One",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "one",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/config/one",
								SchemaPath:        "/parent/child/config/one",
								ShadowSchemaPath:  "/parent/child/state/one",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"config", "one"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"state", "one"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
						"two": {
							Name: "Two",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "two",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/two",
								SchemaPath:        "/parent/child/state/two",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"state", "two"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"three": {
							Name: "Three",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "three",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/config/three",
								SchemaPath:        "/parent/child/config/three",
								ShadowSchemaPath:  "/parent/child/state/three",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "E_Child_Three",
								UnionTypes:        nil,
								IsEnumeratedValue: true,
								ZeroValue:         "0",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"config", "three"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"state", "three"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
						"four": {
							Name: "Four",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "four",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/config/four",
								SchemaPath:        "/parent/child/config/four",
								ShadowSchemaPath:  "/parent/child/state/four",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "Binary",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         "nil",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"config", "four"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"state", "four"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
				},
				"/openconfig-simple/remote-container": {
					Name:       "RemoteContainer",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/remote-container",
					SchemaPath: "/remote-container",
					Fields: map[string]*ygen.NodeDetails{
						"a-leaf": {
							Name: "ALeaf",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "a-leaf",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container/config/a-leaf",
								SchemaPath:        "/remote-container/config/a-leaf",
								ShadowSchemaPath:  "/remote-container/state/a-leaf",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"config", "a-leaf"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"state", "a-leaf"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-remote",
				},
			},
			Enums: map[string]*ygen.EnumeratedYANGType{
				"/openconfig-simple/parent-config/three": {
					Name:     "Child_Three",
					Kind:     ygen.SimpleEnumerationType,
					TypeName: "enumeration",
					ValToYANGDetails: []ygot.EnumDefinition{{
						Name:  "ONE",
						Value: 0,
					}, {
						Name:  "TWO",
						Value: 1,
					}},
					Flags: map[string]string{"typename": "enumeration"},
				},
			},
			ModelData: []*gpb.ModelData{{Name: "openconfig-remote"}, {Name: "openconfig-simple"}, {Name: "openconfig-simple-augment2"}, {Name: "openconfig-simple-grouping"}},
		},
	}, {
		desc: "simple openconfig test compression prefer state no underscores",
		inYANGFiles: []string{
			filepath.Join(datapath, "openconfig-simple.yang"),
			filepath.Join(datapath, "openconfig-simple-augment2.yang"),
		},
		inLangMapper: NewGoLangMapper(true),
		inOpts: ygen.IROptions{
			TransformationOptions: ygen.TransformationOpts{
				CompressBehaviour:                    genutil.PreferOperationalState,
				ShortenEnumLeafNames:                 true,
				EnumOrgPrefixesToTrim:                []string{"openconfig"},
				UseDefiningModuleForTypedefEnumNames: true,
				EnumerationsUseUnderscores:           false,
				GenerateFakeRoot:                     true,
			},
			AppendEnumSuffixForSimpleUnionEnums: true,
		},
		wantIR: &ygen.IR{
			Directories: map[string]*ygen.ParsedDirectory{
				"/device": {
					Name:       "Device",
					Type:       ygen.Container,
					Path:       "/device",
					SchemaPath: "",
					Fields: map[string]*ygen.NodeDetails{
						"parent": {
							Name: "Parent",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "parent",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent",
								SchemaPath:        "/parent",
								LeafrefTargetPath: "",
								Description:       "I am a parent container\nthat has 4 children.",
							},
							Type:                    ygen.ContainerNode,
							MappedPaths:             [][]string{{"parent"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"remote-container": {
							Name: "RemoteContainer",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "remote-container",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container",
								SchemaPath:        "/remote-container",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							MappedPaths:             [][]string{{"remote-container"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					IsFakeRoot: true,
				},
				"/openconfig-simple/parent": {
					Name:       "Parent",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent",
					SchemaPath: "/parent",
					Fields: map[string]*ygen.NodeDetails{
						"child": {
							Name: "Child",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "child",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child",
								SchemaPath:        "/parent/child",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"child"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
				},
				"/openconfig-simple/parent/child": {
					Name:       "Parent_Child",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent/child",
					SchemaPath: "/parent/child",
					Fields: map[string]*ygen.NodeDetails{
						"zero": {
							Name: "Zero",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "zero",
								BelongingModule:   "openconfig-simple-augment2",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple-grouping",
								Path:              "/openconfig-simple/parent/child/state/zero",
								SchemaPath:        "/parent/child/state/zero",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type:                    ygen.LeafNode,
							LangType:                &ygen.MappedType{NativeType: "string", ZeroValue: `""`},
							MappedPaths:             [][]string{{"state", "zero"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple-augment2"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"one": {
							Name: "One",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "one",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/one",
								SchemaPath:        "/parent/child/state/one",
								ShadowSchemaPath:  "/parent/child/config/one",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"state", "one"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"config", "one"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
						"two": {
							Name: "Two",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "two",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/two",
								SchemaPath:        "/parent/child/state/two",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"state", "two"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"three": {
							Name: "Three",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "three",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/three",
								SchemaPath:        "/parent/child/state/three",
								ShadowSchemaPath:  "/parent/child/config/three",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "E_ChildThree",
								UnionTypes:        nil,
								IsEnumeratedValue: true,
								ZeroValue:         "0",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"state", "three"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"config", "three"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
						"four": {
							Name: "Four",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "four",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/four",
								SchemaPath:        "/parent/child/state/four",
								ShadowSchemaPath:  "/parent/child/config/four",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "Binary",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         "nil",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"state", "four"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"config", "four"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
				},
				"/openconfig-simple/remote-container": {
					Name:       "RemoteContainer",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/remote-container",
					SchemaPath: "/remote-container",
					Fields: map[string]*ygen.NodeDetails{
						"a-leaf": {
							Name: "ALeaf",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "a-leaf",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container/state/a-leaf",
								SchemaPath:        "/remote-container/state/a-leaf",
								ShadowSchemaPath:  "/remote-container/config/a-leaf",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"state", "a-leaf"}},
							MappedPathModules:       [][]string{{"openconfig-simple", "openconfig-simple"}},
							ShadowMappedPaths:       [][]string{{"config", "a-leaf"}},
							ShadowMappedPathModules: [][]string{{"openconfig-simple", "openconfig-simple"}},
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-remote",
				},
			},
			Enums: map[string]*ygen.EnumeratedYANGType{
				"/openconfig-simple/parent-config/three": {
					Name:     "ChildThree",
					Kind:     ygen.SimpleEnumerationType,
					TypeName: "enumeration",
					ValToYANGDetails: []ygot.EnumDefinition{{
						Name:  "ONE",
						Value: 0,
					}, {
						Name:  "TWO",
						Value: 1,
					}},
				},
			},
			ModelData: []*gpb.ModelData{{Name: "openconfig-remote"}, {Name: "openconfig-simple"}, {Name: "openconfig-simple-augment2"}, {Name: "openconfig-simple-grouping"}},
		},
	}, {
		desc: "simple openconfig test without compression",
		inYANGFiles: []string{
			filepath.Join(datapath, "openconfig-simple.yang"),
			filepath.Join(datapath, "openconfig-simple-augment2.yang"),
		},
		inLangMapper: NewGoLangMapper(true),
		inOpts: ygen.IROptions{
			TransformationOptions: ygen.TransformationOpts{
				CompressBehaviour:                    genutil.Uncompressed,
				ShortenEnumLeafNames:                 true,
				EnumOrgPrefixesToTrim:                []string{"openconfig"},
				UseDefiningModuleForTypedefEnumNames: true,
				EnumerationsUseUnderscores:           true,
				GenerateFakeRoot:                     true,
			},
			AppendEnumSuffixForSimpleUnionEnums: true,
		},
		wantIR: &ygen.IR{
			Directories: map[string]*ygen.ParsedDirectory{
				"/device": {
					Name:       "Device",
					Type:       ygen.Container,
					Path:       "/device",
					SchemaPath: "",
					Fields: map[string]*ygen.NodeDetails{
						"parent": {
							Name: "Parent",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "parent",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent",
								SchemaPath:        "/parent",
								LeafrefTargetPath: "",
								Description:       "I am a parent container\nthat has 4 children.",
							},
							Type:                    ygen.ContainerNode,
							MappedPaths:             [][]string{{"parent"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"remote-container": {
							Name: "RemoteContainer",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "remote-container",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container",
								SchemaPath:        "/remote-container",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							MappedPaths:             [][]string{{"remote-container"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					IsFakeRoot: true,
				},
				"/openconfig-simple/parent": {
					Name:       "OpenconfigSimple_Parent",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent",
					SchemaPath: "/parent",
					Fields: map[string]*ygen.NodeDetails{
						"child": {
							Name: "Child",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "child",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child",
								SchemaPath:        "/parent/child",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"child"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
				},
				"/openconfig-simple/parent/child": {
					Name:       "OpenconfigSimple_Parent_Child",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent/child",
					SchemaPath: "/parent/child",
					Fields: map[string]*ygen.NodeDetails{
						"config": {
							Name: "Config",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "config",
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/config",
								SchemaPath:        "/parent/child/config",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:              1,
							MappedPaths:       [][]string{{"config"}},
							MappedPathModules: [][]string{{"openconfig-simple"}},
						},
						"state": {
							Name: "State",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "state",
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state",
								SchemaPath:        "/parent/child/state",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type:              1,
							MappedPaths:       [][]string{{"state"}},
							MappedPathModules: [][]string{{"openconfig-simple"}},
						},
					},
					ListKeys:          nil,
					PackageName:       "",
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
				},
				"/openconfig-simple/parent/child/config": {
					Name:       "OpenconfigSimple_Parent_Child_Config",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent/child/config",
					SchemaPath: "/parent/child/config",
					Fields: map[string]*ygen.NodeDetails{
						"four": {
							Name: "Four",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "four",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/config/four",
								SchemaPath:        "/parent/child/config/four",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "Binary",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         "nil",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"four"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"one": {
							Name: "One",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "one",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/config/one",
								SchemaPath:        "/parent/child/config/one",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"one"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"three": {
							Name: "Three",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "three",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/config/three",
								SchemaPath:        "/parent/child/config/three",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "E_Simple_Parent_Child_Config_Three",
								UnionTypes:        nil,
								IsEnumeratedValue: true,
								ZeroValue:         "0",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"three"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					ListKeys:          nil,
					PackageName:       "",
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
				},
				"/openconfig-simple/parent/child/state": {
					Name:       "OpenconfigSimple_Parent_Child_State",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/parent/child/state",
					SchemaPath: "/parent/child/state",
					Fields: map[string]*ygen.NodeDetails{
						"four": {
							Name: "Four",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "four",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/four",
								SchemaPath:        "/parent/child/state/four",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "Binary",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         "nil",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"four"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"one": {
							Name: "One",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "one",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/one",
								SchemaPath:        "/parent/child/state/one",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"one"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"three": {
							Name: "Three",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "three",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/three",
								SchemaPath:        "/parent/child/state/three",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "E_Simple_Parent_Child_Config_Three",
								UnionTypes:        nil,
								IsEnumeratedValue: true,
								ZeroValue:         "0",
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"three"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"two": {
							Name: "Two",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "two",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple",
								Path:              "/openconfig-simple/parent/child/state/two",
								SchemaPath:        "/parent/child/state/two",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"two"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"zero": {
							Name: "Zero",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "zero",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple-augment2",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-simple-grouping",
								Path:              "/openconfig-simple/parent/child/state/zero",
								SchemaPath:        "/parent/child/state/zero",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"zero"}},
							MappedPathModules:       [][]string{{"openconfig-simple-augment2"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					ListKeys:          nil,
					PackageName:       "",
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-simple",
					ConfigFalse:       true,
				},
				"/openconfig-simple/remote-container": {
					Name:       "OpenconfigSimple_RemoteContainer",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/remote-container",
					SchemaPath: "/remote-container",
					Fields: map[string]*ygen.NodeDetails{
						"config": {
							Name: "Config",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "config",
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container/config",
								SchemaPath:        "/remote-container/config",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:              1,
							MappedPaths:       [][]string{{"config"}},
							MappedPathModules: [][]string{{"openconfig-simple"}},
						},
						"state": {
							Name: "State",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "state",
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container/state",
								SchemaPath:        "/remote-container/state",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type:              1,
							MappedPaths:       [][]string{{"state"}},
							MappedPathModules: [][]string{{"openconfig-simple"}},
						},
					},
					ListKeys:          nil,
					PackageName:       "",
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-remote",
				},
				"/openconfig-simple/remote-container/config": {
					Name:       "OpenconfigSimple_RemoteContainer_Config",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/remote-container/config",
					SchemaPath: "/remote-container/config",
					Fields: map[string]*ygen.NodeDetails{
						"a-leaf": {
							Name: "ALeaf",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "a-leaf",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container/config/a-leaf",
								SchemaPath:        "/remote-container/config/a-leaf",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"a-leaf"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-remote",
				},
				"/openconfig-simple/remote-container/state": {
					Name:       "OpenconfigSimple_RemoteContainer_State",
					Type:       ygen.Container,
					Path:       "/openconfig-simple/remote-container/state",
					SchemaPath: "/remote-container/state",
					Fields: map[string]*ygen.NodeDetails{
						"a-leaf": {
							Name: "ALeaf",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "a-leaf",
								Defaults:          nil,
								BelongingModule:   "openconfig-simple",
								RootElementModule: "openconfig-simple",
								DefiningModule:    "openconfig-remote",
								Path:              "/openconfig-simple/remote-container/state/a-leaf",
								SchemaPath:        "/remote-container/state/a-leaf",
								LeafrefTargetPath: "",
								Description:       "",
								ConfigFalse:       true,
							},
							Type: 3,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"a-leaf"}},
							MappedPathModules:       [][]string{{"openconfig-simple"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					BelongingModule:   "openconfig-simple",
					RootElementModule: "openconfig-simple",
					DefiningModule:    "openconfig-remote",
					ConfigFalse:       true,
				},
			},
			Enums: map[string]*ygen.EnumeratedYANGType{
				"/openconfig-simple/parent-config/three": {
					Name:     "Simple_Parent_Child_Config_Three",
					Kind:     1,
					TypeName: "enumeration",
					ValToYANGDetails: []ygot.EnumDefinition{{
						Name:  "ONE",
						Value: 0,
					}, {
						Name:  "TWO",
						Value: 1,
					}},
				},
			},
			ModelData: []*gpb.ModelData{{Name: "openconfig-remote"}, {Name: "openconfig-simple"}, {Name: "openconfig-simple-augment2"}, {Name: "openconfig-simple-grouping"}},
		},
	}, {
		desc:             "exclude module test with compression",
		inYANGFiles:      []string{filepath.Join(datapath, "excluded-module-noimport.yang")},
		inExcludeModules: []string{"excluded-module-two"},
		inLangMapper:     NewGoLangMapper(true),
		inOpts: ygen.IROptions{
			TransformationOptions: ygen.TransformationOpts{
				CompressBehaviour:                    genutil.PreferIntendedConfig,
				ShortenEnumLeafNames:                 true,
				UseDefiningModuleForTypedefEnumNames: true,
				EnumerationsUseUnderscores:           true,
				GenerateFakeRoot:                     true,
			},
			AppendEnumSuffixForSimpleUnionEnums: true,
		},
		wantIR: &ygen.IR{
			Directories: map[string]*ygen.ParsedDirectory{
				"/device": {
					Name:       "Device",
					Type:       ygen.Container,
					Path:       "/device",
					SchemaPath: "",
					Fields: map[string]*ygen.NodeDetails{
						"e1": {
							Name: "E1",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "e1",
								Defaults:          nil,
								BelongingModule:   "excluded-module-noimport",
								RootElementModule: "excluded-module-noimport",
								DefiningModule:    "excluded-module-noimport",
								Path:              "/excluded-module-noimport/e1",
								SchemaPath:        "/e1",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "string",
								UnionTypes:        nil,
								IsEnumeratedValue: false,
								ZeroValue:         `""`,
								DefaultValue:      nil,
							},
							MappedPaths:             [][]string{{"e1"}},
							MappedPathModules:       [][]string{{"excluded-module-noimport"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					IsFakeRoot: true,
				},
			},
			// TODO(wenbli): Determine whether "excluded-module-two" should be here.
			ModelData: []*gpb.ModelData{{Name: "excluded-module-noimport"}, {Name: "excluded-module-two"}},
		},
	}, {
		desc:         "complex openconfig test with compression",
		inYANGFiles:  []string{filepath.Join(datapath, "openconfig-complex.yang")},
		inLangMapper: NewGoLangMapper(true),
		inOpts: ygen.IROptions{
			TransformationOptions: ygen.TransformationOpts{
				CompressBehaviour:                    genutil.PreferIntendedConfig,
				ShortenEnumLeafNames:                 true,
				EnumOrgPrefixesToTrim:                []string{"openconfig"},
				UseDefiningModuleForTypedefEnumNames: true,
				EnumerationsUseUnderscores:           true,
				GenerateFakeRoot:                     true,
			},
			AppendEnumSuffixForSimpleUnionEnums: true,
		},
		wantIR: &ygen.IR{
			Directories: map[string]*ygen.ParsedDirectory{
				"/device": {
					Name:       "Device",
					Type:       ygen.Container,
					Path:       "/device",
					SchemaPath: "",
					Fields: map[string]*ygen.NodeDetails{
						"model": {
							Name: "Model",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "model",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model",
								SchemaPath:        "/model",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"model"}},
							MappedPathModules:       [][]string{{"openconfig-complex"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"example-presence": {
							Name: "ExamplePresence",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "example-presence",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/example-presence",
								SchemaPath:        "/example-presence",
								LeafrefTargetPath: "",
								PresenceStatement: ygot.String("This is an example presence container"),
								Description:       "",
							},
							Type:                    ygen.ContainerNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"example-presence"}},
							MappedPathModules:       [][]string{{"openconfig-complex"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
					},
					IsFakeRoot: true,
				},
				"/openconfig-complex/example-presence": {
					Name:                      "ExamplePresence",
					Type:                      ygen.Container,
					Path:                      "/openconfig-complex/example-presence",
					SchemaPath:                "/example-presence",
					Fields:                    map[string]*ygen.NodeDetails{},
					PackageName:               "",
					ListKeys:                  nil,
					IsFakeRoot:                false,
					BelongingModule:           "openconfig-complex",
					RootElementModule:         "openconfig-complex",
					DefiningModule:            "openconfig-complex",
					TelemetryAtomic:           true,
					CompressedTelemetryAtomic: false,
				},
				"/openconfig-complex/model": {
					Name:       "Model",
					Type:       ygen.Container,
					Path:       "/openconfig-complex/model",
					SchemaPath: "/model",
					Fields: map[string]*ygen.NodeDetails{
						"anydata-leaf": {
							Name: "AnydataLeaf",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "anydata-leaf",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/anydata-leaf",
								SchemaPath:        "/model/anydata-leaf",
								LeafrefTargetPath: "",
								Description:       "some anydata leaf",
							},
							Type:                    ygen.AnyDataNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"anydata-leaf"}},
							MappedPathModules:       [][]string{{"openconfig-complex"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"dateref": {
							Name: "Dateref",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "dateref",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/dateref",
								SchemaPath:        "/model/dateref",
								LeafrefTargetPath: "/openconfig-complex/model/a/single-key/config/dates",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:   "uint8",
								ZeroValue:    "0",
								DefaultValue: ygot.String("5"),
							},
							MappedPaths:             [][]string{{"dateref"}},
							MappedPathModules:       [][]string{{"openconfig-complex"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"multi-key": {
							Name: "MultiKey",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "multi-key",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/b/multi-key",
								SchemaPath:        "/model/b/multi-key",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ListNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"b", "multi-key"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"single-key": {
							Name: "SingleKey",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "single-key",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key",
								SchemaPath:        "/model/a/single-key",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.ListNode,
							LangType:                nil,
							MappedPaths:             [][]string{{"a", "single-key"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       nil,
							ShadowMappedPathModules: nil,
						},
						"unkeyed-list": {
							Name: "UnkeyedList",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "unkeyed-list",
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/c/unkeyed-list",
								SchemaPath:        "/model/c/unkeyed-list",
								ConfigFalse:       true,
							},
							Type:              ygen.ListNode,
							MappedPaths:       [][]string{{"c", "unkeyed-list"}},
							MappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
					},
					ListKeys:          nil,
					PackageName:       "",
					BelongingModule:   "openconfig-complex",
					RootElementModule: "openconfig-complex",
					DefiningModule:    "openconfig-complex",
				},
				"/openconfig-complex/model/a/single-key": {
					Name:       "Model_SingleKey",
					Type:       ygen.List,
					Path:       "/openconfig-complex/model/a/single-key",
					SchemaPath: "/model/a/single-key",
					Fields: map[string]*ygen.NodeDetails{
						"dates": {
							Name: "Dates",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "dates",
								Defaults:          []string{"5"},
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/dates",
								SchemaPath:        "/model/a/single-key/config/dates",
								ShadowSchemaPath:  "/model/a/single-key/state/dates",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.LeafListNode,
							LangType:                &ygen.MappedType{NativeType: "uint8", ZeroValue: "0", DefaultValue: ygot.String("[]uint8{5}")},
							MappedPaths:             [][]string{{"config", "dates"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "dates"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"dates-with-defaults": {
							Name: "DatesWithDefaults",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "dates-with-defaults",
								Defaults:          []string{"1", "2"},
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/dates-with-defaults",
								SchemaPath:        "/model/a/single-key/config/dates-with-defaults",
								ShadowSchemaPath:  "/model/a/single-key/state/dates-with-defaults",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.LeafListNode,
							LangType:                &ygen.MappedType{NativeType: "uint8", ZeroValue: "0", DefaultValue: ygot.String("[]uint8{1, 2}")},
							MappedPaths:             [][]string{{"config", "dates-with-defaults"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "dates-with-defaults"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"iref": {
							Name: "Iref",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "iref",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/iref",
								SchemaPath:        "/model/a/single-key/config/iref",
								ShadowSchemaPath:  "/model/a/single-key/state/iref",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "E_Complex_SOFTWARE",
								IsEnumeratedValue: true,
								ZeroValue:         "0",
							},
							MappedPaths:             [][]string{{"config", "iref"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "iref"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"iref2": {
							Name: "Iref2",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "iref2",
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/iref2",
								SchemaPath:        "/model/a/single-key/config/iref2",
								ShadowSchemaPath:  "/model/a/single-key/state/iref2",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "E_Complex_Program",
								IsEnumeratedValue: true,
								ZeroValue:         "0",
							},
							MappedPaths:             [][]string{{"config", "iref2"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "iref2"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"key": {
							Name: "Key",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "key",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/key",
								SchemaPath:        "/model/a/single-key/config/key",
								ShadowSchemaPath:  "/model/a/single-key/state/key",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType: "Model_SingleKey_Key_Union",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"E_Complex_WeekendDays": {
										Index: 1,
									},
									"uint8": {
										Index: 0,
									},
								},
								ZeroValue: "nil",
							},
							MappedPaths:             [][]string{{"config", "key"}, {"key"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}, {"openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "key"}, {"key"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}, {"openconfig-complex"}},
						},
						"leaf-default-override": {
							Name: "LeafDefaultOverride",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "leaf-default-override",
								Defaults:          []string{"3"},
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/leaf-default-override",
								SchemaPath:        "/model/a/single-key/config/leaf-default-override",
								ShadowSchemaPath:  "/model/a/single-key/state/leaf-default-override",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType: "Model_SingleKey_LeafDefaultOverride_Union",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"E_Complex_CycloneScales_Enum": {
										Index: 1,
									},
									"uint8": {
										Index: 0,
									},
								},
								ZeroValue:    "nil",
								DefaultValue: ygot.String("UnionUint8(3)"),
							},
							MappedPaths:             [][]string{{"config", "leaf-default-override"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "leaf-default-override"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"simple-union-enum": {
							Name: "SimpleUnionEnum",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "simple-union-enum",
								Defaults:          []string{"TWO"},
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/simple-union-enum",
								SchemaPath:        "/model/a/single-key/config/simple-union-enum",
								ShadowSchemaPath:  "/model/a/single-key/state/simple-union-enum",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType: "Model_SingleKey_SimpleUnionEnum_Union",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"E_SingleKey_SimpleUnionEnum_Enum": {
										Index: 1,
									},
									"uint64": {
										Index: 0,
									},
								},
								ZeroValue:    "nil",
								DefaultValue: ygot.String("SingleKey_SimpleUnionEnum_Enum_TWO"),
							},
							MappedPaths:             [][]string{{"config", "simple-union-enum"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "simple-union-enum"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"singleton-union-enum": {
							Name: "SingletonUnionEnum",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "singleton-union-enum",
								Defaults:          []string{"DEUX"},
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/singleton-union-enum",
								SchemaPath:        "/model/a/single-key/config/singleton-union-enum",
								ShadowSchemaPath:  "/model/a/single-key/state/singleton-union-enum",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType: "E_SingleKey_SingletonUnionEnum_Enum",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"E_SingleKey_SingletonUnionEnum_Enum": {
										Index: 0,
									},
								},
								IsEnumeratedValue: true,
								ZeroValue:         "0",
								DefaultValue:      ygot.String("SingleKey_SingletonUnionEnum_Enum_DEUX"),
							},
							MappedPaths:             [][]string{{"config", "singleton-union-enum"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "singleton-union-enum"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"typedef-enum": {
							Name: "TypedefEnum",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "typedef-enum",
								Defaults:          []string{"SATURDAY"},
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/typedef-enum",
								SchemaPath:        "/model/a/single-key/config/typedef-enum",
								ShadowSchemaPath:  "/model/a/single-key/state/typedef-enum",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType:        "E_Complex_WeekendDays",
								IsEnumeratedValue: true,
								ZeroValue:         "0",
								DefaultValue:      ygot.String("Complex_WeekendDays_SATURDAY"),
							},
							MappedPaths:             [][]string{{"config", "typedef-enum"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "typedef-enum"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
						"typedef-union-enum": {
							Name: "TypedefUnionEnum",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "typedef-union-enum",
								Defaults:          []string{"SUPER"},
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/a/single-key/config/typedef-union-enum",
								SchemaPath:        "/model/a/single-key/config/typedef-union-enum",
								ShadowSchemaPath:  "/model/a/single-key/state/typedef-union-enum",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType: "Model_SingleKey_TypedefUnionEnum_Union",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"E_Complex_CycloneScales_Enum": {
										Index: 1,
									},
									"uint8": {
										Index: 0,
									},
								},
								ZeroValue:    "nil",
								DefaultValue: ygot.String("Complex_CycloneScales_Enum_SUPER"),
							},
							MappedPaths:             [][]string{{"config", "typedef-union-enum"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "typedef-union-enum"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}},
						},
					},
					ListKeys: map[string]*ygen.ListKey{
						"key": {
							Name: "Key",
							LangType: &ygen.MappedType{
								NativeType: "Model_SingleKey_Key_Union",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"E_Complex_WeekendDays": {
										Index: 1,
									},
									"uint8": {
										Index: 0,
									},
								},
								ZeroValue: "nil",
							},
						},
					},
					ListKeyYANGNames:          []string{"key"},
					PackageName:               "",
					IsFakeRoot:                false,
					BelongingModule:           "openconfig-complex",
					RootElementModule:         "openconfig-complex",
					DefiningModule:            "openconfig-complex",
					TelemetryAtomic:           false,
					CompressedTelemetryAtomic: true,
				},
				"/openconfig-complex/model/b/multi-key": {
					Name:       "Model_MultiKey",
					Type:       ygen.List,
					Path:       "/openconfig-complex/model/b/multi-key",
					SchemaPath: "/model/b/multi-key",
					Fields: map[string]*ygen.NodeDetails{
						"key1": {
							Name: "Key1",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "key1",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/b/multi-key/config/key1",
								SchemaPath:        "/model/b/multi-key/config/key1",
								ShadowSchemaPath:  "/model/b/multi-key/state/key1",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type: ygen.LeafNode,
							LangType: &ygen.MappedType{
								NativeType: "uint32",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"uint32": {
										Index: 0,
									},
								},
								ZeroValue: "0",
							},
							MappedPaths:             [][]string{{"config", "key1"}, {"key1"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}, {"openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "key1"}, {"key1"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}, {"openconfig-complex"}},
						},
						"key2": {
							Name: "Key2",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "key2",
								Defaults:          nil,
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/b/multi-key/config/key2",
								SchemaPath:        "/model/b/multi-key/config/key2",
								ShadowSchemaPath:  "/model/b/multi-key/state/key2",
								LeafrefTargetPath: "",
								Description:       "",
							},
							Type:                    ygen.LeafNode,
							LangType:                &ygen.MappedType{NativeType: "E_MultiKey_Key2", IsEnumeratedValue: true, ZeroValue: "0"},
							MappedPaths:             [][]string{{"config", "key2"}, {"key2"}},
							MappedPathModules:       [][]string{{"openconfig-complex", "openconfig-complex"}, {"openconfig-complex"}},
							ShadowMappedPaths:       [][]string{{"state", "key2"}, {"key2"}},
							ShadowMappedPathModules: [][]string{{"openconfig-complex", "openconfig-complex"}, {"openconfig-complex"}},
						},
					},
					ListKeys: map[string]*ygen.ListKey{
						"key1": {
							Name: "Key1",
							LangType: &ygen.MappedType{
								NativeType: "uint32",
								UnionTypes: map[string]ygen.MappedUnionSubtype{
									"uint32": {
										Index: 0,
									},
								},
								ZeroValue: "0",
							},
						},
						"key2": {
							Name:     "Key2",
							LangType: &ygen.MappedType{NativeType: "E_MultiKey_Key2", IsEnumeratedValue: true, ZeroValue: "0"},
						},
					},
					ListKeyYANGNames:          []string{"key1", "key2"},
					PackageName:               "",
					IsFakeRoot:                false,
					BelongingModule:           "openconfig-complex",
					RootElementModule:         "openconfig-complex",
					DefiningModule:            "openconfig-complex",
					TelemetryAtomic:           true,
					CompressedTelemetryAtomic: false,
				},
				"/openconfig-complex/model/c/unkeyed-list": {
					Name:       "Model_UnkeyedList",
					Type:       ygen.List,
					Path:       "/openconfig-complex/model/c/unkeyed-list",
					SchemaPath: "/model/c/unkeyed-list",
					Fields: map[string]*ygen.NodeDetails{
						"field": {
							Name: "Field",
							YANGDetails: ygen.YANGNodeDetails{
								Name:              "field",
								BelongingModule:   "openconfig-complex",
								RootElementModule: "openconfig-complex",
								DefiningModule:    "openconfig-complex",
								Path:              "/openconfig-complex/model/c/unkeyed-list/field",
								SchemaPath:        "/model/c/unkeyed-list/field",
								ConfigFalse:       true,
							},
							Type:              ygen.LeafNode,
							LangType:          &ygen.MappedType{NativeType: "Binary", ZeroValue: "nil"},
							MappedPaths:       [][]string{{"field"}},
							MappedPathModules: [][]string{{"openconfig-complex"}},
						},
					},
					BelongingModule:   "openconfig-complex",
					RootElementModule: "openconfig-complex",
					DefiningModule:    "openconfig-complex",
					ConfigFalse:       true,
				},
			},
			Enums: map[string]*ygen.EnumeratedYANGType{
				"/openconfig-complex/cyclone-scales": {
					Name:     "Complex_CycloneScales_Enum",
					Kind:     ygen.DerivedUnionEnumerationType,
					TypeName: "cyclone-scales",
					ValToYANGDetails: []ygot.EnumDefinition{
						{
							Name:           "NORMAL",
							DefiningModule: "",
							Value:          0,
						},
						{
							Name:           "SUPER",
							DefiningModule: "",
							Value:          1,
						},
					},
				},
				"/openconfig-complex/SOFTWARE": {
					Name:             "Complex_SOFTWARE",
					Kind:             ygen.IdentityType,
					IdentityBaseName: "SOFTWARE",
					TypeName:         "identityref",
					ValToYANGDetails: []ygot.EnumDefinition{
						{Name: "OS", DefiningModule: "openconfig-complex"},
					},
				},
				"/openconfig-complex/program": {
					Name:             "Complex_Program",
					Kind:             ygen.IdentityType,
					IdentityBaseName: "SOFTWARE",
					TypeName:         "program",
					ValToYANGDetails: []ygot.EnumDefinition{
						{Name: "OS", DefiningModule: "openconfig-complex"},
					},
				},
				"/openconfig-complex/multi-key-config/key2": {
					Name:     "MultiKey_Key2",
					Kind:     ygen.SimpleEnumerationType,
					TypeName: "enumeration",
					ValToYANGDetails: []ygot.EnumDefinition{
						{
							Name:           "RED",
							DefiningModule: "",
							Value:          0,
						},
						{
							Name:           "BLUE",
							DefiningModule: "",
							Value:          1,
						},
					},
				},
				"/openconfig-complex/weekend-days": {
					Name:     "Complex_WeekendDays",
					Kind:     ygen.DerivedEnumerationType,
					TypeName: "weekend-days",
					ValToYANGDetails: []ygot.EnumDefinition{
						{
							Name:           "SATURDAY",
							DefiningModule: "",
							Value:          0,
						},
						{
							Name:           "SUNDAY",
							DefiningModule: "",
							Value:          1,
						},
					},
				},
				"/openconfig-complex/single-key-config/simple-union-enum": {
					Name:     "SingleKey_SimpleUnionEnum_Enum",
					Kind:     ygen.UnionEnumerationType,
					TypeName: "union",
					ValToYANGDetails: []ygot.EnumDefinition{
						{
							Name:           "ONE",
							DefiningModule: "",
							Value:          0,
						},
						{
							Name:           "TWO",
							DefiningModule: "",
							Value:          1,
						},
						{
							Name:           "THREE",
							DefiningModule: "",
							Value:          2,
						},
					},
				},
				"/openconfig-complex/single-key-config/singleton-union-enum": {
					Name:     "SingleKey_SingletonUnionEnum_Enum",
					Kind:     ygen.UnionEnumerationType,
					TypeName: "union",
					ValToYANGDetails: []ygot.EnumDefinition{
						{
							Name:           "UN",
							DefiningModule: "",
							Value:          0,
						},
						{
							Name:           "DEUX",
							DefiningModule: "",
							Value:          1,
						},
						{
							Name:           "TROIS",
							DefiningModule: "",
							Value:          2,
						},
					},
				},
			},
			ModelData: []*gpb.ModelData{
				{Name: "openconfig-complex"},
				{Name: "openconfig-extensions"},
			},
		},
	}}

	for _, tt := range tests {
		t.Run(tt.desc, func(t *testing.T) {
			tt.inOpts.ParseOptions.ExcludeModules = tt.inExcludeModules
			got, err := ygen.GenerateIR(tt.inYANGFiles, tt.inIncludePaths, tt.inLangMapper, tt.inOpts)
			if diff := errdiff.Substring(err, tt.wantErrSubstring); diff != "" {
				t.Fatalf("did not get expected error, %s", diff)
			}
			if diff := cmp.Diff(got, tt.wantIR, cmpopts.IgnoreUnexported(ygen.IR{}, ygen.ParsedDirectory{}, ygen.EnumeratedYANGType{}), protocmp.Transform()); diff != "" {
				t.Fatalf("did not get expected IR, diff(-got,+want):\n%s", diff)
			}
		})
	}
}
