// Copyright 2023 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package gogen

import (
	"bytes"
	"fmt"
	"sort"

	"github.com/openconfig/ygot/genutil"
	"github.com/openconfig/ygot/ygen"
)

// generatedGoMultiKeyListStruct is used to represent a struct used as a key of a YANG list that has multiple
// key elements.
type generatedGoMultiKeyListStruct struct {
	KeyStructName string          // KeyStructName is the name of the struct being output.
	Keys          []goStructField // Keys is a slice of goStructFields that are contained in the key struct.
	ParentPath    string          // ParentPath is the path to the list's parent in the YANG schema.
	ListName      string          // ListName is the name of the list itself in the YANG schema.
}

// generatedGoListMethod contains the fields required for generating the methods
// that are associated with a list entry within a struct representing a YANG entity.
type generatedGoListMethod struct {
	ListName  string          // ListName is the name of the list for which the method is being generated within its parent struct.
	ListType  string          // ListType is the type (struct name) of the element representing the list.
	Keys      []goStructField // Keys of the list that is being generated (length = 1 if the list is single keyed).
	KeyStruct string          // KeyStruct is the name of the struct used as a key for a multi-keyed list.
	Receiver  string          // Receiver is the name of the parent struct of the list, which is the receiver for the generated method.
}

// generatedGoKeyHelper contains the fields required for generating a method
// associated with a struct that is within a list in the YANG schema.
type generatedGoKeyHelper struct {
	// Receiver is the name of the type which acts as a receiver for a generated method.
	Receiver string
	// Keys specifies the keys of the list, as a map from YANG to Go identifier.
	Keys []*yangFieldMap
}

var (
	// goListKeyTemplate takes an input generatedGoMultiKeyListStruct, which is used to
	// describe the key of a list that has multiple keys, and generates a Go
	// struct definition that can be used to represent that key. For example, if a
	// YANG container or list contains a list, L:
	//
	//	container A {
	//		list L {
	//			key "key-one key-two";
	//			leaf key-one { type string; }
	//			leaf key-two { type uint32; }
	//		}
	//	}
	//
	// A struct is generated to represent the key of list L, of the following form:
	//
	//	type A_L_Key struct {
	//		KeyOne	string
	//		KeyTwo	uint32
	//	}
	//
	// This struct is then used as the key of the map representing the list L, in
	// the generated struct representing the container A.
	goListKeyTemplate = mustMakeTemplate("listkey", `
// {{ .KeyStructName }} represents the key for list {{ .ListName }} of element {{ .ParentPath }}.
type {{ .KeyStructName }} struct {
{{- range $idx, $key := .Keys }}
	{{ $key.Name }}	{{ $key.Type }}	`+"`{{ $key.Tags }}`"+`
{{- end }}
}

// IsYANGGoKeyStruct ensures that {{ .KeyStructName }} partially implements the
// yang.GoKeyStruct interface. This allows functions that need to
// handle this key struct to identify it as being generated by gogen.
func ({{ .KeyStructName }}) IsYANGGoKeyStruct() {}

// ΛListKeyMap returns the values of the {{ .KeyStructName }} key struct.
func (t {{ .KeyStructName }}) ΛListKeyMap() (map[string]interface{}, error) {
	return map[string]interface{}{
		{{- range $key := .Keys }}
		"{{ $key.YANGName }}": t.{{ $key.Name }},
		{{- end }}
	}, nil
}
`)

	// goNewListMemberTemplate takes an input generatedGoListMethod struct and
	// outputs a method, using the specified receiver, that creates a new instance
	// of a struct within a keyed YANG list, and populates the map key, and the
	// key fields of the list's struct according to the input arguments of the
	// function.
	goNewListMemberTemplate = mustMakeTemplate("newListEntry", `
// New{{ .ListName }} creates a new entry in the {{ .ListName }} list of the
// {{ .Receiver}} struct. The keys of the list are populated from the input
// arguments.
func (t *{{ .Receiver }}) New{{ .ListName }}(
  {{- $length := len .Keys -}}
  {{- range $i, $key := .Keys -}}
	{{ $key.Name }} {{ $key.Type -}}
	{{- if ne (inc $i) $length -}}, {{ end -}}
  {{- end -}}
  ) (*{{ .ListType }}, error){

	// Initialise the list within the receiver struct if it has not already been
	// created.
	if t.{{ .ListName }} == nil {
		{{- if ne .KeyStruct "" }}
		t.{{ .ListName }} = make(map[{{ .KeyStruct }}]*{{ .ListType }})
		{{- else }}
			{{- $listName := .ListName -}}
			{{- $listType := .ListType -}}
			{{- range $key := .Keys }}
		t.{{ $listName }} = make(map[{{ $key.Type }}]*{{ $listType }})
			{{- end }} {{- end }}
	}

	{{ if ne .KeyStruct "" -}}
	key := {{ .KeyStruct }}{
		{{- range $key := .Keys }}
		{{ $key.Name }}: {{ $key.Name }},
		{{- end }}
	}
	{{- else -}}
	{{- range $key := .Keys -}}
	key := {{ $key.Name }}
	{{- end -}}
	{{- end }}

	// Ensure that this key has not already been used in the
	// list. Keyed YANG lists do not allow duplicate keys to
	// be created.
	if _, ok := t.{{ .ListName }}[key]; ok {
		return nil, fmt.Errorf("duplicate key %v for list {{ .ListName }}", key)
	}

	t.{{ .ListName }}[key] = &{{ .ListType }}{
		{{- range $key := .Keys }}
		{{- if $key.IsScalarField }}
		{{ $key.Name }}: &{{ $key.Name }},
		{{- else }}
		{{ $key.Name }}: {{ $key.Name }},
		{{- end -}}
		{{- end }}
	}

	return t.{{ .ListName }}[key], nil
}
`)

	// goListGetterTemplate defines a template for a function that, for a particular
	// list key, gets an existing map value.
	goListGetterTemplate = mustMakeTemplate("getList", `
// Get{{ .ListName }} retrieves the value with the specified key from
// the {{ .ListName }} map field of {{ .Receiver }}. If the receiver is nil, or
// the specified key is not present in the list, nil is returned such that Get*
// methods may be safely chained.
func (t *{{ .Receiver }}) Get{{ .ListName }}(
  {{- $length := len .Keys -}}
  {{- range $i, $key := .Keys -}}
	{{ $key.Name }} {{ $key.Type -}}
	{{- if ne (inc $i) $length -}}, {{ end -}}
  {{- end -}}
  ) (*{{ .ListType }}){

	if t == nil {
		return nil
	}

  {{ if ne .KeyStruct "" -}}
	key := {{ .KeyStruct }}{
		{{- range $key := .Keys }}
		{{ $key.Name }}: {{ $key.Name }},
		{{- end }}
	}
	{{- else -}}
	{{- range $key := .Keys -}}
	key := {{ $key.Name }}
	{{- end -}}
	{{- end }}

  if lm, ok := t.{{ .ListName }}[key]; ok {
    return lm
  }
  return nil
}
`)

	// goGetOrCreateListElementTemplate defines a template for a function that, for a
	// particular list key, gets an existing map value, or creates it if it doesn't
	// exist.
	goGetOrCreateListElementTemplate = mustMakeTemplate("getOrCreateListElement", `
// GetOrCreate{{ .ListName }} retrieves the value with the specified keys from
// the receiver {{ .Receiver }}. If the entry does not exist, then it is created.
// It returns the existing or new list member.
func (t *{{ .Receiver }}) GetOrCreate{{ .ListName }}(
  {{- $length := len .Keys -}}
  {{- range $i, $key := .Keys -}}
	{{ $key.Name }} {{ $key.Type -}}
	{{- if ne (inc $i) $length -}}, {{ end -}}
  {{- end -}}
  ) (*{{ .ListType }}){

	{{ if ne .KeyStruct "" -}}
	key := {{ .KeyStruct }}{
		{{- range $key := .Keys }}
		{{ $key.Name }}: {{ $key.Name }},
		{{- end }}
	}
	{{- else -}}
	{{- range $key := .Keys -}}
	key := {{ $key.Name }}
	{{- end -}}
	{{- end }}

	if v, ok := t.{{ .ListName }}[key]; ok {
		return v
	}
	// Panic if we receive an error, since we should have retrieved an existing
	// list member. This allows chaining of GetOrCreate methods.
	v, err := t.New{{ .ListName }}(
		{{- range $i, $key := .Keys -}}
		{{ $key.Name }}
		{{- if ne (inc $i) $length -}}, {{ end -}}
		{{- end -}})
	if err != nil {
		panic(fmt.Sprintf("GetOrCreate{{ .ListName }} got unexpected error: %v", err))
	}
	return v
}
`)

	// goGetOrCreateListTemplate defines a template for a function that
	// returns the current list. It also creates it if it doesn't exist.
	goGetOrCreateListTemplate = mustMakeTemplate("getOrCreateList", `
// GetOrCreate{{ .ListName }}Map returns the list (map) from {{ .Receiver }}.
//
// It initializes the field if not already initialized.
func (t *{{ .Receiver }}) GetOrCreate{{ .ListName }}Map() {{ if ne .KeyStruct "" -}}
		map[{{ .KeyStruct }}]*{{ .ListType }}
		{{- else }}
			{{- $listName := .ListName -}}
			{{- $listType := .ListType -}}
			{{- range $key := .Keys -}}
		map[{{ $key.Type }}]*{{ $listType }}
			{{- end }}
		{{- end }} {
	if t.{{ .ListName }} == nil {
		{{- if ne .KeyStruct "" }}
		t.{{ .ListName }} = make(map[{{ .KeyStruct }}]*{{ .ListType }})
		{{- else }}
			{{- $listName := .ListName -}}
			{{- $listType := .ListType -}}
			{{- range $key := .Keys }}
		t.{{ $listName }} = make(map[{{ $key.Type }}]*{{ $listType }})
			{{- end }}
		{{- end }}
	}
	return t.{{ .ListName }}
}
`)

	// goDeleteListTemplate defines a template for a function that, for a
	// particular list key, deletes an existing map value.
	goDeleteListTemplate = mustMakeTemplate("deleteList", `
// Delete{{ .ListName }} deletes the value with the specified keys from
// the receiver {{ .Receiver }}. If there is no such element, the function
// is a no-op.
func (t *{{ .Receiver }}) Delete{{ .ListName }}(
  {{- $length := len .Keys -}}
  {{- range $i, $key := .Keys -}}
	{{ $key.Name }} {{ $key.Type -}}
	{{- if ne (inc $i) $length -}}, {{ end -}}
  {{- end -}}
  ) {
	{{ if ne .KeyStruct "" -}}
	key := {{ .KeyStruct }}{
		{{- range $key := .Keys }}
		{{ $key.Name }}: {{ $key.Name }},
		{{- end }}
	}
	{{- else -}}
	{{- range $key := .Keys -}}
	key := {{ $key.Name }}
	{{- end -}}
	{{- end }}

	delete(t.{{ .ListName }}, key)
}
`)

	// goListAppendTemplate defines a template for a function that takes an
	// input list member struct, extracts the key value, and appends it to a map.
	// In this template, since all list keys are specified to be pointer types
	// within values by default, we must invert the "IsScalarField" check to
	// ensure that we dereference elements that are pointers in the generated
	// code.
	goListAppendTemplate = mustMakeTemplate("appendList", `
// Append{{ .ListName }} appends the supplied {{ .ListType }} struct to the
// list {{ .ListName }} of {{ .Receiver }}. If the key value(s) specified in
// the supplied {{ .ListType }} already exist in the list, an error is
// returned.
func (t *{{ .Receiver }}) Append{{ .ListName }}(v *{{ .ListType }}) error {
	{{ if ne .KeyStruct "" -}}
	{{- range $key := .Keys }}
	{{- if $key.IsScalarField -}}
	if v.{{ $key.Name }} == nil {
		return fmt.Errorf("invalid nil key for {{ $key.Name }}")
	}

	{{ end -}}
	{{- end -}}
	key := {{ .KeyStruct }}{
		{{- range $key := .Keys }}
		{{- if $key.IsScalarField }}
		{{ $key.Name }}: *v.{{ $key.Name }},
		{{- else }}
		{{ $key.Name }}: v.{{ $key.Name }},
		{{- end -}} 
		{{ end }}
	}
	{{- else -}}
	{{- range $key := .Keys -}}
		{{- if $key.IsScalarField -}}
	if v.{{ $key.Name }} == nil {
		return fmt.Errorf("invalid nil key received for {{ $key.Name }}")
	}

	key := *v.{{ $key.Name }}
		{{- else -}}
	key := v.{{ $key.Name }}
		{{- end -}}
	{{- end -}}
	{{- end }}

	// Initialise the list within the receiver struct if it has not already been
	// created.
	if t.{{ .ListName }} == nil {
		{{- if ne .KeyStruct "" }}
		t.{{ .ListName }} = make(map[{{ .KeyStruct }}]*{{ .ListType }})
		{{- else }}
			{{- $listName := .ListName -}}
			{{- $listType := .ListType -}}
			{{- range $key := .Keys }}
		t.{{ $listName }} = make(map[{{ $key.Type }}]*{{ $listType }})
			{{- end }}
		{{- end }}
	}

	if _, ok := t.{{ .ListName }}[key]; ok {
		return fmt.Errorf("duplicate key for list {{ .ListName }} %v", key)
	}

	t.{{ .ListName }}[key] = v
	return nil
}
`)

	// goListMemberRenameTemplate provides a template for a function which renames
	// an entry within a list. It is used to generate functions for each list within
	// a generated Go struct.
	goListMemberRenameTemplate = mustMakeTemplate("renameListEntry", `
// Rename{{ .ListName }} renames an entry in the list {{ .ListName }} within
// the {{ .Receiver }} struct. The entry with key oldK is renamed to newK updating
// the key within the value.
func (t *{{ .Receiver }}) Rename{{ .ListName }}(
	{{- if ne .KeyStruct "" -}}
	oldK, newK {{ .KeyStruct -}}
  {{- else -}}
	{{- range $key := .Keys -}}
	oldK, newK {{ $key.Type -}}
	{{- end -}}
	{{- end -}}
) error {
	if _, ok := t.{{ .ListName }}[newK]; ok {
		return fmt.Errorf("key %v already exists in {{ .ListName }}", newK)
	}

	e, ok := t.{{ .ListName }}[oldK]
	if !ok {
		return fmt.Errorf("key %v not found in {{ .ListName }}", oldK)
	}

	{{- if ne .KeyStruct "" -}}
	{{- range $key := .Keys -}}
	{{- if $key.IsScalarField }}
	e.{{ $key.Name }} = &newK.{{ $key.Name }}
	{{- else }}
	e.{{ $key.Name }} = newK.{{ $key.Name }}
	{{- end -}}
	{{- end -}}
	{{ else -}}
	{{- $key := index .Keys 0 -}}
	{{- if $key.IsScalarField }}
	e.{{ $key.Name }} = &newK
	{{- else }}
	e.{{ $key.Name }} = newK
	{{- end -}}
	{{- end }}

	t.{{ .ListName }}[newK] = e
	delete(t.{{ .ListName }}, oldK)
	return nil
}
`)

	// goKeyMapTemplate defines the template for a function that is generated for a YANG
	// list type. It returns a map[string]interface{} keyed by the YANG leaf identifier of each
	// key leaf, and containing their values within the struct.
	goKeyMapTemplate = mustMakeTemplate("keyHelper", `
// ΛListKeyMap returns the keys of the {{ .Receiver }} struct, which is a YANG list entry.
func (t *{{ .Receiver }}) ΛListKeyMap() (map[string]interface{}, error) {
{{- range $key := .Keys -}}{{ if $key.IsPtr }}
	if t.{{ $key.GoName }} == nil {
		return nil, fmt.Errorf("nil value for key {{ $key.GoName }}")
	}
	{{- end }}
{{ end }}
	return map[string]interface{}{
		{{- range $key := .Keys }}
		"{{ $key.YANGName }}": {{ if $key.IsPtr -}}
		*
		{{- end -}} t.{{ $key.GoName }},
		{{- end }}
	}, nil
}
`)
)

// generateGetOrCreateList generates a getter function similar to that created
// by the generateGetOrCreateStruct function for maps within the generated Go
// code (which represent YANG lists). It handles both simple and composite key
// lists.
//
// If the list described has a single key, the argument to the function is the
// non-pointer key value. If the list has a complex type, it is an instance of
// the generated key type for the list.
//
// The generated function returns the existing value if the key exists in the
// list, or creates a new value using the NewXXX method if it does not exist.
// The generated function is written to the supplied buffer, using the method
// argument to determine the list's characteristics in the template.
func generateGetOrCreateList(buf *bytes.Buffer, method *generatedGoListMethod) error {
	if err := goGetOrCreateListTemplate.Execute(buf, method); err != nil {
		return err
	}
	return goGetOrCreateListElementTemplate.Execute(buf, method)
}

// generateListGetter generates a getter function for members of the a YANG list
// (Go map) field of the input struct. The generated function takes arguments
// of the same form as those that are given to the GetOrCreate method generated
// by generateGetOrCreateList.
func generateListGetter(buf *bytes.Buffer, method *generatedGoListMethod) error {
	return goListGetterTemplate.Execute(buf, method)
}

// generateListDelete generates a delete function for members of the a YANG list
// (Go map) field of the input struct. The generated function takes arguments
// of the same form as those that are given to the GetOrCreate method generated
// by generateGetOrCreateList.
func generateListDelete(buf *bytes.Buffer, method *generatedGoListMethod) error {
	return goDeleteListTemplate.Execute(buf, method)
}

// generateListAppend generates a function which appends a (key, value) to a
// Go map (YANG list) within the generated code. The argument of the generated
// function is the map's member type - from which the key values are extracted.
// The generated function returns an error if the key already exists in the list.
//
// The generated function is written to the supplied buffer - using the supplied
// method argument to determine the list's characteristics in the template.
func generateListAppend(buf *bytes.Buffer, method *generatedGoListMethod) error {
	return goListAppendTemplate.Execute(buf, method)
}

// generateGetListKey generates a function extracting the keys from a list
// defined in the Directory s, and appends it to the supplier buffer. The
// nameMap stores maps between the key YANG field identifiers and their Go
// identifiers.
//
// If the input Directory is the following list entry:
//
//	list foo {
//	  key "bar baz";
//
//	  leaf bar { type string; }
//	  leaf baz { type uint8; }
//	  leaf colour { type string; }
//	}
//
// Which is mapped into the Go struct:
//
//	type Foo {
//	  Bar *string `path:"bar"`
//	  Baz *uint8  `path:"baz"`
//	  Colour *string `path:"colour"`
//	}
//
// The generated method will;
//   - Check pointer keys to ensure they are non-nil.
//   - Return a map[string]interface{} keyed by the name of the key in the YANG schema, with the value
//     specified in the struct.
//
// i.e.: for the above struct:
//
//	 func (t *Foo) ΛListKeyMap() (map[string]interface{}, error) {
//		if t.Bar == nil {
//		   return nil, fmt.Errorf("key value for Bar is nil")
//		}
//
//		if t.Baz == nil {
//		   return nil, fmt.Errorf("key value for Baz is nil")
//		}
//
//		return map[string]interface{}{
//		  "bar": *t.Bar,
//		  "baz": *t.Baz,
//		}
//	 }
func generateGetListKey(buf *bytes.Buffer, s *ygen.ParsedDirectory, nameMap map[string]*yangFieldMap) error {
	if s.ListKeys == nil {
		return nil
	}

	h := generatedGoKeyHelper{
		Receiver: s.Name,
	}

	kn := []string{}
	for k := range s.ListKeys {
		kn = append(kn, k)
	}
	sort.Strings(kn)

	for _, k := range kn {
		h.Keys = append(h.Keys, nameMap[k])
	}

	return goKeyMapTemplate.Execute(buf, h)
}

// UnorderedMapTypeName returns the map and key type names of an
// unordered, keyed map given go-generated IR information, as well as whether
// it is a defined type rather than a Go built-in type.
//
// e.g. for a list to be represented as map[string]*Foo, it returns
// "map[string]*Foo", "string", false, nil
func UnorderedMapTypeName(listYANGPath, listFieldName, parentName string, goStructElements map[string]*ygen.ParsedDirectory) (string, string, bool, error) {
	// The list itself, since it is a container, has a struct associated with it. Retrieve
	// this from the set of Directory structs for which code (a Go struct) will be
	//  generated such that additional details can be used in the code generation.
	listElem, ok := goStructElements[listYANGPath]
	if !ok {
		return "", "", false, fmt.Errorf("struct for %s did not exist", listYANGPath)
	}

	var listType, keyType string
	var isDefinedType bool
	switch len(listElem.ListKeys) {
	case 0:
		return "", "", false, fmt.Errorf("list does not contain any keys: %s:", listElem.Name)
	case 1:
		// This is a single keyed list, so we can represent it as a map with
		// a simple Go type as the key. Note that a leaf-list can never be
		// a key, so we do not need to handle the case whereby we would have to
		// have a slice which keys the list.
		for _, listKey := range listElem.ListKeys {
			listType = fmt.Sprintf("map[%s]*%s", listKey.LangType.NativeType, listElem.Name)
			keyType = listKey.LangType.NativeType
			isDefinedType = ygen.IsYgenDefinedGoType(listKey.LangType)
		}
	default:
		// This is a list with multiple keys, so we need to generate a new structure
		// that represents the list key itself - this struct is described in a
		// generatedGoMultiKeyListStruct struct, which is then expanded by a template to the struct
		// definition.
		listKeyStructName := fmt.Sprintf("%s_Key", listElem.Name)
		names := make(map[string]bool, len(goStructElements))
		for _, d := range goStructElements {
			names[d.Name] = true
		}
		if names[listKeyStructName] {
			listKeyStructName = fmt.Sprintf("%s_%s_YANGListKey", parentName, listFieldName)
			if names[listKeyStructName] {
				return "", "", false, fmt.Errorf("unexpected generated list key name conflict for %s", listYANGPath)
			}
			names[listKeyStructName] = true
		}
		listType = fmt.Sprintf("map[%s]*%s", listKeyStructName, listElem.Name)
		keyType = listKeyStructName
		isDefinedType = true
	}
	return listType, keyType, isDefinedType, nil
}

// yangListFieldToGoType takes a yang node description (listField) and returns
// a string corresponding to the Go type that should be used to represent it
// within its parent struct (the parent argument). If applicable, it also
// returns generated code specifications for list-associated Go types. If a
// particular specification is not applicable, then nil is returned for that
// type.
//
// In all cases, the type of list field is the struct which is defined to
// reference the list, used as the base type. This type is then modified based
// on how the list is keyed:
//   - If the list is a config false, keyless list - a slice of the list's type is returned.
//   - If the list has a single key, a map, keyed by the single key's type is returned.
//   - If the list has multiple keys, a new struct is defined which represents the set of
//     leaves that make up the key. The type of the list is then a map, keyed by the new struct
//     type.
//   - If the list has "ordered-by user", then for the single and multiple key
//     cases, a struct that represents an ordered map keyed by the same key
//     type as the unordered map representation described above.
//
// In the case that the list has multiple keys, the type generated as the key of the list is returned.
// If errors are encountered during the type generation for the list, the error is returned.
func yangListFieldToGoType(listField *ygen.NodeDetails, listFieldName string, parent *ygen.ParsedDirectory, goStructElements map[string]*ygen.ParsedDirectory, generateOrderedMaps bool) (string, *generatedGoMultiKeyListStruct, *generatedGoListMethod, *generatedOrderedMapStruct, error) {
	// The list itself, since it is a container, has a struct associated with it. Retrieve
	// this from the set of Directory structs for which code (a Go struct) will be
	//  generated such that additional details can be used in the code generation.
	listElem, ok := goStructElements[listField.YANGDetails.Path]
	if !ok {
		return "", nil, nil, nil, fmt.Errorf("struct for %s did not exist", listField.YANGDetails.Path)
	}

	if len(listElem.ListKeys) == 0 {
		// Keyless list therefore represent this as a slice of pointers to
		// the struct that represents the list element itself.
		return fmt.Sprintf("[]*%s", listElem.Name), nil, nil, nil, nil
	}

	listType, keyType, _, err := UnorderedMapTypeName(listField.YANGDetails.Path, listFieldName, parent.Name, goStructElements)
	if err != nil {
		return "", nil, nil, nil, err
	}
	var multiListKey *generatedGoMultiKeyListStruct
	var listKeys []goStructField

	shortestPath := func(ss [][]string) [][]string {
		var shortest []string
		for _, s := range ss {
			if shortest == nil {
				shortest = s
				continue
			}
			if len(s) < len(shortest) {
				shortest = s
			}
		}
		return [][]string{shortest}
	}

	usedKeyElemNames := make(map[string]bool)
	for _, keName := range listElem.ListKeyYANGNames {
		keyType, ok := listElem.Fields[keName]
		if !ok {
			return "", nil, nil, nil, fmt.Errorf("did not find type for key %s", keName)
		}

		keyField := goStructField{
			YANGName: keName,
			Name:     genutil.MakeNameUnique(listElem.ListKeys[keName].Name, usedKeyElemNames),
			Type:     listElem.ListKeys[keName].LangType.NativeType,
			// The shortest mapped path for a list key must be the path to the key.
			Tags: mappedPathTag(shortestPath(keyType.MappedPaths), ""),
		}
		keyField.IsScalarField = IsScalarField(keyType)
		listKeys = append(listKeys, keyField)
	}

	switch {
	case len(listElem.ListKeys) != 1:
		// This is a list with multiple keys, so we need to generate a new structure
		// that represents the list key itself - this struct is described in a
		// generatedGoMultiKeyListStruct struct, which is then expanded by a template to the struct
		// definition.
		multiListKey = &generatedGoMultiKeyListStruct{
			KeyStructName: keyType,
			ParentPath:    parent.Path,
			ListName:      listFieldName,
			Keys:          listKeys,
		}
	}

	var listMethodSpec *generatedGoListMethod
	var orderedMapSpec *generatedOrderedMapStruct

	if listField.YANGDetails.OrderedByUser && generateOrderedMaps {
		structName := OrderedMapTypeName(listElem.Name)
		listType = fmt.Sprintf("*%s", structName)
		// Create spec for generating ordered maps.
		orderedMapSpec = &generatedOrderedMapStruct{
			StructName:       structName,
			KeyName:          keyType,
			ListTypeName:     listElem.Name,
			ListFieldName:    listFieldName,
			Keys:             listKeys,
			ParentStructName: parent.Name,
			YANGPath:         listField.YANGDetails.Path,
		}
	} else {
		// Generate the specification for the methods that should be generated for this
		// list, such that this can be handed to the relevant templates to generate code.
		listMethodSpec = &generatedGoListMethod{
			ListName: listFieldName,
			ListType: listElem.Name,
			Keys:     listKeys,
			Receiver: parent.Name,
		}
		if multiListKey != nil {
			listMethodSpec.KeyStruct = keyType
		}
	}

	return listType, multiListKey, listMethodSpec, orderedMapSpec, nil
}
