// Copyright (C) 2018 Michael J. Fromberger. All Rights Reserved.

// Package ctrl manages passage of control through a main function.
//
// In case of error main programs typically call os.Exit or log.Fatal.
// However, this causes the process to terminate immediately and deferred calls
// are not invoked. Calling log.Panic allows deferred calls to run, but makes a
// noisy log trace.
//
// This package provides a Run function that performs the main action of a
// program. Within its dynamic extent, calls to ctrl.Exit and ctrl.Exitf will
// panic back to Run, which will handle logging and exiting from the process as
// specified.
//
// Example:
//
//    import "github.com/creachadair/ctrl"
//
//    // A stub main to set up the control handlers.
//    func main() { ctrl.Run(realMain) }
//
//    // The real program logic goes into this function.
//    func realMain() error { ... }
//
package ctrl

import (
	"fmt"
	"log"
	"os"
	"strconv"
	"sync"
)

var exitHook = struct {
	sync.Mutex
	hook func(code int, err error)
	exit func(code int)
}{
	hook: func(int, error) {},
	exit: os.Exit,
}

func hookedExit(code int, err error) {
	exitHook.Lock()
	defer exitHook.Unlock()
	exitHook.hook(code, err)
	exitHook.exit(code)
}

// SetHook sets f as an exit hook.
//
// Before Run exits the program, it will call f synchronously with the code and
// error that motivated the exit.  Once the hook returns, Run will exit.
//
// If f == nil, the hook is removed.  There is only one exit hook; subsequent
// calls to SetHook will replace any previous values.
func SetHook(f func(code int, err error)) {
	exitHook.Lock()
	defer exitHook.Unlock()
	if f == nil {
		exitHook.hook = func(int, error) {} // do nothing
	} else {
		exitHook.hook = f
	}
}

// SetPanic sets whether the program should exit by panicking (true) or by
// calling os.Exit (false). The default is false.
//
// This is mainly intended to support testing.
func SetPanic(panicToExit bool) {
	exitHook.Lock()
	defer exitHook.Unlock()
	if panicToExit {
		exitHook.exit = func(int) { panic("ctrl: program terminated") }
	} else {
		exitHook.exit = os.Exit
	}
}

// Run invokes main. If main returns without error, control returns from Run
// normally. If main reports an error, it is logged and Run calls os.Exit(1).
//
// During the execution of main a call to ctrl.Exit or ctrl.Exitf returns
// control to Run, which logs a message and calls os.Exit with the specified
// return code. Control is handled via the panic mechanism, so deferred calls
// within main will be executed normally. Panics not generated by this library
// are propagated normally.
func Run(main func() error) {
	defer func() {
		var err error
		var ecode *int

		v := recover()
		if e, ok := v.(*logExit); ok {
			// control returned via Exitf
			ecode = &e.code
			err = e
		} else if e, ok := v.(quietExit); ok {
			// control returned via Exit
			ecode = new(int)
			*ecode = int(e)
		} else if e, ok := v.(plainError); ok {
			// control returned normally from main
			if e.error != nil {
				// failure, exit with an error
				ecode = new(int)
				*ecode = 1
				err = e.error
			}
		} else {
			// an unrelated panic; pass it along
			panic(v)
		}

		if err != nil {
			log.Print(err)
		}
		if ecode != nil {
			hookedExit(*ecode, err)
		}
		// let control fall off main
	}()
	panic(plainError{main()})
}

type plainError struct{ error }

// Exit returns control to the most recent invocation of Run, instructing it to
// exit the process silently with the specified exit status.
//
// Control does not return from a call to Exit. The return type is error so
// that it can be called in an error return statement.
func Exit(code int) error { panic(quietExit(code)) }

type quietExit int

func (q quietExit) Error() string { return strconv.Itoa(int(q)) }

// Exitf returns control to the most recent invocation of Run, instructing it
// to exit the process with the specified exit status and formatted log.
//
// Control does not return from a call to Exitf. The return type is error so
// that it can be called in an error return statement.
func Exitf(code int, msg string, args ...interface{}) error {
	panic(&logExit{code: code, msg: msg, args: args})
}

type logExit struct {
	code int
	msg  string
	args []interface{}
}

func (e *logExit) Error() string { return fmt.Sprintf(e.msg, e.args...) }

// Fatalf is a shorthand for ctrl.Exitf(1, ...), that can be used as a drop-in
// replacement for calls to log.Fatalf.
func Fatalf(msg string, args ...interface{}) error { return Exitf(1, msg, args...) }
