// Copyright 2022 - MinIO, Inc. All rights reserved.
// Use of this source code is governed by the AGPLv3
// license that can be found in the LICENSE file.

package kes

import (
	"bytes"
	"context"
	"encoding/json"
	"io"
	"net/http"
	"net/url"
	"path"
	"time"
)

// EnclaveInfo describes a KES enclave.
type EnclaveInfo struct {
	Name      string    `json:"name"`       // Enclave name
	CreatedAt time.Time `json:"created_at"` // Point in time when the enclave has been created
	CreatedBy Identity  `json:"created_by"` // Identity that created the enclave
}

// An Enclave is an isolated area within a KES server.
// It stores cryptographic keys, policies and other
// related information securely.
//
// A KES server contains at least one Enclave and,
// depending upon its persistence layer, may be able
// to hold many Enclaves.
//
// With Enclaves, a KES server implements multi-tenancy.
type Enclave struct {
	name      string
	endpoints []string
	client    retry
}

// CreateKey creates a new cryptographic key. The key will
// be generated by the KES server.
//
// It returns ErrKeyExists if a key with the same name already
// exists.
func (e *Enclave) CreateKey(ctx context.Context, name string) error {
	const (
		APIPath  = "/v1/key/create"
		Method   = http.MethodPost
		StatusOK = http.StatusOK
	)
	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), nil)
	if err != nil {
		return err
	}
	if resp.StatusCode != StatusOK {
		return parseErrorResponse(resp)
	}
	return nil
}

// ImportKey imports the given key into a KES server. It
// returns ErrKeyExists if a key with the same key already
// exists.
func (e *Enclave) ImportKey(ctx context.Context, name string, key []byte) error {
	const (
		APIPath  = "/v1/key/import"
		Method   = http.MethodPost
		StatusOK = http.StatusOK
	)
	type Request struct {
		Bytes []byte `json:"bytes"`
	}
	body, err := json.Marshal(Request{
		Bytes: key,
	})
	if err != nil {
		return err
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), bytes.NewReader(body), withHeader("Content-Type", "application/json"))
	if err != nil {
		return err
	}
	if resp.StatusCode != StatusOK {
		return parseErrorResponse(resp)
	}
	return nil
}

// DescribeKey returns the KeyInfo for the given key.
// It returns ErrKeyNotFound if no such key exists.
func (e *Enclave) DescribeKey(ctx context.Context, name string) (*KeyInfo, error) {
	const (
		APIPath         = "/v1/key/describe"
		Method          = http.MethodGet
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20
	)
	type Response struct {
		Name      string    `json:"name"`
		ID        string    `json:"id"`
		CreatedAt time.Time `json:"created_at"`
		CreatedBy Identity  `json:"created_by"`
	}
	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), nil)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}

	var response Response
	if err := json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return nil, err
	}
	return &KeyInfo{
		Name:      response.Name,
		ID:        response.ID,
		CreatedAt: response.CreatedAt,
		CreatedBy: response.CreatedBy,
	}, nil
}

// DeleteKey deletes the key from a KES server. It returns
// ErrKeyNotFound if no such key exists.
func (e *Enclave) DeleteKey(ctx context.Context, name string) error {
	const (
		APIPath  = "/v1/key/delete"
		Method   = http.MethodDelete
		StatusOK = http.StatusOK
	)

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), nil)
	if err != nil {
		return err
	}
	if resp.StatusCode != StatusOK {
		return parseErrorResponse(resp)
	}
	return nil
}

// GenerateKey returns a new generated data encryption key (DEK).
// A DEK has a plaintext and ciphertext representation.
//
// The former should be used for cryptographic operations, like
// encrypting some data.
//
// The later is the result of encrypting the plaintext with the named
// key at the KES server. It should be stored at a durable location but
// does not need to stay secret. The ciphertext can only be decrypted
// with the named key at the KES server.
//
// The context is cryptographically bound to the ciphertext and the
// same context value must be provided when decrypting the ciphertext
// via Decrypt. Therefore, an application must either remember the
// context or must be able to re-generate it.
//
// GenerateKey returns ErrKeyNotFound if no key with the given name
// exists.
func (e *Enclave) GenerateKey(ctx context.Context, name string, context []byte) (DEK, error) {
	const (
		APIPath         = "/v1/key/generate"
		Method          = http.MethodPost
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20 // 1 MiB
	)
	type Request struct {
		Context []byte `json:"context,omitempty"` // A context is optional
	}
	type Response struct {
		Plaintext  []byte `json:"plaintext"`
		Ciphertext []byte `json:"ciphertext"`
	}

	body, err := json.Marshal(Request{
		Context: context,
	})
	if err != nil {
		return DEK{}, err
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), bytes.NewReader(body), withHeader("Content-Type", "application/json"))
	if err != nil {
		return DEK{}, err
	}
	if resp.StatusCode != StatusOK {
		return DEK{}, parseErrorResponse(resp)
	}
	defer resp.Body.Close()

	var response Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return DEK{}, err
	}
	return DEK(response), nil
}

// Encrypt encrypts the given plaintext with the named key at the
// KES server. The optional context is cryptographically bound to
// the returned ciphertext. The exact same context must be provided
// when decrypting the ciphertext again.
//
// Encrypt returns ErrKeyNotFound if no such key exists at the KES
// server.
func (e *Enclave) Encrypt(ctx context.Context, name string, plaintext, context []byte) ([]byte, error) {
	const (
		APIPath         = "/v1/key/encrypt"
		Method          = http.MethodPost
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20 // 1 MiB
	)
	type Request struct {
		Plaintext []byte `json:"plaintext"`
		Context   []byte `json:"context,omitempty"` // A context is optional
	}
	type Response struct {
		Ciphertext []byte `json:"ciphertext"`
	}

	body, err := json.Marshal(Request{
		Plaintext: plaintext,
		Context:   context,
	})
	if err != nil {
		return nil, err
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), bytes.NewReader(body), withHeader("Content-Type", "application/json"))
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}
	defer resp.Body.Close()

	var response Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return nil, err
	}
	return response.Ciphertext, nil
}

// Decrypt decrypts the ciphertext with the named key at the KES
// server. The exact same context, used during Encrypt, must be
// provided.
//
// Decrypt returns ErrKeyNotFound if no such key exists. It returns
// ErrDecrypt when the ciphertext has been modified or a different
// context value is provided.
func (e *Enclave) Decrypt(ctx context.Context, name string, ciphertext, context []byte) ([]byte, error) {
	const (
		APIPath         = "/v1/key/decrypt"
		Method          = http.MethodPost
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20 // 1 MiB
	)
	type Request struct {
		Ciphertext []byte `json:"ciphertext"`
		Context    []byte `json:"context,omitempty"` // A context is optional
	}
	type Response struct {
		Plaintext []byte `json:"plaintext"`
	}
	body, err := json.Marshal(Request{
		Ciphertext: ciphertext,
		Context:    context,
	})
	if err != nil {
		return nil, err
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), bytes.NewReader(body), withHeader("Content-Type", "application/json"))
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}
	defer resp.Body.Close()

	var response Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return nil, err
	}
	return response.Plaintext, nil
}

// DecryptAll decrypts all ciphertexts with the named key at the
// KES server. It either returns all decrypted plaintexts or the
// first decryption error.
//
// DecryptAll returns ErrKeyNotFound if the specified key does not
// exist. It returns ErrDecrypt if any ciphertext has been modified
// or a different context value was used.
func (e *Enclave) DecryptAll(ctx context.Context, name string, ciphertexts ...CCP) ([]PCP, error) {
	const (
		APIPath         = "/v1/key/bulk/decrypt"
		Method          = http.MethodPost
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20
	)
	type Request struct {
		Ciphertext []byte `json:"ciphertext"`
		Context    []byte `json:"context,omitempty"` // A context is optional
	}
	type Response struct {
		Plaintext []byte `json:"plaintext"`
	}
	if len(ciphertexts) == 0 {
		return []PCP{}, nil
	}
	requests := make([]Request, 0, len(ciphertexts))
	for i := range ciphertexts {
		requests = append(requests, Request{
			Ciphertext: ciphertexts[i].Ciphertext,
			Context:    ciphertexts[i].Context,
		})
	}

	body, err := json.Marshal(requests)
	if err != nil {
		return nil, err
	}
	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), bytes.NewReader(body), withHeader("Content-Type", "application/json"))
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}
	defer resp.Body.Close()

	var responses []Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&responses); err != nil {
		return nil, err
	}

	plaintexts := make([]PCP, 0, len(responses))
	for i, response := range responses {
		plaintexts = append(plaintexts, PCP{
			Plaintext: response.Plaintext,
			Context:   requests[i].Context,
		})
	}
	return plaintexts, nil
}

// ListKeys lists all names of cryptographic keys that match the given
// pattern. It returns a KeyIterator that iterates over all matched key
// names.
//
// The pattern matching happens on the server side. If pattern is empty
// the KeyIterator iterates over all key names.
func (e *Enclave) ListKeys(ctx context.Context, pattern string) (*KeyIterator, error) {
	const (
		APIPath  = "/v1/key/list"
		Method   = http.MethodGet
		StatusOK = http.StatusOK
	)

	if pattern == "" { // The empty pattern never matches anything
		const MatchAll = "*"
		pattern = MatchAll
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, pattern), nil)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}
	return &KeyIterator{
		decoder: json.NewDecoder(resp.Body),
		closer:  resp.Body,
	}, nil
}

// AssignPolicy assigns the policy to the identity.
// The KES admin identity cannot be assigned to any
// policy.
//
// AssignPolicy returns PolicyNotFound if no such policy exists.
func (e *Enclave) AssignPolicy(ctx context.Context, policy string, identity Identity) error {
	const (
		APIPath  = "/v1/policy/assign"
		Method   = http.MethodPost
		StatusOK = http.StatusOK
	)
	type Request struct {
		Identity Identity `json:"identity"`
	}

	body, err := json.Marshal(Request{Identity: identity})
	if err != nil {
		return err
	}
	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, policy), bytes.NewReader(body))
	if err != nil {
		return err
	}
	if resp.StatusCode != StatusOK {
		return parseErrorResponse(resp)
	}
	return nil
}

// SetPolicy creates the given policy. If a policy with the same
// name already exists, SetPolicy overwrites the existing policy
// with the given one. Any existing identites will be assigned to
// the given policy.
func (e *Enclave) SetPolicy(ctx context.Context, name string, policy *Policy) error {
	const (
		APIPath  = "/v1/policy/write"
		Method   = http.MethodPost
		StatusOK = http.StatusOK
	)

	body, err := json.Marshal(policy)
	if err != nil {
		return err
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), bytes.NewReader(body), withHeader("Content-Type", "application/json"))
	if err != nil {
		return err
	}
	if resp.StatusCode != StatusOK {
		return parseErrorResponse(resp)
	}
	return nil
}

// DescribePolicy returns the PolicyInfo for the given policy.
// It returns ErrPolicyNotFound if no such policy exists.
func (e *Enclave) DescribePolicy(ctx context.Context, name string) (*PolicyInfo, error) {
	const (
		APIPath         = "/v1/policy/describe"
		Method          = http.MethodGet
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20 // 1 MiB
	)
	type Response struct {
		CreatedAt time.Time `json:"created_at"`
		CreatedBy Identity  `json:"created_by"`
	}
	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), nil)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}

	var response Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return nil, err
	}
	return &PolicyInfo{
		Name:      name,
		CreatedAt: response.CreatedAt,
		CreatedBy: response.CreatedBy,
	}, nil
}

// GetPolicy returns the policy with the given name.
// It returns ErrPolicyNotFound if no such policy
// exists.
func (e *Enclave) GetPolicy(ctx context.Context, name string) (*Policy, error) {
	const (
		APIPath         = "/v1/policy/read"
		Method          = http.MethodGet
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20 // 1 MiB
	)
	type Response struct {
		Allow     []string  `json:"allow"`
		Deny      []string  `json:"deny"`
		CreatedAt time.Time `json:"created_at"`
		CreatedBy Identity  `json:"created_by"`
	}
	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), nil)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}

	var response Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return nil, err
	}
	return &Policy{
		Allow: response.Allow,
		Deny:  response.Deny,
		Info: PolicyInfo{
			Name:      name,
			CreatedAt: response.CreatedAt,
			CreatedBy: response.CreatedBy,
		},
	}, nil
}

// DeletePolicy deletes the policy with the given name. Any
// assigned identities will be removed as well.
//
// It returns ErrPolicyNotFound if no such policy exists.
func (e *Enclave) DeletePolicy(ctx context.Context, name string) error {
	const (
		APIPath  = "/v1/policy/delete"
		Method   = http.MethodDelete
		StatusOK = http.StatusOK
	)

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, name), nil)
	if err != nil {
		return err
	}
	if resp.StatusCode != StatusOK {
		return parseErrorResponse(resp)
	}
	return nil
}

// ListPolicies lists all policy names that match the given pattern.
//
// The pattern matching happens on the server side. If pattern is empty
// ListPolicies returns all policy names.
func (e *Enclave) ListPolicies(ctx context.Context, pattern string) (*PolicyIterator, error) {
	const (
		APIPath  = "/v1/policy/list"
		Method   = http.MethodGet
		StatusOK = http.StatusOK
	)

	if pattern == "" { // The empty pattern never matches anything
		const MatchAll = "*"
		pattern = MatchAll
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, pattern), nil)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}
	return &PolicyIterator{
		decoder: json.NewDecoder(resp.Body),
		closer:  resp.Body,
	}, nil
}

// DescribeIdentity returns an IdentityInfo describing the given identity.
func (e *Enclave) DescribeIdentity(ctx context.Context, identity Identity) (*IdentityInfo, error) {
	const (
		APIPath         = "/v1/identity/describe"
		Method          = http.MethodGet
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20 // 1 MiB
	)
	type Response struct {
		IsAdmin   bool      `json:"admin"`
		Policy    string    `json:"policy"`
		CreatedAt time.Time `json:"created_at"`
		CreatedBy Identity  `json:"created_by"`
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, identity.String()), nil)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}
	var response Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return nil, err
	}
	return &IdentityInfo{
		Identity:  identity,
		Policy:    response.Policy,
		IsAdmin:   response.IsAdmin,
		CreatedAt: response.CreatedAt,
		CreatedBy: response.CreatedBy,
	}, nil
}

// DescribeSelf returns an IdentityInfo describing the identity
// making the API request. It also returns the assigned policy,
// if any.
//
// DescribeSelf allows an application to obtain identity and
// policy information about itself.
func (e *Enclave) DescribeSelf(ctx context.Context) (*IdentityInfo, *Policy, error) {
	const (
		APIPath         = "/v1/identity/self/describe"
		Method          = http.MethodGet
		StatusOK        = http.StatusOK
		MaxResponseSize = 1 << 20 // 1 MiB
	)
	type InlinePolicy struct {
		Allow     []string  `json:"allow"`
		Deny      []string  `json:"deny"`
		CreatedAt time.Time `json:"created_at"`
		CreatedBy Identity  `json:"created_by"`
	}
	type Response struct {
		Identity   Identity     `json:"identity"`
		IsAdmin    bool         `json:"admin"`
		PolicyName string       `json:"policy_name"`
		CreatedAt  time.Time    `json:"created_at"`
		CreatedBy  Identity     `json:"created_by"`
		Policy     InlinePolicy `json:"policy"`
	}

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath), nil)
	if err != nil {
		return nil, nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, nil, parseErrorResponse(resp)
	}
	var response Response
	if err = json.NewDecoder(io.LimitReader(resp.Body, MaxResponseSize)).Decode(&response); err != nil {
		return nil, nil, err
	}
	info := &IdentityInfo{
		Identity:  response.Identity,
		Policy:    response.PolicyName,
		CreatedAt: response.CreatedAt,
		CreatedBy: response.CreatedBy,
		IsAdmin:   response.IsAdmin,
	}
	policy := &Policy{
		Allow: response.Policy.Allow,
		Deny:  response.Policy.Deny,
		Info: PolicyInfo{
			Name:      response.PolicyName,
			CreatedAt: response.Policy.CreatedAt,
			CreatedBy: response.Policy.CreatedBy,
		},
	}
	return info, policy, nil
}

// DeleteIdentity removes the identity. Once removed, any
// operation issued by this identity will fail with
// ErrNotAllowed.
//
// The KES admin identity cannot be removed.
func (e *Enclave) DeleteIdentity(ctx context.Context, identity Identity) error {
	const (
		APIPath  = "/v1/identity/delete"
		Method   = http.MethodDelete
		StatusOK = http.StatusOK
	)

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, identity.String()), nil)
	if err != nil {
		return err
	}
	if resp.StatusCode != StatusOK {
		return parseErrorResponse(resp)
	}
	return nil
}

// ListIdentities lists all identites that match the given pattern.
//
// The pattern matching happens on the server side. If pattern is empty
// ListIdentities returns all identities.
func (e *Enclave) ListIdentities(ctx context.Context, pattern string) (*IdentityIterator, error) {
	const (
		APIPath  = "/v1/identity/list"
		Method   = http.MethodGet
		StatusOK = http.StatusOK
	)

	resp, err := e.client.Send(ctx, Method, e.endpoints, e.path(APIPath, pattern), nil)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != StatusOK {
		return nil, parseErrorResponse(resp)
	}
	return &IdentityIterator{
		decoder: json.NewDecoder(resp.Body),
		closer:  resp.Body,
	}, nil
}

func (e *Enclave) path(api string, args ...string) string {
	for _, arg := range args {
		api = path.Join(api, url.PathEscape(arg))
	}
	if e.name != "" {
		api = "?enclave=" + url.QueryEscape(e.name)
	}
	return api
}
