// Copyright 2026 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package meet provides access to the Google Meet API.
//
// For product documentation, see: https://developers.google.com/workspace/meet/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/meet/v2"
//	...
//	ctx := context.Background()
//	meetService, err := meet.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	meetService, err := meet.NewService(ctx, option.WithScopes(meet.MeetingsSpaceSettingsScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	meetService, err := meet.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	meetService, err := meet.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package meet // import "google.golang.org/api/meet/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "meet:v2"
const apiName = "meet"
const apiVersion = "v2"
const basePath = "https://meet.googleapis.com/"
const basePathTemplate = "https://meet.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://meet.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Create, edit, and see information about your Google Meet conferences created
	// by the app.
	MeetingsSpaceCreatedScope = "https://www.googleapis.com/auth/meetings.space.created"

	// Read information about any of your Google Meet conferences
	MeetingsSpaceReadonlyScope = "https://www.googleapis.com/auth/meetings.space.readonly"

	// Edit, and see settings for all of your Google Meet calls.
	MeetingsSpaceSettingsScope = "https://www.googleapis.com/auth/meetings.space.settings"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/meetings.space.created",
		"https://www.googleapis.com/auth/meetings.space.readonly",
		"https://www.googleapis.com/auth/meetings.space.settings",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.ConferenceRecords = NewConferenceRecordsService(s)
	s.Spaces = NewSpacesService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	ConferenceRecords *ConferenceRecordsService

	Spaces *SpacesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewConferenceRecordsService(s *Service) *ConferenceRecordsService {
	rs := &ConferenceRecordsService{s: s}
	rs.Participants = NewConferenceRecordsParticipantsService(s)
	rs.Recordings = NewConferenceRecordsRecordingsService(s)
	rs.Transcripts = NewConferenceRecordsTranscriptsService(s)
	return rs
}

type ConferenceRecordsService struct {
	s *Service

	Participants *ConferenceRecordsParticipantsService

	Recordings *ConferenceRecordsRecordingsService

	Transcripts *ConferenceRecordsTranscriptsService
}

func NewConferenceRecordsParticipantsService(s *Service) *ConferenceRecordsParticipantsService {
	rs := &ConferenceRecordsParticipantsService{s: s}
	rs.ParticipantSessions = NewConferenceRecordsParticipantsParticipantSessionsService(s)
	return rs
}

type ConferenceRecordsParticipantsService struct {
	s *Service

	ParticipantSessions *ConferenceRecordsParticipantsParticipantSessionsService
}

func NewConferenceRecordsParticipantsParticipantSessionsService(s *Service) *ConferenceRecordsParticipantsParticipantSessionsService {
	rs := &ConferenceRecordsParticipantsParticipantSessionsService{s: s}
	return rs
}

type ConferenceRecordsParticipantsParticipantSessionsService struct {
	s *Service
}

func NewConferenceRecordsRecordingsService(s *Service) *ConferenceRecordsRecordingsService {
	rs := &ConferenceRecordsRecordingsService{s: s}
	return rs
}

type ConferenceRecordsRecordingsService struct {
	s *Service
}

func NewConferenceRecordsTranscriptsService(s *Service) *ConferenceRecordsTranscriptsService {
	rs := &ConferenceRecordsTranscriptsService{s: s}
	rs.Entries = NewConferenceRecordsTranscriptsEntriesService(s)
	return rs
}

type ConferenceRecordsTranscriptsService struct {
	s *Service

	Entries *ConferenceRecordsTranscriptsEntriesService
}

func NewConferenceRecordsTranscriptsEntriesService(s *Service) *ConferenceRecordsTranscriptsEntriesService {
	rs := &ConferenceRecordsTranscriptsEntriesService{s: s}
	return rs
}

type ConferenceRecordsTranscriptsEntriesService struct {
	s *Service
}

func NewSpacesService(s *Service) *SpacesService {
	rs := &SpacesService{s: s}
	return rs
}

type SpacesService struct {
	s *Service
}

// ActiveConference: Active conference.
type ActiveConference struct {
	// ConferenceRecord: Output only. Reference to 'ConferenceRecord' resource.
	// Format: `conferenceRecords/{conference_record}` where `{conference_record}`
	// is a unique ID for each instance of a call within a space.
	ConferenceRecord string `json:"conferenceRecord,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ConferenceRecord") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ConferenceRecord") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ActiveConference) MarshalJSON() ([]byte, error) {
	type NoMethod ActiveConference
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AnonymousUser: User who joins anonymously (meaning not signed into a Google
// Account).
type AnonymousUser struct {
	// DisplayName: Output only. User provided name when they join a conference
	// anonymously.
	DisplayName string `json:"displayName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AnonymousUser) MarshalJSON() ([]byte, error) {
	type NoMethod AnonymousUser
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ArtifactConfig: Configuration related to meeting artifacts potentially
// generated by this meeting space.
type ArtifactConfig struct {
	// RecordingConfig: Configuration for recording.
	RecordingConfig *RecordingConfig `json:"recordingConfig,omitempty"`
	// SmartNotesConfig: Configuration for auto-smart-notes.
	SmartNotesConfig *SmartNotesConfig `json:"smartNotesConfig,omitempty"`
	// TranscriptionConfig: Configuration for auto-transcript.
	TranscriptionConfig *TranscriptionConfig `json:"transcriptionConfig,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RecordingConfig") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RecordingConfig") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ArtifactConfig) MarshalJSON() ([]byte, error) {
	type NoMethod ArtifactConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ConferenceRecord: Single instance of a meeting held in a space.
type ConferenceRecord struct {
	// EndTime: Output only. Timestamp when the conference ended. Set for past
	// conferences. Unset if the conference is ongoing.
	EndTime string `json:"endTime,omitempty"`
	// ExpireTime: Output only. Server enforced expiration time for when this
	// conference record resource is deleted. The resource is deleted 30 days after
	// the conference ends.
	ExpireTime string `json:"expireTime,omitempty"`
	// Name: Identifier. Resource name of the conference record. Format:
	// `conferenceRecords/{conference_record}` where `{conference_record}` is a
	// unique ID for each instance of a call within a space.
	Name string `json:"name,omitempty"`
	// Space: Output only. The space where the conference was held.
	Space string `json:"space,omitempty"`
	// StartTime: Output only. Timestamp when the conference started. Always set.
	StartTime string `json:"startTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "EndTime") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ConferenceRecord) MarshalJSON() ([]byte, error) {
	type NoMethod ConferenceRecord
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DocsDestination: Google Docs location where the transcript file is saved.
type DocsDestination struct {
	// Document: Output only. The document ID for the underlying Google Docs
	// transcript file. For example, "1kuceFZohVoCh6FulBHxwy6I15Ogpc4hP". Use the
	// `documents.get` method of the Google Docs API
	// (https://developers.google.com/docs/api/reference/rest/v1/documents/get) to
	// fetch the content.
	Document string `json:"document,omitempty"`
	// ExportUri: Output only. URI for the Google Docs transcript file. Use
	// `https://docs.google.com/document/d/{$DocumentId}/view` to browse the
	// transcript in the browser.
	ExportUri string `json:"exportUri,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Document") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Document") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DocsDestination) MarshalJSON() ([]byte, error) {
	type NoMethod DocsDestination
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveDestination: Export location where a recording file is saved in Google
// Drive.
type DriveDestination struct {
	// ExportUri: Output only. Link used to play back the recording file in the
	// browser. For example, `https://drive.google.com/file/d/{$fileId}/view`.
	ExportUri string `json:"exportUri,omitempty"`
	// File: Output only. The `fileId` for the underlying MP4 file. For example,
	// "1kuceFZohVoCh6FulBHxwy6I15Ogpc4hP". Use `$ GET
	// https://www.googleapis.com/drive/v3/files/{$fileId}?alt=media` to download
	// the blob. For more information, see
	// https://developers.google.com/drive/api/v3/reference/files/get.
	File string `json:"file,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExportUri") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExportUri") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveDestination) MarshalJSON() ([]byte, error) {
	type NoMethod DriveDestination
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// EndActiveConferenceRequest: Request to end an ongoing conference of a space.
type EndActiveConferenceRequest struct {
}

// ListConferenceRecordsResponse: Response of ListConferenceRecords method.
type ListConferenceRecordsResponse struct {
	// ConferenceRecords: List of conferences in one page.
	ConferenceRecords []*ConferenceRecord `json:"conferenceRecords,omitempty"`
	// NextPageToken: Token to be circulated back for further List call if current
	// List does NOT include all the Conferences. Unset if all conferences have
	// been returned.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ConferenceRecords") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ConferenceRecords") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListConferenceRecordsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListConferenceRecordsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListParticipantSessionsResponse: Response of ListParticipants method.
type ListParticipantSessionsResponse struct {
	// NextPageToken: Token to be circulated back for further List call if current
	// List doesn't include all the participants. Unset if all participants are
	// returned.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// ParticipantSessions: List of participants in one page.
	ParticipantSessions []*ParticipantSession `json:"participantSessions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListParticipantSessionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListParticipantSessionsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListParticipantsResponse: Response of ListParticipants method.
type ListParticipantsResponse struct {
	// NextPageToken: Token to be circulated back for further List call if current
	// List doesn't include all the participants. Unset if all participants are
	// returned.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Participants: List of participants in one page.
	Participants []*Participant `json:"participants,omitempty"`
	// TotalSize: Total, exact number of `participants`. By default, this field
	// isn't included in the response. Set the field mask in SystemParameterContext
	// (https://cloud.google.com/apis/docs/system-parameters) to receive this field
	// in the response.
	TotalSize int64 `json:"totalSize,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListParticipantsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListParticipantsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListRecordingsResponse: Response for ListRecordings method.
type ListRecordingsResponse struct {
	// NextPageToken: Token to be circulated back for further List call if current
	// List doesn't include all the recordings. Unset if all recordings are
	// returned.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Recordings: List of recordings in one page.
	Recordings []*Recording `json:"recordings,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListRecordingsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListRecordingsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListTranscriptEntriesResponse: Response for ListTranscriptEntries method.
type ListTranscriptEntriesResponse struct {
	// NextPageToken: Token to be circulated back for further List call if current
	// List doesn't include all the transcript entries. Unset if all entries are
	// returned.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// TranscriptEntries: List of TranscriptEntries in one page.
	TranscriptEntries []*TranscriptEntry `json:"transcriptEntries,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListTranscriptEntriesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListTranscriptEntriesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListTranscriptsResponse: Response for ListTranscripts method.
type ListTranscriptsResponse struct {
	// NextPageToken: Token to be circulated back for further List call if current
	// List doesn't include all the transcripts. Unset if all transcripts are
	// returned.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Transcripts: List of transcripts in one page.
	Transcripts []*Transcript `json:"transcripts,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListTranscriptsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListTranscriptsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ModerationRestrictions: Defines restrictions for features when the meeting
// is moderated.
type ModerationRestrictions struct {
	// ChatRestriction: Defines who has permission to send chat messages in the
	// meeting space.
	//
	// Possible values:
	//   "RESTRICTION_TYPE_UNSPECIFIED" - Default value specified by user policy.
	// This should never be returned.
	//   "HOSTS_ONLY" - Meeting owner and co-host have the permission.
	//   "NO_RESTRICTION" - All Participants have permissions.
	ChatRestriction string `json:"chatRestriction,omitempty"`
	// DefaultJoinAsViewerType: Defines whether to restrict the default role
	// assigned to users as viewer.
	//
	// Possible values:
	//   "DEFAULT_JOIN_AS_VIEWER_TYPE_UNSPECIFIED" - Default value specified by
	// user policy. This should never be returned.
	//   "ON" - Users will by default join as viewers.
	//   "OFF" - Users will by default join as contributors.
	DefaultJoinAsViewerType string `json:"defaultJoinAsViewerType,omitempty"`
	// PresentRestriction: Defines who has permission to share their screen in the
	// meeting space.
	//
	// Possible values:
	//   "RESTRICTION_TYPE_UNSPECIFIED" - Default value specified by user policy.
	// This should never be returned.
	//   "HOSTS_ONLY" - Meeting owner and co-host have the permission.
	//   "NO_RESTRICTION" - All Participants have permissions.
	PresentRestriction string `json:"presentRestriction,omitempty"`
	// ReactionRestriction: Defines who has permission to send reactions in the
	// meeting space.
	//
	// Possible values:
	//   "RESTRICTION_TYPE_UNSPECIFIED" - Default value specified by user policy.
	// This should never be returned.
	//   "HOSTS_ONLY" - Meeting owner and co-host have the permission.
	//   "NO_RESTRICTION" - All Participants have permissions.
	ReactionRestriction string `json:"reactionRestriction,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ChatRestriction") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChatRestriction") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ModerationRestrictions) MarshalJSON() ([]byte, error) {
	type NoMethod ModerationRestrictions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Participant: User who attended or is attending a conference.
type Participant struct {
	// AnonymousUser: Anonymous user.
	AnonymousUser *AnonymousUser `json:"anonymousUser,omitempty"`
	// EarliestStartTime: Output only. Time when the participant first joined the
	// meeting.
	EarliestStartTime string `json:"earliestStartTime,omitempty"`
	// LatestEndTime: Output only. Time when the participant left the meeting for
	// the last time. This can be null if it's an active meeting.
	LatestEndTime string `json:"latestEndTime,omitempty"`
	// Name: Output only. Resource name of the participant. Format:
	// `conferenceRecords/{conference_record}/participants/{participant}`
	Name string `json:"name,omitempty"`
	// PhoneUser: User calling from their phone.
	PhoneUser *PhoneUser `json:"phoneUser,omitempty"`
	// SignedinUser: Signed-in user.
	SignedinUser *SignedinUser `json:"signedinUser,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AnonymousUser") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AnonymousUser") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Participant) MarshalJSON() ([]byte, error) {
	type NoMethod Participant
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ParticipantSession: Refers to each unique join or leave session when a user
// joins a conference from a device. Note that any time a user joins the
// conference a new unique ID is assigned. That means if a user joins a space
// multiple times from the same device, they're assigned different IDs, and are
// also be treated as different participant sessions.
type ParticipantSession struct {
	// EndTime: Output only. Timestamp when the user session ends. Unset if the
	// user session hasn’t ended.
	EndTime string `json:"endTime,omitempty"`
	// Name: Identifier. Session id.
	Name string `json:"name,omitempty"`
	// StartTime: Output only. Timestamp when the user session starts.
	StartTime string `json:"startTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "EndTime") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ParticipantSession) MarshalJSON() ([]byte, error) {
	type NoMethod ParticipantSession
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PhoneUser: User dialing in from a phone where the user's identity is unknown
// because they haven't signed in with a Google Account.
type PhoneUser struct {
	// DisplayName: Output only. Partially redacted user's phone number when
	// calling.
	DisplayName string `json:"displayName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PhoneUser) MarshalJSON() ([]byte, error) {
	type NoMethod PhoneUser
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Recording: Metadata about a recording created during a conference.
type Recording struct {
	// DriveDestination: Output only. Recording is saved to Google Drive as an MP4
	// file. The `drive_destination` includes the Drive `fileId` that can be used
	// to download the file using the `files.get` method of the Drive API.
	DriveDestination *DriveDestination `json:"driveDestination,omitempty"`
	// EndTime: Output only. Timestamp when the recording ended.
	EndTime string `json:"endTime,omitempty"`
	// Name: Output only. Resource name of the recording. Format:
	// `conferenceRecords/{conference_record}/recordings/{recording}` where
	// `{recording}` is a 1:1 mapping to each unique recording session during the
	// conference.
	Name string `json:"name,omitempty"`
	// StartTime: Output only. Timestamp when the recording started.
	StartTime string `json:"startTime,omitempty"`
	// State: Output only. Current state.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Default, never used.
	//   "STARTED" - An active recording session has started.
	//   "ENDED" - This recording session has ended, but the recording file hasn't
	// been generated yet.
	//   "FILE_GENERATED" - Recording file is generated and ready to download.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DriveDestination") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DriveDestination") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Recording) MarshalJSON() ([]byte, error) {
	type NoMethod Recording
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RecordingConfig: Configuration related to recording in a meeting space.
type RecordingConfig struct {
	// AutoRecordingGeneration: Defines whether a meeting space is automatically
	// recorded when someone with the privilege to record joins the meeting.
	//
	// Possible values:
	//   "AUTO_GENERATION_TYPE_UNSPECIFIED" - Default value specified by user
	// policy. This should never be returned.
	//   "ON" - The artifact is generated automatically.
	//   "OFF" - The artifact is not generated automatically.
	AutoRecordingGeneration string `json:"autoRecordingGeneration,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AutoRecordingGeneration") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AutoRecordingGeneration") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RecordingConfig) MarshalJSON() ([]byte, error) {
	type NoMethod RecordingConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SignedinUser: A signed-in user can be: a) An individual joining from a
// personal computer, mobile device, or through companion mode. b) A robot
// account used by conference room devices.
type SignedinUser struct {
	// DisplayName: Output only. For a personal device, it's the user's first name
	// and last name. For a robot account, it's the administrator-specified device
	// name. For example, "Altostrat Room".
	DisplayName string `json:"displayName,omitempty"`
	// User: Output only. Unique ID for the user. Interoperable with Admin SDK API
	// and People API. Format: `users/{user}`
	User string `json:"user,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SignedinUser) MarshalJSON() ([]byte, error) {
	type NoMethod SignedinUser
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SmartNotesConfig: Configuration related to smart notes in a meeting space.
// For more information about smart notes, see "Take notes for me" in Google
// Meet (https://support.google.com/meet/answer/14754931).
type SmartNotesConfig struct {
	// AutoSmartNotesGeneration: Defines whether to automatically generate a
	// summary and recap of the meeting for all invitees in the organization when
	// someone with the privilege to enable smart notes joins the meeting.
	//
	// Possible values:
	//   "AUTO_GENERATION_TYPE_UNSPECIFIED" - Default value specified by user
	// policy. This should never be returned.
	//   "ON" - The artifact is generated automatically.
	//   "OFF" - The artifact is not generated automatically.
	AutoSmartNotesGeneration string `json:"autoSmartNotesGeneration,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AutoSmartNotesGeneration")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AutoSmartNotesGeneration") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SmartNotesConfig) MarshalJSON() ([]byte, error) {
	type NoMethod SmartNotesConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Space: Virtual place where conferences are held. Only one active conference
// can be held in one space at any given time.
type Space struct {
	// ActiveConference: Active conference, if it exists.
	ActiveConference *ActiveConference `json:"activeConference,omitempty"`
	// Config: Configuration pertaining to the meeting space.
	Config *SpaceConfig `json:"config,omitempty"`
	// MeetingCode: Output only. Type friendly unique string used to join the
	// meeting. Format: `[a-z]+-[a-z]+-[a-z]+`. For example, `abc-mnop-xyz`. The
	// maximum length is 128 characters. Can only be used as an alias of the space
	// name to get the space.
	MeetingCode string `json:"meetingCode,omitempty"`
	// MeetingUri: Output only. URI used to join meetings consisting of
	// `https://meet.google.com/` followed by the `meeting_code`. For example,
	// `https://meet.google.com/abc-mnop-xyz`.
	MeetingUri string `json:"meetingUri,omitempty"`
	// Name: Immutable. Resource name of the space. Format: `spaces/{space}`.
	// `{space}` is the resource identifier for the space. It's a unique,
	// server-generated ID and is case sensitive. For example, `jQCFfuBOdN5z`. For
	// more information, see How Meet identifies a meeting space
	// (https://developers.google.com/workspace/meet/api/guides/meeting-spaces#identify-meeting-space).
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ActiveConference") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActiveConference") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Space) MarshalJSON() ([]byte, error) {
	type NoMethod Space
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SpaceConfig: The configuration pertaining to a meeting space.
type SpaceConfig struct {
	// AccessType: Access type of the meeting space that determines who can join
	// without knocking. Default: The user's default access settings. Controlled by
	// the user's admin for enterprise users or RESTRICTED.
	//
	// Possible values:
	//   "ACCESS_TYPE_UNSPECIFIED" - Default value specified by the user's
	// organization. Note: This is never returned, as the configured access type is
	// returned instead.
	//   "OPEN" - Anyone with the join information (for example, the URL or phone
	// access information) can join without knocking.
	//   "TRUSTED" - Members of the host's organization, invited external users,
	// and dial-in users can join without knocking. Everyone else must knock.
	//   "RESTRICTED" - Only invitees can join without knocking. Everyone else must
	// knock.
	AccessType string `json:"accessType,omitempty"`
	// ArtifactConfig: Configuration pertaining to the auto-generated artifacts
	// that the meeting supports.
	ArtifactConfig *ArtifactConfig `json:"artifactConfig,omitempty"`
	// AttendanceReportGenerationType: Whether attendance report is enabled for the
	// meeting space.
	//
	// Possible values:
	//   "ATTENDANCE_REPORT_GENERATION_TYPE_UNSPECIFIED" - Default value specified
	// by user policy. This should never be returned.
	//   "GENERATE_REPORT" - Attendance report will be generated and sent to
	// drive/email.
	//   "DO_NOT_GENERATE" - Attendance report will not be generated.
	AttendanceReportGenerationType string `json:"attendanceReportGenerationType,omitempty"`
	// EntryPointAccess: Defines the entry points that can be used to join meetings
	// hosted in this meeting space. Default: EntryPointAccess.ALL
	//
	// Possible values:
	//   "ENTRY_POINT_ACCESS_UNSPECIFIED" - Unused.
	//   "ALL" - All entry points are allowed.
	//   "CREATOR_APP_ONLY" - Only entry points owned by the Google Cloud project
	// that created the space can be used to join meetings in this space. Apps can
	// use the Meet Embed SDK Web or mobile Meet SDKs to create owned entry points.
	EntryPointAccess string `json:"entryPointAccess,omitempty"`
	// Moderation: The pre-configured moderation mode for the Meeting. Default:
	// Controlled by the user's policies.
	//
	// Possible values:
	//   "MODERATION_UNSPECIFIED" - Moderation type is not specified. This is used
	// to indicate the user hasn't specified any value as the user does not intend
	// to update the state. Users are not allowed to set the value as unspecified.
	//   "OFF" - Moderation is off.
	//   "ON" - Moderation is on.
	Moderation string `json:"moderation,omitempty"`
	// ModerationRestrictions: When moderation.ON, these restrictions go into
	// effect for the meeting. When moderation.OFF, will be reset to default
	// ModerationRestrictions.
	ModerationRestrictions *ModerationRestrictions `json:"moderationRestrictions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SpaceConfig) MarshalJSON() ([]byte, error) {
	type NoMethod SpaceConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Transcript: Metadata for a transcript generated from a conference. It refers
// to the ASR (Automatic Speech Recognition) result of user's speech during the
// conference.
type Transcript struct {
	// DocsDestination: Output only. Where the Google Docs transcript is saved.
	DocsDestination *DocsDestination `json:"docsDestination,omitempty"`
	// EndTime: Output only. Timestamp when the transcript stopped.
	EndTime string `json:"endTime,omitempty"`
	// Name: Output only. Resource name of the transcript. Format:
	// `conferenceRecords/{conference_record}/transcripts/{transcript}`, where
	// `{transcript}` is a 1:1 mapping to each unique transcription session of the
	// conference.
	Name string `json:"name,omitempty"`
	// StartTime: Output only. Timestamp when the transcript started.
	StartTime string `json:"startTime,omitempty"`
	// State: Output only. Current state.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Default, never used.
	//   "STARTED" - An active transcript session has started.
	//   "ENDED" - This transcript session has ended, but the transcript file
	// hasn't been generated yet.
	//   "FILE_GENERATED" - Transcript file is generated and ready to download.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DocsDestination") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DocsDestination") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Transcript) MarshalJSON() ([]byte, error) {
	type NoMethod Transcript
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TranscriptEntry: Single entry for one user’s speech during a transcript
// session.
type TranscriptEntry struct {
	// EndTime: Output only. Timestamp when the transcript entry ended.
	EndTime string `json:"endTime,omitempty"`
	// LanguageCode: Output only. Language of spoken text, such as "en-US". IETF
	// BCP 47 syntax (https://tools.ietf.org/html/bcp47)
	LanguageCode string `json:"languageCode,omitempty"`
	// Name: Output only. Resource name of the entry. Format:
	// "conferenceRecords/{conference_record}/transcripts/{transcript}/entries/{entr
	// y}"
	Name string `json:"name,omitempty"`
	// Participant: Output only. Refers to the participant who speaks.
	Participant string `json:"participant,omitempty"`
	// StartTime: Output only. Timestamp when the transcript entry started.
	StartTime string `json:"startTime,omitempty"`
	// Text: Output only. The transcribed text of the participant's voice, at
	// maximum 10K words. Note that the limit is subject to change.
	Text string `json:"text,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "EndTime") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TranscriptEntry) MarshalJSON() ([]byte, error) {
	type NoMethod TranscriptEntry
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TranscriptionConfig: Configuration related to transcription in a meeting
// space.
type TranscriptionConfig struct {
	// AutoTranscriptionGeneration: Defines whether the content of a meeting is
	// automatically transcribed when someone with the privilege to transcribe
	// joins the meeting.
	//
	// Possible values:
	//   "AUTO_GENERATION_TYPE_UNSPECIFIED" - Default value specified by user
	// policy. This should never be returned.
	//   "ON" - The artifact is generated automatically.
	//   "OFF" - The artifact is not generated automatically.
	AutoTranscriptionGeneration string `json:"autoTranscriptionGeneration,omitempty"`
	// ForceSendFields is a list of field names (e.g.
	// "AutoTranscriptionGeneration") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. See https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields
	// for more details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AutoTranscriptionGeneration") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TranscriptionConfig) MarshalJSON() ([]byte, error) {
	type NoMethod TranscriptionConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type ConferenceRecordsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a conference record by conference ID.
//
// - name: Resource name of the conference.
func (r *ConferenceRecordsService) Get(name string) *ConferenceRecordsGetCall {
	c := &ConferenceRecordsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsGetCall) Fields(s ...googleapi.Field) *ConferenceRecordsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsGetCall) IfNoneMatch(entityTag string) *ConferenceRecordsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsGetCall) Context(ctx context.Context) *ConferenceRecordsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ConferenceRecord.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ConferenceRecordsGetCall) Do(opts ...googleapi.CallOption) (*ConferenceRecord, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ConferenceRecord{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ConferenceRecordsListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the conference records. By default, ordered by start time and in
// descending order.
func (r *ConferenceRecordsService) List() *ConferenceRecordsListCall {
	c := &ConferenceRecordsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Filter sets the optional parameter "filter": User specified filtering
// condition in EBNF format
// (https://en.wikipedia.org/wiki/Extended_Backus%E2%80%93Naur_form). The
// following are the filterable fields: * `space.meeting_code` * `space.name` *
// `start_time` * `end_time` For example, consider the following filters: *
// `space.name = "spaces/NAME" * `space.meeting_code = "abc-mnop-xyz" *
// `start_time>="2024-01-01T00:00:00.000Z" AND
// start_time<="2024-01-02T00:00:00.000Z" * `end_time IS NULL`
func (c *ConferenceRecordsListCall) Filter(filter string) *ConferenceRecordsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of
// conference records to return. The service might return fewer than this
// value. If unspecified, at most 25 conference records are returned. The
// maximum value is 100; values above 100 are coerced to 100. Maximum might
// change in the future.
func (c *ConferenceRecordsListCall) PageSize(pageSize int64) *ConferenceRecordsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token returned from
// previous List Call.
func (c *ConferenceRecordsListCall) PageToken(pageToken string) *ConferenceRecordsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsListCall) Fields(s ...googleapi.Field) *ConferenceRecordsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsListCall) IfNoneMatch(entityTag string) *ConferenceRecordsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsListCall) Context(ctx context.Context) *ConferenceRecordsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/conferenceRecords")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListConferenceRecordsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ConferenceRecordsListCall) Do(opts ...googleapi.CallOption) (*ListConferenceRecordsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListConferenceRecordsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ConferenceRecordsListCall) Pages(ctx context.Context, f func(*ListConferenceRecordsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ConferenceRecordsParticipantsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a participant by participant ID.
//
// - name: Resource name of the participant.
func (r *ConferenceRecordsParticipantsService) Get(name string) *ConferenceRecordsParticipantsGetCall {
	c := &ConferenceRecordsParticipantsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsParticipantsGetCall) Fields(s ...googleapi.Field) *ConferenceRecordsParticipantsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsParticipantsGetCall) IfNoneMatch(entityTag string) *ConferenceRecordsParticipantsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsParticipantsGetCall) Context(ctx context.Context) *ConferenceRecordsParticipantsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsParticipantsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsParticipantsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.participants.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Participant.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ConferenceRecordsParticipantsGetCall) Do(opts ...googleapi.CallOption) (*Participant, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Participant{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ConferenceRecordsParticipantsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the participants in a conference record. By default, ordered by
// join time and in descending order. This API supports `fields` as standard
// parameters like every other API. However, when the `fields` request
// parameter is omitted, this API defaults to `'participants/*,
// next_page_token'`.
//
// - parent: Format: `conferenceRecords/{conference_record}`.
func (r *ConferenceRecordsParticipantsService) List(parent string) *ConferenceRecordsParticipantsListCall {
	c := &ConferenceRecordsParticipantsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": User specified filtering
// condition in EBNF format
// (https://en.wikipedia.org/wiki/Extended_Backus%E2%80%93Naur_form). The
// following are the filterable fields: * `earliest_start_time` *
// `latest_end_time` For example, `latest_end_time IS NULL` returns active
// participants in the conference.
func (c *ConferenceRecordsParticipantsListCall) Filter(filter string) *ConferenceRecordsParticipantsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of
// participants to return. The service might return fewer than this value. If
// unspecified, at most 100 participants are returned. The maximum value is
// 250; values above 250 are coerced to 250. Maximum might change in the
// future.
func (c *ConferenceRecordsParticipantsListCall) PageSize(pageSize int64) *ConferenceRecordsParticipantsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token returned from
// previous List Call.
func (c *ConferenceRecordsParticipantsListCall) PageToken(pageToken string) *ConferenceRecordsParticipantsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsParticipantsListCall) Fields(s ...googleapi.Field) *ConferenceRecordsParticipantsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsParticipantsListCall) IfNoneMatch(entityTag string) *ConferenceRecordsParticipantsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsParticipantsListCall) Context(ctx context.Context) *ConferenceRecordsParticipantsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsParticipantsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsParticipantsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/participants")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.participants.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListParticipantsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ConferenceRecordsParticipantsListCall) Do(opts ...googleapi.CallOption) (*ListParticipantsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListParticipantsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ConferenceRecordsParticipantsListCall) Pages(ctx context.Context, f func(*ListParticipantsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ConferenceRecordsParticipantsParticipantSessionsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a participant session by participant session ID.
//
// - name: Resource name of the participant.
func (r *ConferenceRecordsParticipantsParticipantSessionsService) Get(name string) *ConferenceRecordsParticipantsParticipantSessionsGetCall {
	c := &ConferenceRecordsParticipantsParticipantSessionsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsParticipantsParticipantSessionsGetCall) Fields(s ...googleapi.Field) *ConferenceRecordsParticipantsParticipantSessionsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsParticipantsParticipantSessionsGetCall) IfNoneMatch(entityTag string) *ConferenceRecordsParticipantsParticipantSessionsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsParticipantsParticipantSessionsGetCall) Context(ctx context.Context) *ConferenceRecordsParticipantsParticipantSessionsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsParticipantsParticipantSessionsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsParticipantsParticipantSessionsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.participantSessions.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.participants.participantSessions.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ParticipantSession.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ConferenceRecordsParticipantsParticipantSessionsGetCall) Do(opts ...googleapi.CallOption) (*ParticipantSession, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ParticipantSession{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.participantSessions.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ConferenceRecordsParticipantsParticipantSessionsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the participant sessions of a participant in a conference
// record. By default, ordered by join time and in descending order. This API
// supports `fields` as standard parameters like every other API. However, when
// the `fields` request parameter is omitted this API defaults to
// `'participantsessions/*, next_page_token'`.
//
//   - parent: Format:
//     `conferenceRecords/{conference_record}/participants/{participant}`.
func (r *ConferenceRecordsParticipantsParticipantSessionsService) List(parent string) *ConferenceRecordsParticipantsParticipantSessionsListCall {
	c := &ConferenceRecordsParticipantsParticipantSessionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": User specified filtering
// condition in EBNF format
// (https://en.wikipedia.org/wiki/Extended_Backus%E2%80%93Naur_form). The
// following are the filterable fields: * `start_time` * `end_time` For
// example, `end_time IS NULL` returns active participant sessions in the
// conference record.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) Filter(filter string) *ConferenceRecordsParticipantsParticipantSessionsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of
// participant sessions to return. The service might return fewer than this
// value. If unspecified, at most 100 participants are returned. The maximum
// value is 250; values above 250 are coerced to 250. Maximum might change in
// the future.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) PageSize(pageSize int64) *ConferenceRecordsParticipantsParticipantSessionsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token returned from
// previous List Call.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) PageToken(pageToken string) *ConferenceRecordsParticipantsParticipantSessionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) Fields(s ...googleapi.Field) *ConferenceRecordsParticipantsParticipantSessionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) IfNoneMatch(entityTag string) *ConferenceRecordsParticipantsParticipantSessionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) Context(ctx context.Context) *ConferenceRecordsParticipantsParticipantSessionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/participantSessions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.participantSessions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.participants.participantSessions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListParticipantSessionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) Do(opts ...googleapi.CallOption) (*ListParticipantSessionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListParticipantSessionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.participants.participantSessions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ConferenceRecordsParticipantsParticipantSessionsListCall) Pages(ctx context.Context, f func(*ListParticipantSessionsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ConferenceRecordsRecordingsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a recording by recording ID.
//
// - name: Resource name of the recording.
func (r *ConferenceRecordsRecordingsService) Get(name string) *ConferenceRecordsRecordingsGetCall {
	c := &ConferenceRecordsRecordingsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsRecordingsGetCall) Fields(s ...googleapi.Field) *ConferenceRecordsRecordingsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsRecordingsGetCall) IfNoneMatch(entityTag string) *ConferenceRecordsRecordingsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsRecordingsGetCall) Context(ctx context.Context) *ConferenceRecordsRecordingsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsRecordingsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsRecordingsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.recordings.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.recordings.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Recording.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ConferenceRecordsRecordingsGetCall) Do(opts ...googleapi.CallOption) (*Recording, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Recording{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.recordings.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ConferenceRecordsRecordingsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the recording resources from the conference record. By default,
// ordered by start time and in ascending order.
//
// - parent: Format: `conferenceRecords/{conference_record}`.
func (r *ConferenceRecordsRecordingsService) List(parent string) *ConferenceRecordsRecordingsListCall {
	c := &ConferenceRecordsRecordingsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of
// recordings to return. The service might return fewer than this value. If
// unspecified, at most 10 recordings are returned. The maximum value is 100;
// values above 100 are coerced to 100. Maximum might change in the future.
func (c *ConferenceRecordsRecordingsListCall) PageSize(pageSize int64) *ConferenceRecordsRecordingsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token returned from
// previous List Call.
func (c *ConferenceRecordsRecordingsListCall) PageToken(pageToken string) *ConferenceRecordsRecordingsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsRecordingsListCall) Fields(s ...googleapi.Field) *ConferenceRecordsRecordingsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsRecordingsListCall) IfNoneMatch(entityTag string) *ConferenceRecordsRecordingsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsRecordingsListCall) Context(ctx context.Context) *ConferenceRecordsRecordingsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsRecordingsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsRecordingsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/recordings")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.recordings.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.recordings.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListRecordingsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ConferenceRecordsRecordingsListCall) Do(opts ...googleapi.CallOption) (*ListRecordingsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListRecordingsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.recordings.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ConferenceRecordsRecordingsListCall) Pages(ctx context.Context, f func(*ListRecordingsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ConferenceRecordsTranscriptsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a transcript by transcript ID.
//
// - name: Resource name of the transcript.
func (r *ConferenceRecordsTranscriptsService) Get(name string) *ConferenceRecordsTranscriptsGetCall {
	c := &ConferenceRecordsTranscriptsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsTranscriptsGetCall) Fields(s ...googleapi.Field) *ConferenceRecordsTranscriptsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsTranscriptsGetCall) IfNoneMatch(entityTag string) *ConferenceRecordsTranscriptsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsTranscriptsGetCall) Context(ctx context.Context) *ConferenceRecordsTranscriptsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsTranscriptsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsTranscriptsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.transcripts.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Transcript.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ConferenceRecordsTranscriptsGetCall) Do(opts ...googleapi.CallOption) (*Transcript, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Transcript{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ConferenceRecordsTranscriptsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the set of transcripts from the conference record. By default,
// ordered by start time and in ascending order.
//
// - parent: Format: `conferenceRecords/{conference_record}`.
func (r *ConferenceRecordsTranscriptsService) List(parent string) *ConferenceRecordsTranscriptsListCall {
	c := &ConferenceRecordsTranscriptsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of
// transcripts to return. The service might return fewer than this value. If
// unspecified, at most 10 transcripts are returned. The maximum value is 100;
// values above 100 are coerced to 100. Maximum might change in the future.
func (c *ConferenceRecordsTranscriptsListCall) PageSize(pageSize int64) *ConferenceRecordsTranscriptsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token returned from
// previous List Call.
func (c *ConferenceRecordsTranscriptsListCall) PageToken(pageToken string) *ConferenceRecordsTranscriptsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsTranscriptsListCall) Fields(s ...googleapi.Field) *ConferenceRecordsTranscriptsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsTranscriptsListCall) IfNoneMatch(entityTag string) *ConferenceRecordsTranscriptsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsTranscriptsListCall) Context(ctx context.Context) *ConferenceRecordsTranscriptsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsTranscriptsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsTranscriptsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/transcripts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.transcripts.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListTranscriptsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ConferenceRecordsTranscriptsListCall) Do(opts ...googleapi.CallOption) (*ListTranscriptsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListTranscriptsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ConferenceRecordsTranscriptsListCall) Pages(ctx context.Context, f func(*ListTranscriptsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ConferenceRecordsTranscriptsEntriesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a `TranscriptEntry` resource by entry ID. Note: The transcript
// entries returned by the Google Meet API might not match the transcription
// found in the Google Docs transcript file. This can occur when 1) we have
// interleaved speakers within milliseconds, or 2) the Google Docs transcript
// file is modified after generation.
//
// - name: Resource name of the `TranscriptEntry`.
func (r *ConferenceRecordsTranscriptsEntriesService) Get(name string) *ConferenceRecordsTranscriptsEntriesGetCall {
	c := &ConferenceRecordsTranscriptsEntriesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsTranscriptsEntriesGetCall) Fields(s ...googleapi.Field) *ConferenceRecordsTranscriptsEntriesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsTranscriptsEntriesGetCall) IfNoneMatch(entityTag string) *ConferenceRecordsTranscriptsEntriesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsTranscriptsEntriesGetCall) Context(ctx context.Context) *ConferenceRecordsTranscriptsEntriesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsTranscriptsEntriesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsTranscriptsEntriesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.entries.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.transcripts.entries.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *TranscriptEntry.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ConferenceRecordsTranscriptsEntriesGetCall) Do(opts ...googleapi.CallOption) (*TranscriptEntry, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TranscriptEntry{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.entries.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ConferenceRecordsTranscriptsEntriesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the structured transcript entries per transcript. By default,
// ordered by start time and in ascending order. Note: The transcript entries
// returned by the Google Meet API might not match the transcription found in
// the Google Docs transcript file. This can occur when 1) we have interleaved
// speakers within milliseconds, or 2) the Google Docs transcript file is
// modified after generation.
//
//   - parent: Format:
//     `conferenceRecords/{conference_record}/transcripts/{transcript}`.
func (r *ConferenceRecordsTranscriptsEntriesService) List(parent string) *ConferenceRecordsTranscriptsEntriesListCall {
	c := &ConferenceRecordsTranscriptsEntriesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of entries
// to return. The service might return fewer than this value. If unspecified,
// at most 10 entries are returned. The maximum value is 100; values above 100
// are coerced to 100. Maximum might change in the future.
func (c *ConferenceRecordsTranscriptsEntriesListCall) PageSize(pageSize int64) *ConferenceRecordsTranscriptsEntriesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token returned from
// previous List Call.
func (c *ConferenceRecordsTranscriptsEntriesListCall) PageToken(pageToken string) *ConferenceRecordsTranscriptsEntriesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ConferenceRecordsTranscriptsEntriesListCall) Fields(s ...googleapi.Field) *ConferenceRecordsTranscriptsEntriesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ConferenceRecordsTranscriptsEntriesListCall) IfNoneMatch(entityTag string) *ConferenceRecordsTranscriptsEntriesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ConferenceRecordsTranscriptsEntriesListCall) Context(ctx context.Context) *ConferenceRecordsTranscriptsEntriesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ConferenceRecordsTranscriptsEntriesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ConferenceRecordsTranscriptsEntriesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+parent}/entries")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.entries.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.conferenceRecords.transcripts.entries.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListTranscriptEntriesResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ConferenceRecordsTranscriptsEntriesListCall) Do(opts ...googleapi.CallOption) (*ListTranscriptEntriesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListTranscriptEntriesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.conferenceRecords.transcripts.entries.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ConferenceRecordsTranscriptsEntriesListCall) Pages(ctx context.Context, f func(*ListTranscriptEntriesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type SpacesCreateCall struct {
	s          *Service
	space      *Space
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a space.
func (r *SpacesService) Create(space *Space) *SpacesCreateCall {
	c := &SpacesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.space = space
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesCreateCall) Fields(s ...googleapi.Field) *SpacesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesCreateCall) Context(ctx context.Context) *SpacesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.space)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/spaces")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.spaces.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.spaces.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesCreateCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.spaces.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesEndActiveConferenceCall struct {
	s                          *Service
	nameid                     string
	endactiveconferencerequest *EndActiveConferenceRequest
	urlParams_                 gensupport.URLParams
	ctx_                       context.Context
	header_                    http.Header
}

// EndActiveConference: Ends an active conference (if there's one). For an
// example, see End active conference
// (https://developers.google.com/workspace/meet/api/guides/meeting-spaces#end-active-conference).
//
//   - name: Resource name of the space. Format: `spaces/{space}`. `{space}` is
//     the resource identifier for the space. It's a unique, server-generated ID
//     and is case sensitive. For example, `jQCFfuBOdN5z`. For more information,
//     see How Meet identifies a meeting space
//     (https://developers.google.com/workspace/meet/api/guides/meeting-spaces#identify-meeting-space).
func (r *SpacesService) EndActiveConference(nameid string, endactiveconferencerequest *EndActiveConferenceRequest) *SpacesEndActiveConferenceCall {
	c := &SpacesEndActiveConferenceCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.nameid = nameid
	c.endactiveconferencerequest = endactiveconferencerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesEndActiveConferenceCall) Fields(s ...googleapi.Field) *SpacesEndActiveConferenceCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesEndActiveConferenceCall) Context(ctx context.Context) *SpacesEndActiveConferenceCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesEndActiveConferenceCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesEndActiveConferenceCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.endactiveconferencerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:endActiveConference")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.nameid,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.spaces.endActiveConference", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.spaces.endActiveConference" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesEndActiveConferenceCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.spaces.endActiveConference", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesGetCall struct {
	s            *Service
	nameid       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets details about a meeting space. For an example, see Get a meeting
// space
// (https://developers.google.com/workspace/meet/api/guides/meeting-spaces#get-meeting-space).
//
//   - name: Resource name of the space. Format: `spaces/{space}` or
//     `spaces/{meetingCode}`. `{space}` is the resource identifier for the
//     space. It's a unique, server-generated ID and is case sensitive. For
//     example, `jQCFfuBOdN5z`. `{meetingCode}` is an alias for the space. It's a
//     typeable, unique character string and is non-case sensitive. For example,
//     `abc-mnop-xyz`. The maximum length is 128 characters. A `meetingCode`
//     shouldn't be stored long term as it can become dissociated from a meeting
//     space and can be reused for different meeting spaces in the future.
//     Generally, a `meetingCode` expires 365 days after last use. For more
//     information, see Learn about meeting codes in Google Meet
//     (https://support.google.com/meet/answer/10710509). For more information,
//     see How Meet identifies a meeting space
//     (https://developers.google.com/workspace/meet/api/guides/meeting-spaces#identify-meeting-space).
func (r *SpacesService) Get(nameid string) *SpacesGetCall {
	c := &SpacesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.nameid = nameid
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesGetCall) Fields(s ...googleapi.Field) *SpacesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SpacesGetCall) IfNoneMatch(entityTag string) *SpacesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesGetCall) Context(ctx context.Context) *SpacesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.nameid,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.spaces.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.spaces.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesGetCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.spaces.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SpacesPatchCall struct {
	s          *Service
	nameid     string
	space      *Space
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates details about a meeting space. For an example, see Update a
// meeting space
// (https://developers.google.com/workspace/meet/api/guides/meeting-spaces#update-meeting-space).
//
//   - name: Immutable. Resource name of the space. Format: `spaces/{space}`.
//     `{space}` is the resource identifier for the space. It's a unique,
//     server-generated ID and is case sensitive. For example, `jQCFfuBOdN5z`.
//     For more information, see How Meet identifies a meeting space
//     (https://developers.google.com/workspace/meet/api/guides/meeting-spaces#identify-meeting-space).
func (r *SpacesService) Patch(nameid string, space *Space) *SpacesPatchCall {
	c := &SpacesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.nameid = nameid
	c.space = space
	return c
}

// UpdateMask sets the optional parameter "updateMask": Field mask used to
// specify the fields to be updated in the space. If update_mask isn't
// provided(not set, set with empty paths, or only has "" as paths), it
// defaults to update all fields provided with values in the request. Using "*"
// as update_mask will update all fields, including deleting fields not set in
// the request.
func (c *SpacesPatchCall) UpdateMask(updateMask string) *SpacesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SpacesPatchCall) Fields(s ...googleapi.Field) *SpacesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SpacesPatchCall) Context(ctx context.Context) *SpacesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SpacesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SpacesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.space)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.nameid,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "meet.spaces.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "meet.spaces.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Space.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SpacesPatchCall) Do(opts ...googleapi.CallOption) (*Space, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Space{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "meet.spaces.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
