// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package apkutil provides utilities for reading APK package records.
package apkutil

import (
	"bufio"
	"fmt"
	"io"
	"strings"
)

// KeyValue represents a key-value pair from an APK record.
type KeyValue struct {
	Key   string
	Value string
}

// Scanner provides a convenient way to read APK package records from an APK database.
// For details on the file format, see:
// https://wiki.alpinelinux.org/wiki/Apk_spec
type Scanner struct {
	scanner *bufio.Scanner
	err     error
	eof     bool
	cur     []KeyValue
}

// NewScanner returns a new [Scanner] to read from r.
func NewScanner(r io.Reader) *Scanner {
	return &Scanner{
		scanner: bufio.NewScanner(r),
	}
}

// Scan reads from the scanner a single APK record.
func (s *Scanner) Scan() bool {
	// return false if an error occurred or EOF has been reached
	if s.err != nil || s.eof {
		return false
	}

	s.cur = nil

	for s.scanner.Scan() {
		line := s.scanner.Text()

		if line != "" {
			key, val, found := strings.Cut(line, ":")

			if !found {
				s.err = fmt.Errorf("invalid line: %q", line)
				return false
			}

			s.cur = append(s.cur, KeyValue{Key: key, Value: val})
			continue
		}

		// line is empty, so we check if we have filled out data in group,
		// this avoids double empty lines returning early
		if len(s.cur) > 0 {
			return true
		}
	}

	s.err = s.scanner.Err()
	s.eof = s.err == nil

	// return true only if the EOF has been reached and record has something to return.
	// otherwise either an error occurred, or eof is reached with no more records, both require
	// no additional parsing of records.
	return s.eof && len(s.cur) > 0
}

// Record returns the most recent APK record generated by a call to [Scanner.Scan]
// For keys with multiple values, it returns the last one to maintain compatibility.
func (s *Scanner) Record() map[string]string {
	res := make(map[string]string)
	for _, kv := range s.cur {
		res[kv.Key] = kv.Value
	}
	return res
}

// RecordMultiValue returns the most recent APK record generated by a call to [Scanner.Scan]
// with all values for repeated keys.
func (s *Scanner) RecordMultiValue() map[string][]string {
	res := make(map[string][]string)
	for _, kv := range s.cur {
		res[kv.Key] = append(res[kv.Key], kv.Value)
	}
	return res
}

// FullRecord returns the full record with ordered key-value pairs.
func (s *Scanner) FullRecord() []KeyValue {
	return s.cur
}

// Err returns the first non-EOF error that was encountered by the [Scanner].
func (s *Scanner) Err() error {
	return s.err
}
