package gonotify

import "path/filepath"

// FileWatcher waits for events generated by filesystem for a specific list of file paths, including
// IN_CREATE for not yet existing files and IN_DELETE for removed.
type FileWatcher struct {
	stopC chan struct{}
	C     chan FileEvent
}

// NewFileWatcher creates FileWatcher with provided inotify mask and list of files to wait events for.
func NewFileWatcher(mask uint32, files ...string) (*FileWatcher, error) {

	f := &FileWatcher{
		stopC: make(chan struct{}),
		C:     make(chan FileEvent),
	}

	inotify, err := NewInotify()
	if err != nil {
		return nil, err
	}

	expectedPaths := make(map[string]bool)

	for _, file := range files {
		err := inotify.AddWatch(filepath.Dir(file), mask)
		if err != nil {
			inotify.Close()
			return nil, err
		}
		expectedPaths[file] = true
	}

	events := make(chan FileEvent)

	go func() {
		for {
			raw, err := inotify.Read()

			if err != nil {
				close(events)
				return
			}

			for _, event := range raw {
				events <- FileEvent{
					InotifyEvent: event,
				}

			}
		}
	}()

	go func() {
		for {
			select {
			case <-f.stopC:
				inotify.Close()
			case event, ok := <-events:

				if !ok {
					f.C <- FileEvent{
						Eof: true,
					}
					return
				}

				if !expectedPaths[event.Name] {
					continue
				}

				f.C <- event
			}
		}
	}()

	return f, nil
}

func (f *FileWatcher) Close() {
	f.stopC <- struct{}{}
}
