// Copyright (c) HashiCorp, Inc.
// SPDX-License-Identifier: MPL-2.0

package logging

import (
	"context"

	"github.com/hashicorp/go-hclog"
)

// GetProviderRootLogger returns the root logger used for writing logs
// from a provider. If no root logger has been created, it will return nil.
func GetProviderRootLogger(ctx context.Context) hclog.Logger {
	logger := ctx.Value(ProviderRootLoggerKey)
	if logger == nil {
		return nil
	}

	hclogger, ok := logger.(hclog.Logger)
	if !ok {
		return nil
	}
	return hclogger
}

// GetProviderRootLoggerOptions returns the root logger options used for
// creating the root provider logger. If the root logger has not been created
// or the options are not present, it will return nil.
func GetProviderRootLoggerOptions(ctx context.Context) *hclog.LoggerOptions {
	if GetProviderRootLogger(ctx) == nil {
		return nil
	}

	loggerOptionsRaw := ctx.Value(ProviderRootLoggerOptionsKey)

	if loggerOptionsRaw == nil {
		return nil
	}

	loggerOptions, ok := loggerOptionsRaw.(*hclog.LoggerOptions)

	if !ok {
		return nil
	}

	return loggerOptions
}

// SetProviderRootLogger sets `logger` as the root logger used for writing
// logs from a provider.
func SetProviderRootLogger(ctx context.Context, logger hclog.Logger) context.Context {
	return context.WithValue(ctx, ProviderRootLoggerKey, logger)
}

// SetProviderRootLoggerOptions sets `loggerOptions` as the root logger options
// used for creating the provider root logger.
func SetProviderRootLoggerOptions(ctx context.Context, loggerOptions *hclog.LoggerOptions) context.Context {
	return context.WithValue(ctx, ProviderRootLoggerOptionsKey, loggerOptions)
}

// NewProviderSubsystemLoggerWarning is the text included in log output when a
// subsystem is auto-generated by terraform-plugin-log because it was used
// before the provider instantiated it.
const NewProviderSubsystemLoggerWarning = "This log was generated by a subsystem logger that wasn't created before being used. Use tflog.NewSubsystem to create this logger before it is used."

// GetProviderSubsystemLogger returns the subsystem logger for the named
// subsystem in provider space. If no such subsystem logger has been created,
// it will return nil.
func GetProviderSubsystemLogger(ctx context.Context, subsystem string) hclog.Logger {
	logger := ctx.Value(providerSubsystemLoggerKey(subsystem))
	if logger == nil {
		return nil
	}

	hclogger, ok := logger.(hclog.Logger)
	if !ok {
		return nil
	}

	return hclogger
}

// SetProviderSubsystemLogger sets `logger` as the logger for the named
// subsystem in provider space.
func SetProviderSubsystemLogger(ctx context.Context, subsystem string, logger hclog.Logger) context.Context {
	return context.WithValue(ctx, providerSubsystemLoggerKey(subsystem), logger)
}

// GetProviderRootTFLoggerOpts retrieves the LoggerOpts of the provider root logger.
// The value is stored in the context.Context: if none is found, a new one will be created.
func GetProviderRootTFLoggerOpts(ctx context.Context) LoggerOpts {
	lOpts, ok := ctx.Value(providerRootTFLoggerOptsKey()).(LoggerOpts)
	if !ok {
		lOpts = LoggerOpts{}
	}

	return lOpts
}

// SetProviderRootTFLoggerOpts sets the LoggerOpts of the provider root logger, in the context.
func SetProviderRootTFLoggerOpts(ctx context.Context, lOpts LoggerOpts) context.Context {
	return context.WithValue(ctx, providerRootTFLoggerOptsKey(), lOpts)
}

// GetProviderSubsystemTFLoggerOpts retrieves the LoggerOpts of the logger for the named provider subsystem.
// The value is stored in the context.Context: if none is found, a new one will be created.
func GetProviderSubsystemTFLoggerOpts(ctx context.Context, subsystem string) LoggerOpts {
	lOpts, ok := ctx.Value(providerSubsystemTFLoggerOptsKey(subsystem)).(LoggerOpts)
	if !ok {
		lOpts = LoggerOpts{}
	}

	return lOpts
}

// SetProviderSubsystemTFLoggerOpts sets the LoggerOpts of the logger for the named provider subsystem, in the context.
func SetProviderSubsystemTFLoggerOpts(ctx context.Context, subsystem string, lOpts LoggerOpts) context.Context {
	return context.WithValue(ctx, providerSubsystemTFLoggerOptsKey(subsystem), lOpts)
}
