// Copyright © 2017 The Kubicorn Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package cmd

import (
	"errors"
	"fmt"
	"os"
	"strings"

	"io/ioutil"

	"github.com/kubicorn/kubicorn/pkg"
	"github.com/kubicorn/kubicorn/pkg/cli"
	"github.com/kubicorn/kubicorn/pkg/initapi"
	"github.com/kubicorn/kubicorn/pkg/kubeconfig"
	"github.com/kubicorn/kubicorn/pkg/local"
	"github.com/kubicorn/kubicorn/pkg/logger"
	"github.com/kubicorn/kubicorn/pkg/resourcedeploy"
	"github.com/kubicorn/kubicorn/pkg/state/crd"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
	"k8s.io/api/core/v1"
)

// ApplyCmd represents the apply command
func ApplyCmd() *cobra.Command {
	var ao = &cli.ApplyOptions{}
	var applyCmd = &cobra.Command{
		Use:   "apply <NAME>",
		Short: "Apply a cluster resource to a cloud",
		Long: `Use this command to apply an API model in a cloud.

	This command will attempt to find an API model in a defined state store, and then apply any changes needed directly to a cloud.
	The apply will run once, and ultimately time out if something goes wrong.`,
		Run: func(cmd *cobra.Command, args []string) {
			switch len(args) {
			case 0:
				ao.Name = viper.GetString(keyKubicornName)
			case 1:
				ao.Name = args[0]
			default:
				logger.Critical("Too many arguments.")
				os.Exit(1)
			}

			if err := runApply(ao); err != nil {
				logger.Critical(err.Error())
				os.Exit(1)
			}
		},
	}

	fs := applyCmd.Flags()

	bindCommonStateStoreFlags(&ao.StateStoreOptions, fs)
	bindCommonAwsFlags(&ao.AwsOptions, fs)

	fs.StringArrayVarP(&ao.Set, keyKubicornSet, "e", viper.GetStringSlice(keyKubicornSet), descSet)
	fs.StringVar(&ao.AwsProfile, keyAwsProfile, viper.GetString(keyAwsProfile), descAwsProfile)
	fs.StringVar(&ao.GitRemote, keyGitConfig, viper.GetString(keyGitConfig), descGitConfig)

	return applyCmd
}

func runApply(options *cli.ApplyOptions) error {

	// Ensure we have a name
	name := options.Name
	if name == "" {
		return errors.New("Empty name. Must specify the name of the cluster to apply")
	}

	// Expand state store path
	options.StateStorePath = cli.ExpandPath(options.StateStorePath)

	// Register state store
	stateStore, err := options.NewStateStore()
	if err != nil {
		return err
	}

	cluster, err := stateStore.GetCluster()
	if err != nil {
		return fmt.Errorf("Unable to get cluster [%s]: %v", name, err)
	}
	logger.Info("Loaded cluster: %s", cluster.Name)

	if len(options.Set) > 0 {
		for _, set := range options.Set {
			parts := strings.SplitN(set, "=", 2)
			if len(parts) == 1 {
				continue
			}
			err := cli.SwalkerWrite(strings.Title(parts[0]), cluster, parts[1])
			if err != nil {
				logger.Critical("Error expanding set flag: %#v", err)
			}
		}
	}

	// TODO(?) match create.go for options.MasterSet
	// TODO(?) match create.go for options.NodeSet

	logger.Info("Init Cluster")
	cluster, err = initapi.InitCluster(cluster)
	if err != nil {
		return err
	}

	runtimeParams := &pkg.RuntimeParameters{}

	if len(options.AwsProfile) > 0 {
		runtimeParams.AwsProfile = options.AwsProfile
	}

	reconciler, err := pkg.GetReconciler(cluster, runtimeParams)
	if err != nil {
		return fmt.Errorf("Unable to get reconciler: %v", err)
	}

	logger.Info("Query existing resources")
	actual, err := reconciler.Actual(cluster)
	if err != nil {
		return fmt.Errorf("Unable to get actual cluster: %v", err)
	}
	logger.Info("Resolving expected resources")
	expected, err := reconciler.Expected(cluster)
	if err != nil {
		return fmt.Errorf("Unable to get expected cluster: %v", err)
	}

	logger.Info("Reconciling")
	newCluster, err := reconciler.Reconcile(actual, expected)
	if err != nil {
		return fmt.Errorf("Unable to reconcile cluster: %v", err)
	}

	if err = stateStore.Commit(newCluster); err != nil {
		return fmt.Errorf("Unable to commit state store: %v", err)
	}

	logger.Info("Updating state store for cluster [%s]", options.Name)
	logger.Info("Hanging while fetching kube config...")
	if err = kubeconfig.RetryGetConfig(newCluster); err != nil {
		return fmt.Errorf("Unable to write kubeconfig: %v", err)
	}

	if newCluster.ControllerDeployment != nil {
		// -------------------------------------------------------------------------------------------------------------
		//
		// Here is where we hook in for the new controller logic
		// This is exclusive to profiles that have a controller defined
		//
		logger.Info("Deploying cluster controller: %s", newCluster.ControllerDeployment.Spec.Template.Spec.Containers[0].Image)

		// Add members to deployment now that we have them
		// TODO @kris-nova this is super hacky
		// TODO @kris-nova no seriously - this is SUPER hacky we need to fix this
		kubeConfigContent, err := ioutil.ReadFile(local.Expand("~/.kube/config"))
		if err != nil {
			return fmt.Errorf("Unable to parse kube config file: %v", err)
		}
		pc := newCluster.ProviderConfig()
		newCluster.ControllerDeployment.Spec.Template.Spec.Containers[0].Env = []v1.EnvVar{
			{
				Name:  "KUBECONFIG_CONTENT",
				Value: string(kubeConfigContent),
			},
			{
				Name:  "AWS_ACCESS_KEY_ID",
				Value: os.Getenv("AWS_ACCESS_KEY_ID"),
			},
			{
				Name:  "AWS_SECRET_ACCESS_KEY",
				Value: os.Getenv("AWS_SECRET_ACCESS_KEY"),
			},
			{
				Name:  "AWS_REGION",
				Value: pc.Location,
			},
			{
				Name:  "AWS_PROFILE",
				Value: "default",
			},
		}

		err = resourcedeploy.DeployClusterControllerDeployment(newCluster)
		if err != nil {
			return fmt.Errorf("Unable to deploy cluster controller: %v", err)
		}
		crdStateStore := crd.NewCRDStore(&crd.CRDStoreOptions{
			BasePath:    options.StateStorePath,
			ClusterName: options.Name,
		})
		crdStateStore.Commit(newCluster)
	}

	if newCluster.APITokenSecret != nil {
		// -------------------------------------------------------------------------------------------------------------
		//
		// Here is where we hook in for the new controller logic
		// This is exclusive to profiles that have a cloud-manager secret defined
		//
		logger.Info("Deploying cloud manager secret: %s", newCluster.APITokenSecret.ObjectMeta.Name)
		err = resourcedeploy.DeployCloudManagerSecret(newCluster)
		if err != nil {
			return fmt.Errorf("Unable to deploy cloud manager secret: %v", err)
		}
	}

	logger.Always("The [%s] cluster has applied successfully!", newCluster.Name)
	if path, ok := newCluster.Annotations[kubeconfig.ClusterAnnotationKubeconfigLocalFile]; ok {
		path = local.Expand(path)
		logger.Always("To start using your cluster, you need to run")
		logger.Always("  export KUBECONFIG=\"${KUBECONFIG}:%s\"", path)
	}
	logger.Always("You can now `kubectl get nodes`")
	privKeyPath := strings.Replace(cluster.ProviderConfig().SSH.PublicKeyPath, ".pub", "", 1)
	logger.Always("You can SSH into your cluster ssh -i %s %s@%s", privKeyPath, newCluster.ProviderConfig().SSH.User, newCluster.ProviderConfig().KubernetesAPI.Endpoint)

	return nil
}
