// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package doubleclickbidmanager provides access to the DoubleClick Bid Manager API.
//
// For product documentation, see: https://developers.google.com/bid-manager/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/doubleclickbidmanager/v2"
//	...
//	ctx := context.Background()
//	doubleclickbidmanagerService, err := doubleclickbidmanager.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	doubleclickbidmanagerService, err := doubleclickbidmanager.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	doubleclickbidmanagerService, err := doubleclickbidmanager.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package doubleclickbidmanager // import "google.golang.org/api/doubleclickbidmanager/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "doubleclickbidmanager:v2"
const apiName = "doubleclickbidmanager"
const apiVersion = "v2"
const basePath = "https://doubleclickbidmanager.googleapis.com/v2/"
const basePathTemplate = "https://doubleclickbidmanager.UNIVERSE_DOMAIN/v2/"
const mtlsBasePath = "https://doubleclickbidmanager.mtls.googleapis.com/v2/"

// OAuth2 scopes used by this API.
const (
	// View and manage your reports in DoubleClick Bid Manager
	DoubleclickbidmanagerScope = "https://www.googleapis.com/auth/doubleclickbidmanager"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/doubleclickbidmanager",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Queries = NewQueriesService(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.Background(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Queries *QueriesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewQueriesService(s *Service) *QueriesService {
	rs := &QueriesService{s: s}
	rs.Reports = NewQueriesReportsService(s)
	return rs
}

type QueriesService struct {
	s *Service

	Reports *QueriesReportsService
}

func NewQueriesReportsService(s *Service) *QueriesReportsService {
	rs := &QueriesReportsService{s: s}
	return rs
}

type QueriesReportsService struct {
	s *Service
}

// DataRange: The date range to be reported on.
type DataRange struct {
	// CustomEndDate: If `CUSTOM_DATES` is assigned to range, this field specifies
	// the end date for the date range that is reported on. This field is required
	// if using `CUSTOM_DATES` range and will be ignored otherwise.
	CustomEndDate *Date `json:"customEndDate,omitempty"`
	// CustomStartDate: If `CUSTOM_DATES` is assigned to range, this field
	// specifies the starting date for the date range that is reported on. This
	// field is required if using `CUSTOM_DATES` range and will be ignored
	// otherwise.
	CustomStartDate *Date `json:"customStartDate,omitempty"`
	// Range: The preset date range to be reported on. If `CUSTOM_DATES` is
	// assigned to this field, fields custom_start_date and custom_end_date must be
	// set to specify the custom date range.
	//
	// Possible values:
	//   "RANGE_UNSPECIFIED" - Default value when range is not specified or is
	// unknown in this version.
	//   "CUSTOM_DATES" - Custom date range.
	//   "CURRENT_DAY" - Current day.
	//   "PREVIOUS_DAY" - Previous day.
	//   "WEEK_TO_DATE" - All days, including the current day, since the most
	// recent Sunday.
	//   "MONTH_TO_DATE" - All days, including the current day, since the start of
	// the current month.
	//   "QUARTER_TO_DATE" - All days, including the current day, since the start
	// of the current quarter.
	//   "YEAR_TO_DATE" - All days, including the current day, since the start of
	// the current calendar year.
	//   "PREVIOUS_WEEK" - The previous completed week, beginning from Sunday.
	//   "PREVIOUS_MONTH" - The previous completed calendar month.
	//   "PREVIOUS_QUARTER" - The previous completed quarter.
	//   "PREVIOUS_YEAR" - The previous completed calendar year.
	//   "LAST_7_DAYS" - The previous 7 days, excluding the current day.
	//   "LAST_30_DAYS" - The previous 30 days, excluding the current day.
	//   "LAST_90_DAYS" - The previous 90 days, excluding the current day.
	//   "LAST_365_DAYS" - The previous 365 days, excluding the current day.
	//   "ALL_TIME" - All time for which data is available, excluding the current
	// day.
	//   "LAST_14_DAYS" - The previous 14 days, excluding the current day.
	//   "LAST_60_DAYS" - The previous 60 days, excluding the current day.
	Range string `json:"range,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomEndDate") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomEndDate") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DataRange) MarshalJSON() ([]byte, error) {
	type NoMethod DataRange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Date: Represents a whole or partial calendar date, such as a birthday. The
// time of day and time zone are either specified elsewhere or are
// insignificant. The date is relative to the Gregorian Calendar. This can
// represent one of the following: * A full date, with non-zero year, month,
// and day values. * A month and day, with a zero year (for example, an
// anniversary). * A year on its own, with a zero month and a zero day. * A
// year and month, with a zero day (for example, a credit card expiration
// date). Related types: * google.type.TimeOfDay * google.type.DateTime *
// google.protobuf.Timestamp
type Date struct {
	// Day: Day of a month. Must be from 1 to 31 and valid for the year and month,
	// or 0 to specify a year by itself or a year and month where the day isn't
	// significant.
	Day int64 `json:"day,omitempty"`
	// Month: Month of a year. Must be from 1 to 12, or 0 to specify a year without
	// a month and day.
	Month int64 `json:"month,omitempty"`
	// Year: Year of the date. Must be from 1 to 9999, or 0 to specify a date
	// without a year.
	Year int64 `json:"year,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Day") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Day") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Date) MarshalJSON() ([]byte, error) {
	type NoMethod Date
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FilterPair: Represents a single filter rule.
type FilterPair struct {
	// Type: The type of value to filter by. Defined by a Filter
	// (/bid-manager/reference/rest/v2/filters-metrics#filters) value.
	Type string `json:"type,omitempty"`
	// Value: The identifying value to filter by, such as a relevant resource ID.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Type") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Type") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FilterPair) MarshalJSON() ([]byte, error) {
	type NoMethod FilterPair
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type ListQueriesResponse struct {
	// NextPageToken: A token to retrieve the next page of results. Pass this value
	// in the page_token field in the subsequent call to `queries.list` method to
	// retrieve the next page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Queries: The list of queries. This field will be absent if empty.
	Queries []*Query `json:"queries,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListQueriesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListQueriesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type ListReportsResponse struct {
	// NextPageToken: A token to retrieve the next page of results. Pass this value
	// in the page_token field in the subsequent call to `queries.reports.list`
	// method to retrieve the next page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Reports: The list of reports. This field will be absent if empty.
	Reports []*Report `json:"reports,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListReportsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListReportsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Options: Report parameter options.
type Options struct {
	// IncludeOnlyTargetedUserLists: Whether to include data for audience lists
	// specifically targeted by filtered line items or insertion orders. Requires
	// the use of `FILTER_INSERTION_ORDER` or `FILTER_LINE_ITEM` filters.
	IncludeOnlyTargetedUserLists bool `json:"includeOnlyTargetedUserLists,omitempty"`
	// ForceSendFields is a list of field names (e.g.
	// "IncludeOnlyTargetedUserLists") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. See https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields
	// for more details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IncludeOnlyTargetedUserLists") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Options) MarshalJSON() ([]byte, error) {
	type NoMethod Options
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Parameters: Parameters of a generated report.
type Parameters struct {
	// Filters: Filters to limit the scope of reported data.
	Filters []*FilterPair `json:"filters,omitempty"`
	// GroupBys: Dimensions by which to segment and group the data. Defined by
	// Filter (/bid-manager/reference/rest/v2/filters-metrics#filters) values.
	GroupBys []string `json:"groupBys,omitempty"`
	// Metrics: Metrics to define the data populating the report. Defined by Metric
	// (/bid-manager/reference/rest/v2/filters-metrics#metrics) values.
	Metrics []string `json:"metrics,omitempty"`
	// Options: Additional report parameter options.
	Options *Options `json:"options,omitempty"`
	// Type: The type of the report. The type of the report determines the
	// dimesions, filters, and metrics that can be used.
	//
	// Possible values:
	//   "REPORT_TYPE_UNSPECIFIED" - Default value when report type is not
	// specified or is unknown in this version.
	//   "STANDARD" - Standard report.
	//   "INVENTORY_AVAILABILITY" - Inventory Availability report.
	//   "AUDIENCE_COMPOSITION" - Audience Composition report.
	//   "FLOODLIGHT" - Floodlight report.
	//   "YOUTUBE" - YouTube report.
	//   "GRP" - GRP report.
	//   "YOUTUBE_PROGRAMMATIC_GUARANTEED" - YouTube Programmatic Guaranteed
	// report.
	//   "REACH" - Reach report.
	//   "UNIQUE_REACH_AUDIENCE" - Unique Reach Audience report.
	//   "FULL_PATH" - Full Path report.
	//   "PATH_ATTRIBUTION" - Path Attribution report.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Filters") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Filters") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Parameters) MarshalJSON() ([]byte, error) {
	type NoMethod Parameters
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Query: A single query used to generate a report.
type Query struct {
	// Metadata: The metadata of the query.
	Metadata *QueryMetadata `json:"metadata,omitempty"`
	// Params: The parameters of the report generated by the query.
	Params *Parameters `json:"params,omitempty"`
	// QueryId: Output only. The unique ID of the query.
	QueryId int64 `json:"queryId,omitempty,string"`
	// Schedule: When and how often the query is scheduled to run. If the frequency
	// field is set to `ONE_TIME`, the query will only run when queries.run is
	// called.
	Schedule *QuerySchedule `json:"schedule,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Metadata") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Metadata") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Query) MarshalJSON() ([]byte, error) {
	type NoMethod Query
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QueryMetadata: The metadata of the query.
type QueryMetadata struct {
	// DataRange: The date range the report generated by the query will report on.
	// This date range will be defined by the time zone as used by the advertiser.
	DataRange *DataRange `json:"dataRange,omitempty"`
	// Format: The format of the report generated by the query.
	//
	// Possible values:
	//   "FORMAT_UNSPECIFIED" - Default value when format is not specified or is
	// unknown in this version.
	//   "CSV" - CSV.
	//   "XLSX" - Excel.
	Format string `json:"format,omitempty"`
	// SendNotification: Whether an email notification is sent to the query creator
	// when a report generated by the query is ready. This value is `false` by
	// default.
	SendNotification bool `json:"sendNotification,omitempty"`
	// ShareEmailAddress: List of additional email addresses with which to share
	// the query. If send_notification is `true`, these email addresses will
	// receive a notification when a report generated by the query is ready. If
	// these email addresses are connected to Display & Video 360 users, the query
	// will be available to them in the Display & Video 360 interface.
	ShareEmailAddress []string `json:"shareEmailAddress,omitempty"`
	// Title: The display name of the query. This value will be used in the file
	// name of reports generated by the query.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DataRange") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DataRange") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QueryMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod QueryMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// QuerySchedule: Settings on when and how frequently to run a query.
type QuerySchedule struct {
	// EndDate: The date on which to end the scheduled runs. This field is required
	// if frequency is not set to `ONE_TIME`. Otherwise, it will be ignored.
	EndDate *Date `json:"endDate,omitempty"`
	// Frequency: How frequently to run the query. If set to `ONE_TIME`, the query
	// will only be run when queries.run is called.
	//
	// Possible values:
	//   "FREQUENCY_UNSPECIFIED" - Default value when frequency is not specified or
	// is unknown in this version.
	//   "ONE_TIME" - Only when the query is run manually.
	//   "DAILY" - Once a day.
	//   "WEEKLY" - Once a week.
	//   "SEMI_MONTHLY" - Twice a month.
	//   "MONTHLY" - Once a month.
	//   "QUARTERLY" - Once a quarter.
	//   "YEARLY" - Once a year.
	Frequency string `json:"frequency,omitempty"`
	// NextRunTimezoneCode: The canonical code for the timezone the query schedule
	// is based on. Scheduled runs are usually conducted in the morning of a given
	// day. Defaults to `America/New_York`.
	NextRunTimezoneCode string `json:"nextRunTimezoneCode,omitempty"`
	// StartDate: The date on which to begin the scheduled runs. This field is
	// required if frequency is not set to `ONE_TIME`. Otherwise, it will be
	// ignored.
	StartDate *Date `json:"startDate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EndDate") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndDate") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s QuerySchedule) MarshalJSON() ([]byte, error) {
	type NoMethod QuerySchedule
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Report: A single report generated by its parent report.
type Report struct {
	// Key: The key information identifying the report.
	Key *ReportKey `json:"key,omitempty"`
	// Metadata: The metadata of the report.
	Metadata *ReportMetadata `json:"metadata,omitempty"`
	// Params: The parameters of the report.
	Params *Parameters `json:"params,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Key") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Key") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Report) MarshalJSON() ([]byte, error) {
	type NoMethod Report
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReportKey: Identifying information of a report.
type ReportKey struct {
	// QueryId: Output only. The unique ID of the query that generated the report.
	QueryId int64 `json:"queryId,omitempty,string"`
	// ReportId: Output only. The unique ID of the report.
	ReportId int64 `json:"reportId,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "QueryId") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "QueryId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReportKey) MarshalJSON() ([]byte, error) {
	type NoMethod ReportKey
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReportMetadata: The metadata of a report.
type ReportMetadata struct {
	// GoogleCloudStoragePath: Output only. The location of the generated report
	// file in Google Cloud Storage. This field will be absent if status.state is
	// not `DONE`.
	GoogleCloudStoragePath string `json:"googleCloudStoragePath,omitempty"`
	// ReportDataEndDate: The end date of the report data date range.
	ReportDataEndDate *Date `json:"reportDataEndDate,omitempty"`
	// ReportDataStartDate: The start date of the report data date range.
	ReportDataStartDate *Date `json:"reportDataStartDate,omitempty"`
	// Status: The status of the report.
	Status *ReportStatus `json:"status,omitempty"`
	// ForceSendFields is a list of field names (e.g. "GoogleCloudStoragePath") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GoogleCloudStoragePath") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReportMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod ReportMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReportStatus: The status of a report.
type ReportStatus struct {
	// FinishTime: Output only. The timestamp of when report generation finished
	// successfully or in failure. This field will not be set unless state is
	// `DONE` or `FAILED`.
	FinishTime string `json:"finishTime,omitempty"`
	// Format: The format of the generated report file.
	//
	// Possible values:
	//   "FORMAT_UNSPECIFIED" - Default value when format is not specified or is
	// unknown in this version.
	//   "CSV" - CSV.
	//   "XLSX" - Excel.
	Format string `json:"format,omitempty"`
	// State: Output only. The state of the report generation.
	//
	// Possible values:
	//   "STATE_UNSPECIFIED" - Default value when state is not specified or is
	// unknown in this version.
	//   "QUEUED" - The report is queued to run.
	//   "RUNNING" - The report is currently running.
	//   "DONE" - The report has finished running successfully.
	//   "FAILED" - The report has finished running in failure.
	State string `json:"state,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FinishTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FinishTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReportStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ReportStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RunQueryRequest: Details specifying how to run a query.
type RunQueryRequest struct {
	// DataRange: The date range used by the query to generate the report. If
	// unspecified, the query's original data_range is used.
	DataRange *DataRange `json:"dataRange,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DataRange") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DataRange") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RunQueryRequest) MarshalJSON() ([]byte, error) {
	type NoMethod RunQueryRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type QueriesCreateCall struct {
	s          *Service
	query      *Query
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a new query.
func (r *QueriesService) Create(query *Query) *QueriesCreateCall {
	c := &QueriesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.query = query
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *QueriesCreateCall) Fields(s ...googleapi.Field) *QueriesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *QueriesCreateCall) Context(ctx context.Context) *QueriesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *QueriesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *QueriesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.query)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "queries")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "doubleclickbidmanager.queries.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Query.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *QueriesCreateCall) Do(opts ...googleapi.CallOption) (*Query, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Query{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type QueriesDeleteCall struct {
	s          *Service
	queryId    int64
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an existing query as well as its generated reports.
//
// - queryId: The ID of the query to delete.
func (r *QueriesService) Delete(queryId int64) *QueriesDeleteCall {
	c := &QueriesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.queryId = queryId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *QueriesDeleteCall) Fields(s ...googleapi.Field) *QueriesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *QueriesDeleteCall) Context(ctx context.Context) *QueriesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *QueriesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *QueriesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "queries/{queryId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"queryId": strconv.FormatInt(c.queryId, 10),
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "doubleclickbidmanager.queries.delete" call.
func (c *QueriesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type QueriesGetCall struct {
	s            *Service
	queryId      int64
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves a query.
//
// - queryId: The ID of the query to retrieve.
func (r *QueriesService) Get(queryId int64) *QueriesGetCall {
	c := &QueriesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.queryId = queryId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *QueriesGetCall) Fields(s ...googleapi.Field) *QueriesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *QueriesGetCall) IfNoneMatch(entityTag string) *QueriesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *QueriesGetCall) Context(ctx context.Context) *QueriesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *QueriesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *QueriesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "queries/{queryId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"queryId": strconv.FormatInt(c.queryId, 10),
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "doubleclickbidmanager.queries.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Query.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *QueriesGetCall) Do(opts ...googleapi.CallOption) (*Query, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Query{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type QueriesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists queries created by the current user.
func (r *QueriesService) List() *QueriesListCall {
	c := &QueriesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// OrderBy sets the optional parameter "orderBy": Field to sort the list by.
// Accepts the following values: * `queryId` (default) * `metadata.title` The
// default sorting order is ascending. To specify descending order for a field,
// add the suffix `desc` to the field name. For example, `queryId desc`.
func (c *QueriesListCall) OrderBy(orderBy string) *QueriesListCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of results
// per page. Must be between `1` and `100`. Defaults to `100` if unspecified.
func (c *QueriesListCall) PageSize(pageSize int64) *QueriesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A token identifying which
// page of results the server should return. Typically, this is the value of
// nextPageToken, returned from the previous call to the `queries.list` method.
// If unspecified, the first page of results is returned.
func (c *QueriesListCall) PageToken(pageToken string) *QueriesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *QueriesListCall) Fields(s ...googleapi.Field) *QueriesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *QueriesListCall) IfNoneMatch(entityTag string) *QueriesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *QueriesListCall) Context(ctx context.Context) *QueriesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *QueriesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *QueriesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "queries")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "doubleclickbidmanager.queries.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListQueriesResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *QueriesListCall) Do(opts ...googleapi.CallOption) (*ListQueriesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListQueriesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *QueriesListCall) Pages(ctx context.Context, f func(*ListQueriesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type QueriesRunCall struct {
	s               *Service
	queryId         int64
	runqueryrequest *RunQueryRequest
	urlParams_      gensupport.URLParams
	ctx_            context.Context
	header_         http.Header
}

// Run: Runs an existing query to generate a report.
//
// - queryId: The ID of the query to run.
func (r *QueriesService) Run(queryId int64, runqueryrequest *RunQueryRequest) *QueriesRunCall {
	c := &QueriesRunCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.queryId = queryId
	c.runqueryrequest = runqueryrequest
	return c
}

// Synchronous sets the optional parameter "synchronous": Whether the query
// should be run synchronously. When `true`, the request won't return until the
// resulting report has finished running. This parameter is `false` by default.
// Setting this parameter to `true` is **not recommended**.
func (c *QueriesRunCall) Synchronous(synchronous bool) *QueriesRunCall {
	c.urlParams_.Set("synchronous", fmt.Sprint(synchronous))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *QueriesRunCall) Fields(s ...googleapi.Field) *QueriesRunCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *QueriesRunCall) Context(ctx context.Context) *QueriesRunCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *QueriesRunCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *QueriesRunCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.runqueryrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "queries/{queryId}:run")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"queryId": strconv.FormatInt(c.queryId, 10),
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.run", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "doubleclickbidmanager.queries.run" call.
// Any non-2xx status code is an error. Response headers are in either
// *Report.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *QueriesRunCall) Do(opts ...googleapi.CallOption) (*Report, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Report{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.run", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type QueriesReportsGetCall struct {
	s            *Service
	queryId      int64
	reportId     int64
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves a report.
//
// - queryId: The ID of the query that generated the report.
// - reportId: The ID of the query to retrieve.
func (r *QueriesReportsService) Get(queryId int64, reportId int64) *QueriesReportsGetCall {
	c := &QueriesReportsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.queryId = queryId
	c.reportId = reportId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *QueriesReportsGetCall) Fields(s ...googleapi.Field) *QueriesReportsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *QueriesReportsGetCall) IfNoneMatch(entityTag string) *QueriesReportsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *QueriesReportsGetCall) Context(ctx context.Context) *QueriesReportsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *QueriesReportsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *QueriesReportsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "queries/{queryId}/reports/{reportId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"queryId":  strconv.FormatInt(c.queryId, 10),
		"reportId": strconv.FormatInt(c.reportId, 10),
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.reports.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "doubleclickbidmanager.queries.reports.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Report.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *QueriesReportsGetCall) Do(opts ...googleapi.CallOption) (*Report, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Report{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.reports.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type QueriesReportsListCall struct {
	s            *Service
	queryId      int64
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists reports generated by the provided query.
//
// - queryId: The ID of the query that generated the reports.
func (r *QueriesReportsService) List(queryId int64) *QueriesReportsListCall {
	c := &QueriesReportsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.queryId = queryId
	return c
}

// OrderBy sets the optional parameter "orderBy": Field to sort the list by.
// Accepts the following values: * `key.reportId` (default) The default sorting
// order is ascending. To specify descending order for a field, add the suffix
// `desc` to the field name. For example, `key.reportId desc`.
func (c *QueriesReportsListCall) OrderBy(orderBy string) *QueriesReportsListCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageSize sets the optional parameter "pageSize": Maximum number of results
// per page. Must be between `1` and `100`. Defaults to `100` if unspecified.
func (c *QueriesReportsListCall) PageSize(pageSize int64) *QueriesReportsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A token identifying which
// page of results the server should return. Typically, this is the value of
// nextPageToken returned from the previous call to the `queries.reports.list`
// method. If unspecified, the first page of results is returned.
func (c *QueriesReportsListCall) PageToken(pageToken string) *QueriesReportsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *QueriesReportsListCall) Fields(s ...googleapi.Field) *QueriesReportsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *QueriesReportsListCall) IfNoneMatch(entityTag string) *QueriesReportsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *QueriesReportsListCall) Context(ctx context.Context) *QueriesReportsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *QueriesReportsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *QueriesReportsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "queries/{queryId}/reports")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"queryId": strconv.FormatInt(c.queryId, 10),
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.reports.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "doubleclickbidmanager.queries.reports.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListReportsResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *QueriesReportsListCall) Do(opts ...googleapi.CallOption) (*ListReportsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListReportsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "doubleclickbidmanager.queries.reports.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *QueriesReportsListCall) Pages(ctx context.Context, f func(*ListReportsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
