// Copyright 2026 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package chromewebstore provides access to the Chrome Web Store API.
//
// For product documentation, see: https://developer.chrome.com/docs/webstore/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/chromewebstore/v2"
//	...
//	ctx := context.Background()
//	chromewebstoreService, err := chromewebstore.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	chromewebstoreService, err := chromewebstore.NewService(ctx, option.WithScopes(chromewebstore.ChromewebstoreReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	chromewebstoreService, err := chromewebstore.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	chromewebstoreService, err := chromewebstore.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package chromewebstore // import "google.golang.org/api/chromewebstore/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "chromewebstore:v2"
const apiName = "chromewebstore"
const apiVersion = "v2"
const basePath = "https://chromewebstore.googleapis.com/"
const basePathTemplate = "https://chromewebstore.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://chromewebstore.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, update, or publish your Chrome Web Store extensions, themes,
	// apps, and licences you have access to
	ChromewebstoreScope = "https://www.googleapis.com/auth/chromewebstore"

	// See and download your Chrome Web Store extensions and apps, and see licenses
	// you have access to
	ChromewebstoreReadonlyScope = "https://www.googleapis.com/auth/chromewebstore.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/chromewebstore",
		"https://www.googleapis.com/auth/chromewebstore.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Media = NewMediaService(s)
	s.Publishers = NewPublishersService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Media *MediaService

	Publishers *PublishersService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewMediaService(s *Service) *MediaService {
	rs := &MediaService{s: s}
	return rs
}

type MediaService struct {
	s *Service
}

func NewPublishersService(s *Service) *PublishersService {
	rs := &PublishersService{s: s}
	rs.Items = NewPublishersItemsService(s)
	return rs
}

type PublishersService struct {
	s *Service

	Items *PublishersItemsService
}

func NewPublishersItemsService(s *Service) *PublishersItemsService {
	rs := &PublishersItemsService{s: s}
	return rs
}

type PublishersItemsService struct {
	s *Service
}

// CancelSubmissionRequest: Request message for CancelSubmission.
type CancelSubmissionRequest struct {
}

// CancelSubmissionResponse: Response message for `CancelSubmission`.
type CancelSubmissionResponse struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// DeployInfo: Deployment information for a specific release channel. Used in
// requests to update deployment parameters.
type DeployInfo struct {
	// DeployPercentage: Required. The current deploy percentage for the release
	// channel (nonnegative number between 0 and 100).
	DeployPercentage int64 `json:"deployPercentage,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DeployPercentage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DeployPercentage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DeployInfo) MarshalJSON() ([]byte, error) {
	type NoMethod DeployInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DistributionChannel: Deployment information for a specific release channel
type DistributionChannel struct {
	// CrxVersion: The extension version provided in the manifest of the uploaded
	// package.
	CrxVersion string `json:"crxVersion,omitempty"`
	// DeployPercentage: The current deploy percentage for the release channel
	// (nonnegative number between 0 and 100).
	DeployPercentage int64 `json:"deployPercentage,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CrxVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CrxVersion") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DistributionChannel) MarshalJSON() ([]byte, error) {
	type NoMethod DistributionChannel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FetchItemStatusResponse: Response message for `FetchItemStatus`.
type FetchItemStatusResponse struct {
	// ItemId: Output only. The ID of the item.
	ItemId string `json:"itemId,omitempty"`
	// LastAsyncUploadState: Output only. The state of the last async upload for an
	// item. Only set when there has been an async upload for the item in the past
	// 24 hours.
	//
	// Possible values:
	//   "UPLOAD_STATE_UNSPECIFIED" - The default value.
	//   "SUCCEEDED" - The upload succeeded.
	//   "IN_PROGRESS" - The upload is currently being processed.
	//   "FAILED" - The upload failed.
	//   "NOT_FOUND" - Used as the value of `lastAsyncUploadState` in a
	// `fetchStatus` response indicating that an upload attempt was not found.
	LastAsyncUploadState string `json:"lastAsyncUploadState,omitempty"`
	// Name: The name of the requested item.
	Name string `json:"name,omitempty"`
	// PublicKey: The public key of the item, which may be generated by the store.
	PublicKey string `json:"publicKey,omitempty"`
	// PublishedItemRevisionStatus: Output only. Status of the current published
	// revision of the item. Will be unset if the item is not published.
	PublishedItemRevisionStatus *ItemRevisionStatus `json:"publishedItemRevisionStatus,omitempty"`
	// SubmittedItemRevisionStatus: Status of the item revision submitted to be
	// published. Will be unset if the item has not been submitted for publishing
	// since the last successful publish.
	SubmittedItemRevisionStatus *ItemRevisionStatus `json:"submittedItemRevisionStatus,omitempty"`
	// TakenDown: If true, the item has been taken down for a policy violation.
	// Check the developer dashboard for details.
	TakenDown bool `json:"takenDown,omitempty"`
	// Warned: If true, the item has been warned for a policy violation and will be
	// taken down if not resolved. Check the developer dashboard for details.
	Warned bool `json:"warned,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ItemId") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ItemId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FetchItemStatusResponse) MarshalJSON() ([]byte, error) {
	type NoMethod FetchItemStatusResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ItemRevisionStatus: Details on the status of an item revision.
type ItemRevisionStatus struct {
	// DistributionChannels: Details on the package of the item
	DistributionChannels []*DistributionChannel `json:"distributionChannels,omitempty"`
	// State: Output only. Current state of the item
	//
	// Possible values:
	//   "ITEM_STATE_UNSPECIFIED" - Default value. This value is unused.
	//   "PENDING_REVIEW" - The item is pending review.
	//   "STAGED" - The item has been approved and is ready to be published.
	//   "PUBLISHED" - The item is published publicly.
	//   "PUBLISHED_TO_TESTERS" - The item is published to trusted testers.
	//   "REJECTED" - The item has been rejected for publishing.
	//   "CANCELLED" - The item submission has been cancelled.
	State string `json:"state,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DistributionChannels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DistributionChannels") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ItemRevisionStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ItemRevisionStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PublishItemRequest: Request message for PublishItem.
type PublishItemRequest struct {
	// DeployInfos: Optional. Additional deploy information including the desired
	// initial percentage rollout. Defaults to the current value saved in the
	// developer dashboard if unset.
	DeployInfos []*DeployInfo `json:"deployInfos,omitempty"`
	// PublishType: Optional. Use this to control if the item is published
	// immediately on approval or staged for publishing in the future. Defaults to
	// `DEFAULT_PUBLISH` if unset.
	//
	// Possible values:
	//   "PUBLISH_TYPE_UNSPECIFIED" - Default value. This is the same as
	// DEFAULT_PUBLISH.
	//   "DEFAULT_PUBLISH" - The submission will be published immediately after
	// being approved.
	//   "STAGED_PUBLISH" - After approval the submission will be staged and can
	// then be published by the developer.
	PublishType string `json:"publishType,omitempty"`
	// SkipReview: Optional. Whether to attempt to skip item review. The API will
	// validate if the item qualifies and return a validation error if the item
	// requires review. Defaults to `false` if unset.
	SkipReview bool `json:"skipReview,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DeployInfos") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DeployInfos") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PublishItemRequest) MarshalJSON() ([]byte, error) {
	type NoMethod PublishItemRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PublishItemResponse: Response message for `PublishItem`.
type PublishItemResponse struct {
	// ItemId: Output only. The ID of the item.
	ItemId string `json:"itemId,omitempty"`
	// Name: The name of the item that was submitted
	Name string `json:"name,omitempty"`
	// State: Output only. The current state of the submission.
	//
	// Possible values:
	//   "ITEM_STATE_UNSPECIFIED" - Default value. This value is unused.
	//   "PENDING_REVIEW" - The item is pending review.
	//   "STAGED" - The item has been approved and is ready to be published.
	//   "PUBLISHED" - The item is published publicly.
	//   "PUBLISHED_TO_TESTERS" - The item is published to trusted testers.
	//   "REJECTED" - The item has been rejected for publishing.
	//   "CANCELLED" - The item submission has been cancelled.
	State string `json:"state,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ItemId") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ItemId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PublishItemResponse) MarshalJSON() ([]byte, error) {
	type NoMethod PublishItemResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SetPublishedDeployPercentageRequest: Request message for
// SetPublishedDeployPercentage.
type SetPublishedDeployPercentageRequest struct {
	// DeployPercentage: Required. Unscaled percentage value for the publised
	// revision (nonnegative number between 0 and 100). It must be larger than the
	// existing target percentage.
	DeployPercentage int64 `json:"deployPercentage,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DeployPercentage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DeployPercentage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SetPublishedDeployPercentageRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SetPublishedDeployPercentageRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SetPublishedDeployPercentageResponse: Response message for
// `SetPublishedDeployPercentage`.
type SetPublishedDeployPercentageResponse struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// UploadItemPackageRequest: Request message for UploadItemPackage.
type UploadItemPackageRequest struct {
}

// UploadItemPackageResponse: Response message for `UploadItemPackage`.
type UploadItemPackageResponse struct {
	// CrxVersion: The extension version provided in the manifest of the uploaded
	// package. This will not be set if the upload is still in progress
	// (`upload_state` is `UPLOAD_IN_PROGRESS`).
	CrxVersion string `json:"crxVersion,omitempty"`
	// ItemId: Output only. The ID of the item the package was uploaded to.
	ItemId string `json:"itemId,omitempty"`
	// Name: The name of the item the package was uploaded to.
	Name string `json:"name,omitempty"`
	// UploadState: Output only. The state of the upload. If `upload_state` is
	// `UPLOAD_IN_PROGRESS`, you can poll for updates using the fetchStatus method.
	//
	// Possible values:
	//   "UPLOAD_STATE_UNSPECIFIED" - The default value.
	//   "SUCCEEDED" - The upload succeeded.
	//   "IN_PROGRESS" - The upload is currently being processed.
	//   "FAILED" - The upload failed.
	//   "NOT_FOUND" - Used as the value of `lastAsyncUploadState` in a
	// `fetchStatus` response indicating that an upload attempt was not found.
	UploadState string `json:"uploadState,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CrxVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CrxVersion") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UploadItemPackageResponse) MarshalJSON() ([]byte, error) {
	type NoMethod UploadItemPackageResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type MediaUploadCall struct {
	s                        *Service
	name                     string
	uploaditempackagerequest *UploadItemPackageRequest
	urlParams_               gensupport.URLParams
	mediaInfo_               *gensupport.MediaInfo
	ctx_                     context.Context
	header_                  http.Header
}

// Upload: Upload a new package to an existing item.
//
//   - name: Name of the item to upload the new package to in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *MediaService) Upload(name string, uploaditempackagerequest *UploadItemPackageRequest) *MediaUploadCall {
	c := &MediaUploadCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.uploaditempackagerequest = uploaditempackagerequest
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk size
// may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the upload
// request will be determined by sniffing the contents of r, unless a
// MediaOption generated by googleapi.ContentType is supplied.
// At most one of Media and ResumableMedia may be set.
func (c *MediaUploadCall) Media(r io.Reader, options ...googleapi.MediaOption) *MediaUploadCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be canceled
// with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType identifies the
// MIME media type of the upload, such as "image/png". If mediaType is "", it
// will be auto-detected. The provided ctx will supersede any context
// previously provided to the Context method.
func (c *MediaUploadCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *MediaUploadCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called after every
// chunk. It should be a low-latency function in order to not slow down the
// upload operation. This should only be called when using ResumableMedia (as
// opposed to Media).
func (c *MediaUploadCall) ProgressUpdater(pu googleapi.ProgressUpdater) *MediaUploadCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *MediaUploadCall) Fields(s ...googleapi.Field) *MediaUploadCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *MediaUploadCall) Context(ctx context.Context) *MediaUploadCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *MediaUploadCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *MediaUploadCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.uploaditempackagerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:upload")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/v2/{+name}:upload")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	newBody, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, newBody)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.media.upload", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.media.upload" call.
// Any non-2xx status code is an error. Response headers are in either
// *UploadItemPackageResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *MediaUploadCall) Do(opts ...googleapi.CallOption) (*UploadItemPackageResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &UploadItemPackageResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.media.upload", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsCancelSubmissionCall struct {
	s                       *Service
	name                    string
	cancelsubmissionrequest *CancelSubmissionRequest
	urlParams_              gensupport.URLParams
	ctx_                    context.Context
	header_                 http.Header
}

// CancelSubmission: Cancel the current active submission of an item if
// present. This can be used to cancel the review of a pending submission.
//
//   - name: Name of the item to cancel the submission of in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) CancelSubmission(name string, cancelsubmissionrequest *CancelSubmissionRequest) *PublishersItemsCancelSubmissionCall {
	c := &PublishersItemsCancelSubmissionCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.cancelsubmissionrequest = cancelsubmissionrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsCancelSubmissionCall) Fields(s ...googleapi.Field) *PublishersItemsCancelSubmissionCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsCancelSubmissionCall) Context(ctx context.Context) *PublishersItemsCancelSubmissionCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsCancelSubmissionCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsCancelSubmissionCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.cancelsubmissionrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:cancelSubmission")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.cancelSubmission", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.cancelSubmission" call.
// Any non-2xx status code is an error. Response headers are in either
// *CancelSubmissionResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *PublishersItemsCancelSubmissionCall) Do(opts ...googleapi.CallOption) (*CancelSubmissionResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CancelSubmissionResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.cancelSubmission", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsFetchStatusCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// FetchStatus: Fetch the status of an item.
//
//   - name: Name of the item to retrieve the status of in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) FetchStatus(name string) *PublishersItemsFetchStatusCall {
	c := &PublishersItemsFetchStatusCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsFetchStatusCall) Fields(s ...googleapi.Field) *PublishersItemsFetchStatusCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PublishersItemsFetchStatusCall) IfNoneMatch(entityTag string) *PublishersItemsFetchStatusCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsFetchStatusCall) Context(ctx context.Context) *PublishersItemsFetchStatusCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsFetchStatusCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsFetchStatusCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:fetchStatus")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.fetchStatus", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.fetchStatus" call.
// Any non-2xx status code is an error. Response headers are in either
// *FetchItemStatusResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *PublishersItemsFetchStatusCall) Do(opts ...googleapi.CallOption) (*FetchItemStatusResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &FetchItemStatusResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.fetchStatus", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsPublishCall struct {
	s                  *Service
	name               string
	publishitemrequest *PublishItemRequest
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Publish: Submit the item to be published in the store. The item will be
// submitted for review unless `skip_review` is set to true, or the item is
// staged from a previous submission with `publish_type` set to
// `STAGED_PUBLISH`.
//
//   - name: Name of the item in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) Publish(name string, publishitemrequest *PublishItemRequest) *PublishersItemsPublishCall {
	c := &PublishersItemsPublishCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.publishitemrequest = publishitemrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsPublishCall) Fields(s ...googleapi.Field) *PublishersItemsPublishCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsPublishCall) Context(ctx context.Context) *PublishersItemsPublishCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsPublishCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsPublishCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.publishitemrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:publish")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.publish", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.publish" call.
// Any non-2xx status code is an error. Response headers are in either
// *PublishItemResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *PublishersItemsPublishCall) Do(opts ...googleapi.CallOption) (*PublishItemResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PublishItemResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.publish", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PublishersItemsSetPublishedDeployPercentageCall struct {
	s                                   *Service
	name                                string
	setpublisheddeploypercentagerequest *SetPublishedDeployPercentageRequest
	urlParams_                          gensupport.URLParams
	ctx_                                context.Context
	header_                             http.Header
}

// SetPublishedDeployPercentage: Set a higher target deploy percentage for the
// item's published revision. This will be updated without the item being
// submitted for review. This is only available to items with over 10,000
// seven-day active users.
//
//   - name: Name of the item to update the published revision of in the form
//     `publishers/{publisherId}/items/{itemId}`.
func (r *PublishersItemsService) SetPublishedDeployPercentage(name string, setpublisheddeploypercentagerequest *SetPublishedDeployPercentageRequest) *PublishersItemsSetPublishedDeployPercentageCall {
	c := &PublishersItemsSetPublishedDeployPercentageCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.setpublisheddeploypercentagerequest = setpublisheddeploypercentagerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Fields(s ...googleapi.Field) *PublishersItemsSetPublishedDeployPercentageCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Context(ctx context.Context) *PublishersItemsSetPublishedDeployPercentageCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PublishersItemsSetPublishedDeployPercentageCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.setpublisheddeploypercentagerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v2/{+name}:setPublishedDeployPercentage")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.setPublishedDeployPercentage", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "chromewebstore.publishers.items.setPublishedDeployPercentage" call.
// Any non-2xx status code is an error. Response headers are in either
// *SetPublishedDeployPercentageResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *PublishersItemsSetPublishedDeployPercentageCall) Do(opts ...googleapi.CallOption) (*SetPublishedDeployPercentageResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SetPublishedDeployPercentageResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "chromewebstore.publishers.items.setPublishedDeployPercentage", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
