// Copyright 2026 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package datamanager provides access to the Data Manager API.
//
// For product documentation, see: https://developers.google.com/data-manager
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/datamanager/v1"
//	...
//	ctx := context.Background()
//	datamanagerService, err := datamanager.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	datamanagerService, err := datamanager.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	datamanagerService, err := datamanager.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package datamanager // import "google.golang.org/api/datamanager/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "datamanager:v1"
const apiName = "datamanager"
const apiVersion = "v1"
const basePath = "https://datamanager.googleapis.com/"
const basePathTemplate = "https://datamanager.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://datamanager.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, create, import, or delete your customer data in Google Ads,
	// Google Marketing Platform (Campaign Manager 360, Search Ads 360, Display &
	// Video 360), and Google Analytics
	DatamanagerScope = "https://www.googleapis.com/auth/datamanager"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/datamanager",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.AudienceMembers = NewAudienceMembersService(s)
	s.Events = NewEventsService(s)
	s.RequestStatus = NewRequestStatusService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	AudienceMembers *AudienceMembersService

	Events *EventsService

	RequestStatus *RequestStatusService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAudienceMembersService(s *Service) *AudienceMembersService {
	rs := &AudienceMembersService{s: s}
	return rs
}

type AudienceMembersService struct {
	s *Service
}

func NewEventsService(s *Service) *EventsService {
	rs := &EventsService{s: s}
	return rs
}

type EventsService struct {
	s *Service
}

func NewRequestStatusService(s *Service) *RequestStatusService {
	rs := &RequestStatusService{s: s}
	return rs
}

type RequestStatusService struct {
	s *Service
}

// AdIdentifiers: Identifiers and other information used to match the
// conversion event with other online activity (such as ad clicks).
type AdIdentifiers struct {
	// Gbraid: Optional. The click identifier for clicks associated with app events
	// and originating from iOS devices starting with iOS14.
	Gbraid string `json:"gbraid,omitempty"`
	// Gclid: Optional. The Google click ID (gclid) associated with this event.
	Gclid string `json:"gclid,omitempty"`
	// LandingPageDeviceInfo: Optional. Information gathered about the device being
	// used (if any) at the time of landing onto the advertiser’s site after
	// interacting with the ad.
	LandingPageDeviceInfo *DeviceInfo `json:"landingPageDeviceInfo,omitempty"`
	// SessionAttributes: Optional. Session attributes for event attribution and
	// modeling.
	SessionAttributes string `json:"sessionAttributes,omitempty"`
	// Wbraid: Optional. The click identifier for clicks associated with web events
	// and originating from iOS devices starting with iOS14.
	Wbraid string `json:"wbraid,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Gbraid") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Gbraid") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AdIdentifiers) MarshalJSON() ([]byte, error) {
	type NoMethod AdIdentifiers
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AddressInfo: Address information for the user.
type AddressInfo struct {
	// FamilyName: Required. Family (last) name of the user, all lowercase, with no
	// punctuation, no leading or trailing whitespace, and hashed as SHA-256.
	FamilyName string `json:"familyName,omitempty"`
	// GivenName: Required. Given (first) name of the user, all lowercase, with no
	// punctuation, no leading or trailing whitespace, and hashed as SHA-256.
	GivenName string `json:"givenName,omitempty"`
	// PostalCode: Required. The postal code of the user's address.
	PostalCode string `json:"postalCode,omitempty"`
	// RegionCode: Required. The 2-letter region code in ISO-3166-1 alpha-2 of the
	// user's address.
	RegionCode string `json:"regionCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FamilyName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FamilyName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AddressInfo) MarshalJSON() ([]byte, error) {
	type NoMethod AddressInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AudienceMember: The audience member to be operated on.
type AudienceMember struct {
	// Consent: Optional. The consent setting for the user.
	Consent *Consent `json:"consent,omitempty"`
	// DestinationReferences: Optional. Defines which Destination to send the
	// audience member to.
	DestinationReferences []string `json:"destinationReferences,omitempty"`
	// MobileData: Data identifying the user's mobile devices.
	MobileData *MobileData `json:"mobileData,omitempty"`
	// PairData: Publisher Advertiser Identity Reconciliation (PAIR) IDs
	// (//support.google.com/admanager/answer/15067908).
	PairData *PairData `json:"pairData,omitempty"`
	// UserData: User-provided data that identifies the user.
	UserData *UserData `json:"userData,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Consent") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Consent") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AudienceMember) MarshalJSON() ([]byte, error) {
	type NoMethod AudienceMember
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AwsWrappedKeyInfo: A data encryption key wrapped by an AWS KMS key.
type AwsWrappedKeyInfo struct {
	// EncryptedDek: Required. The base64 encoded encrypted data encryption key.
	EncryptedDek string `json:"encryptedDek,omitempty"`
	// KekUri: Required. The URI of the AWS KMS key used to decrypt the DEK. Should
	// be in the format of `arn:{partition}:kms:{region}:{account_id}:key/{key_id}`
	// or `aws-kms://arn:{partition}:kms:{region}:{account_id}:key/{key_id}`
	KekUri string `json:"kekUri,omitempty"`
	// KeyType: Required. The type of algorithm used to encrypt the data.
	//
	// Possible values:
	//   "KEY_TYPE_UNSPECIFIED" - Unspecified key type. Should never be used.
	//   "XCHACHA20_POLY1305" - Algorithm XChaCha20-Poly1305
	KeyType string `json:"keyType,omitempty"`
	// RoleArn: Required. The Amazon Resource Name of the IAM Role to assume for
	// KMS decryption access. Should be in the format of
	// `arn:{partition}:iam::{account_id}:role/{role_name}`
	RoleArn string `json:"roleArn,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EncryptedDek") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EncryptedDek") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AwsWrappedKeyInfo) MarshalJSON() ([]byte, error) {
	type NoMethod AwsWrappedKeyInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CartData: The cart data associated with the event.
type CartData struct {
	// Items: Optional. The list of items associated with the event.
	Items []*Item `json:"items,omitempty"`
	// MerchantFeedLabel: Optional. The Merchant Center feed label associated with
	// the feed of the items.
	MerchantFeedLabel string `json:"merchantFeedLabel,omitempty"`
	// MerchantFeedLanguageCode: Optional. The language code in ISO 639-1
	// associated with the Merchant Center feed of the items.where your items are
	// uploaded.
	MerchantFeedLanguageCode string `json:"merchantFeedLanguageCode,omitempty"`
	// MerchantId: Optional. The Merchant Center ID associated with the items.
	MerchantId string `json:"merchantId,omitempty"`
	// TransactionDiscount: Optional. The sum of all discounts associated with the
	// transaction.
	TransactionDiscount float64 `json:"transactionDiscount,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CartData) MarshalJSON() ([]byte, error) {
	type NoMethod CartData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *CartData) UnmarshalJSON(data []byte) error {
	type NoMethod CartData
	var s1 struct {
		TransactionDiscount gensupport.JSONFloat64 `json:"transactionDiscount"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.TransactionDiscount = float64(s1.TransactionDiscount)
	return nil
}

// Consent: Digital Markets Act (DMA)
// (//digital-markets-act.ec.europa.eu/index_en) consent settings for the user.
type Consent struct {
	// AdPersonalization: Optional. Represents if the user consents to ad
	// personalization.
	//
	// Possible values:
	//   "CONSENT_STATUS_UNSPECIFIED" - Not specified.
	//   "CONSENT_GRANTED" - Granted.
	//   "CONSENT_DENIED" - Denied.
	AdPersonalization string `json:"adPersonalization,omitempty"`
	// AdUserData: Optional. Represents if the user consents to ad user data.
	//
	// Possible values:
	//   "CONSENT_STATUS_UNSPECIFIED" - Not specified.
	//   "CONSENT_GRANTED" - Granted.
	//   "CONSENT_DENIED" - Denied.
	AdUserData string `json:"adUserData,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdPersonalization") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdPersonalization") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Consent) MarshalJSON() ([]byte, error) {
	type NoMethod Consent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomVariable: Custom variable for ads conversions.
type CustomVariable struct {
	// DestinationReferences: Optional. Reference string used to determine which of
	// the Event.destination_references the custom variable should be sent to. If
	// empty, the Event.destination_references will be used.
	DestinationReferences []string `json:"destinationReferences,omitempty"`
	// Value: Optional. The value to store for the custom variable.
	Value string `json:"value,omitempty"`
	// Variable: Optional. The name of the custom variable to set. If the variable
	// is not found for the given destination, it will be ignored.
	Variable string `json:"variable,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DestinationReferences") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DestinationReferences") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomVariable) MarshalJSON() ([]byte, error) {
	type NoMethod CustomVariable
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Destination: The Google product you're sending data to. For example, a
// Google Ads account.
type Destination struct {
	// LinkedAccount: Optional. An account that the calling user's `login_account`
	// has access to, through an established account link. For example, a data
	// partner's `login_account` might have access to a client's `linked_account`.
	// The partner might use this field to send data from the `linked_account` to
	// another `operating_account`.
	LinkedAccount *ProductAccount `json:"linkedAccount,omitempty"`
	// LoginAccount: Optional. The account used to make this API call. To add or
	// remove data from the `operating_account`, this `login_account` must have
	// write access to the `operating_account`. For example, a manager account of
	// the `operating_account`, or an account with an established link to the
	// `operating_account`.
	LoginAccount *ProductAccount `json:"loginAccount,omitempty"`
	// OperatingAccount: Required. The account to send the data to or remove the
	// data from.
	OperatingAccount *ProductAccount `json:"operatingAccount,omitempty"`
	// ProductDestinationId: Required. The object within the product account to
	// ingest into. For example, a Google Ads audience ID, a Display & Video 360
	// audience ID or a Google Ads conversion action ID.
	ProductDestinationId string `json:"productDestinationId,omitempty"`
	// Reference: Optional. ID for this `Destination` resource, unique within the
	// request. Use to reference this `Destination` in the IngestEventsRequest and
	// IngestAudienceMembersRequest.
	Reference string `json:"reference,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LinkedAccount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LinkedAccount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Destination) MarshalJSON() ([]byte, error) {
	type NoMethod Destination
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DeviceInfo: Information about the device being used (if any) when the event
// happened.
type DeviceInfo struct {
	// IpAddress: Optional. The IP address of the device for the given context.
	// **Note:** Google Ads does not support IP address matching for end users in
	// the European Economic Area (EEA), United Kingdom (UK), or Switzerland (CH).
	// Add logic to conditionally exclude sharing IP addresses from users from
	// these regions and ensure that you provide users with clear and comprehensive
	// information about the data you collect on your sites, apps, and other
	// properties and get consent where required by law or any applicable Google
	// policies. See the About offline conversion imports
	// (https://support.google.com/google-ads/answer/2998031) page for more
	// details.
	IpAddress string `json:"ipAddress,omitempty"`
	// UserAgent: Optional. The user-agent string of the device for the given
	// context.
	UserAgent string `json:"userAgent,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IpAddress") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IpAddress") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DeviceInfo) MarshalJSON() ([]byte, error) {
	type NoMethod DeviceInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// EncryptionInfo: Encryption information for the data being ingested.
type EncryptionInfo struct {
	// AwsWrappedKeyInfo: Amazon Web Services wrapped key information.
	AwsWrappedKeyInfo *AwsWrappedKeyInfo `json:"awsWrappedKeyInfo,omitempty"`
	// GcpWrappedKeyInfo: Google Cloud Platform wrapped key information.
	GcpWrappedKeyInfo *GcpWrappedKeyInfo `json:"gcpWrappedKeyInfo,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AwsWrappedKeyInfo") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AwsWrappedKeyInfo") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EncryptionInfo) MarshalJSON() ([]byte, error) {
	type NoMethod EncryptionInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ErrorCount: The error count for a given error reason.
type ErrorCount struct {
	// Reason: The error reason of the failed records.
	//
	// Possible values:
	//   "PROCESSING_ERROR_REASON_UNSPECIFIED" - The processing error reason is
	// unknown.
	//   "PROCESSING_ERROR_REASON_INVALID_CUSTOM_VARIABLE" - The custom variable is
	// invalid.
	//   "PROCESSING_ERROR_REASON_CUSTOM_VARIABLE_NOT_ENABLED" - The status of the
	// custom variable is not enabled.
	//   "PROCESSING_ERROR_REASON_EVENT_TOO_OLD" - The conversion is older than max
	// supported age.
	//   "PROCESSING_ERROR_REASON_DENIED_CONSENT" - The ad user data is denied,
	// either by the user or in the advertiser default settings.
	//   "PROCESSING_ERROR_REASON_NO_CONSENT" - Advertiser did not give 3P consent
	// for the Ads core platform services.
	//   "PROCESSING_ERROR_REASON_UNKNOWN_CONSENT" - The overall consent
	// (determined from row level consent, request level consent, and account
	// settings) could not be determined for this user
	//   "PROCESSING_ERROR_REASON_DUPLICATE_GCLID" - A conversion with the same
	// GCLID and conversion time already exists in the system.
	//   "PROCESSING_ERROR_REASON_DUPLICATE_TRANSACTION_ID" - A conversion with the
	// same order id and conversion action combination was already uploaded.
	//   "PROCESSING_ERROR_REASON_INVALID_GBRAID" - The gbraid could not be
	// decoded.
	//   "PROCESSING_ERROR_REASON_INVALID_GCLID" - The google click ID could not be
	// decoded.
	//   "PROCESSING_ERROR_REASON_INVALID_MERCHANT_ID" - Merchant id contains
	// non-digit characters.
	//   "PROCESSING_ERROR_REASON_INVALID_WBRAID" - The wbraid could not be
	// decoded.
	//   "PROCESSING_ERROR_REASON_INTERNAL_ERROR" - Internal error.
	//
	// "PROCESSING_ERROR_REASON_DESTINATION_ACCOUNT_ENHANCED_CONVERSIONS_TERMS_NOT_S
	// IGNED" - Enhanced conversions terms are not signed in the destination
	// account.
	//   "PROCESSING_ERROR_REASON_INVALID_EVENT" - The event is invalid.
	//   "PROCESSING_ERROR_REASON_INSUFFICIENT_MATCHED_TRANSACTIONS" - The matched
	// transactions are less than the minimum threshold.
	//   "PROCESSING_ERROR_REASON_INSUFFICIENT_TRANSACTIONS" - The transactions are
	// less than the minimum threshold.
	//   "PROCESSING_ERROR_REASON_INVALID_FORMAT" - The event has format error.
	//   "PROCESSING_ERROR_REASON_DECRYPTION_ERROR" - The event has a decryption
	// error.
	//   "PROCESSING_ERROR_REASON_DEK_DECRYPTION_ERROR" - The DEK failed to be
	// decrypted.
	//   "PROCESSING_ERROR_REASON_INVALID_WIP" - The WIP is formatted incorrectly
	// or the WIP does not exist.
	//   "PROCESSING_ERROR_REASON_INVALID_KEK" - The KEK cannot decrypt data
	// because it is the wrong KEK, or it does not exist.
	//   "PROCESSING_ERROR_REASON_WIP_AUTH_FAILED" - The WIP could not be used
	// because it was rejected by its attestation condition.
	//   "PROCESSING_ERROR_REASON_KEK_PERMISSION_DENIED" - The system did not have
	// the permissions needed to access the KEK.
	//   "PROCESSING_ERROR_REASON_AWS_AUTH_FAILED" - The system failed to
	// authenticate with AWS.
	//   "PROCESSING_ERROR_REASON_USER_IDENTIFIER_DECRYPTION_ERROR" - Failed to
	// decrypt the UserIdentifier data using the DEK.
	//   "PROCESSING_ERROR_OPERATING_ACCOUNT_MISMATCH_FOR_AD_IDENTIFIER" - The user
	// attempted to ingest events with an ad identifier that isn't from the
	// operating account's ads.
	Reason string `json:"reason,omitempty"`
	// RecordCount: The count of records that failed to upload for a given reason.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Reason") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reason") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ErrorCount) MarshalJSON() ([]byte, error) {
	type NoMethod ErrorCount
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ErrorInfo: Error counts for each type of error.
type ErrorInfo struct {
	// ErrorCounts: A list of errors and counts per error reason. May not be
	// populated in all cases.
	ErrorCounts []*ErrorCount `json:"errorCounts,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ErrorCounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ErrorCounts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ErrorInfo) MarshalJSON() ([]byte, error) {
	type NoMethod ErrorInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Event: An event representing a user interaction with an advertiser's website
// or app.
type Event struct {
	// AdIdentifiers: Optional. Identifiers and other information used to match the
	// conversion event with other online activity (such as ad clicks).
	AdIdentifiers *AdIdentifiers `json:"adIdentifiers,omitempty"`
	// AdditionalEventParameters: Optional. A bucket of any event parameters
	// (https://developers.google.com/analytics/devguides/collection/protocol/ga4/reference/events)
	// to be included within the event that were not already specified using other
	// structured fields.
	AdditionalEventParameters []*EventParameter `json:"additionalEventParameters,omitempty"`
	// CartData: Optional. Information about the transaction and items associated
	// with the event.
	CartData *CartData `json:"cartData,omitempty"`
	// ClientId: Optional. A unique identifier for the user instance of a web
	// client for this GA4 web stream.
	ClientId string `json:"clientId,omitempty"`
	// Consent: Optional. Information about whether the associated user has
	// provided different types of consent.
	Consent *Consent `json:"consent,omitempty"`
	// ConversionValue: Optional. The conversion value associated with the event,
	// for value-based conversions.
	ConversionValue float64 `json:"conversionValue,omitempty"`
	// Currency: Optional. The currency code associated with all monetary values
	// within this event.
	Currency string `json:"currency,omitempty"`
	// CustomVariables: Optional. Additional key/value pair information to send to
	// the conversion containers (conversion action or FL activity).
	CustomVariables []*CustomVariable `json:"customVariables,omitempty"`
	// DestinationReferences: Optional. Reference string used to determine the
	// destination. If empty, the event will be sent to all destinations in the
	// request.
	DestinationReferences []string `json:"destinationReferences,omitempty"`
	// EventDeviceInfo: Optional. Information gathered about the device being used
	// (if any) when the event happened.
	EventDeviceInfo *DeviceInfo `json:"eventDeviceInfo,omitempty"`
	// EventName: Optional. The name of the event. Required for GA4 events.
	EventName string `json:"eventName,omitempty"`
	// EventSource: Optional. Signal for where the event happened (web, app,
	// in-store, etc.).
	//
	// Possible values:
	//   "EVENT_SOURCE_UNSPECIFIED" - Unspecified EventSource. Should never be
	// used.
	//   "WEB" - The event was generated from a web browser.
	//   "APP" - The event was generated from an app.
	//   "IN_STORE" - The event was generated from an in-store transaction.
	//   "PHONE" - The event was generated from a phone call.
	//   "OTHER" - The event was generated from other sources.
	EventSource string `json:"eventSource,omitempty"`
	// EventTimestamp: Required. The time the event occurred.
	EventTimestamp string `json:"eventTimestamp,omitempty"`
	// ExperimentalFields: Optional. A list of key/value pairs for experimental
	// fields that may eventually be promoted to be part of the API.
	ExperimentalFields []*ExperimentalField `json:"experimentalFields,omitempty"`
	// LastUpdatedTimestamp: Optional. The last time the event was updated.
	LastUpdatedTimestamp string `json:"lastUpdatedTimestamp,omitempty"`
	// TransactionId: Optional. The unique identifier for this event. Required for
	// conversions using multiple data sources.
	TransactionId string `json:"transactionId,omitempty"`
	// UserData: Optional. Pieces of user provided data, representing the user the
	// event is associated with.
	UserData *UserData `json:"userData,omitempty"`
	// UserId: Optional. A unique identifier for a user, as defined by the
	// advertiser.
	UserId string `json:"userId,omitempty"`
	// UserProperties: Optional. Advertiser-assessed information about the user at
	// the time that the event happened.
	UserProperties *UserProperties `json:"userProperties,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdIdentifiers") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdIdentifiers") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Event) MarshalJSON() ([]byte, error) {
	type NoMethod Event
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Event) UnmarshalJSON(data []byte) error {
	type NoMethod Event
	var s1 struct {
		ConversionValue gensupport.JSONFloat64 `json:"conversionValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.ConversionValue = float64(s1.ConversionValue)
	return nil
}

// EventParameter: Event parameter for GA4 events.
type EventParameter struct {
	// ParameterName: Required. The name of the parameter to use.
	ParameterName string `json:"parameterName,omitempty"`
	// Value: Required. The string representation of the value of the parameter to
	// set.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ParameterName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ParameterName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EventParameter) MarshalJSON() ([]byte, error) {
	type NoMethod EventParameter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ExperimentalField: Experimental field representing unofficial fields.
type ExperimentalField struct {
	// Field: Optional. The name of the field to use.
	Field string `json:"field,omitempty"`
	// Value: Optional. The value the field to set.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Field") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Field") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ExperimentalField) MarshalJSON() ([]byte, error) {
	type NoMethod ExperimentalField
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GcpWrappedKeyInfo: Information about the Google Cloud Platform wrapped key.
type GcpWrappedKeyInfo struct {
	// EncryptedDek: Required. The base64 encoded encrypted data encryption key.
	EncryptedDek string `json:"encryptedDek,omitempty"`
	// KekUri: Required. Google Cloud Platform Cloud Key Management Service
	// resource ID (//cloud.google.com/kms/docs/getting-resource-ids). Should be in
	// the format of
	// `projects/{project}/locations/{location}/keyRings/{key_ring}/cryptoKeys/{key}
	// ` or
	// `gcp-kms://projects/{project}/locations/{location}/keyRings/{key_ring}/crypto
	// Keys/{key}`
	KekUri string `json:"kekUri,omitempty"`
	// KeyType: Required. The type of algorithm used to encrypt the data.
	//
	// Possible values:
	//   "KEY_TYPE_UNSPECIFIED" - Unspecified key type. Should never be used.
	//   "XCHACHA20_POLY1305" - Algorithm XChaCha20-Poly1305
	KeyType string `json:"keyType,omitempty"`
	// WipProvider: Required. The Workload Identity
	// (//cloud.google.com/iam/docs/workload-identity-federation) pool provider
	// required to use KEK.
	WipProvider string `json:"wipProvider,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EncryptedDek") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EncryptedDek") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GcpWrappedKeyInfo) MarshalJSON() ([]byte, error) {
	type NoMethod GcpWrappedKeyInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestAudienceMembersRequest: Request to upload audience members to the
// provided destinations. Returns an IngestAudienceMembersResponse.
type IngestAudienceMembersRequest struct {
	// AudienceMembers: Required. The list of users to send to the specified
	// destinations. At most 10000 AudienceMember resources can be sent in a single
	// request.
	AudienceMembers []*AudienceMember `json:"audienceMembers,omitempty"`
	// Consent: Optional. Request-level consent to apply to all users in the
	// request. User-level consent overrides request-level consent, and can be
	// specified in each AudienceMember.
	Consent *Consent `json:"consent,omitempty"`
	// Destinations: Required. The list of destinations to send the audience
	// members to.
	Destinations []*Destination `json:"destinations,omitempty"`
	// Encoding: Optional. Required for UserData uploads. The encoding type of the
	// user identifiers. For hashed user identifiers, this is the encoding type of
	// the hashed string. For encrypted hashed user identifiers, this is the
	// encoding type of the outer encrypted string, but not necessarily the inner
	// hashed string, meaning the inner hashed string could be encoded in a
	// different way than the outer encrypted string. For non `UserData` uploads,
	// this field is ignored.
	//
	// Possible values:
	//   "ENCODING_UNSPECIFIED" - Unspecified Encoding type. Should never be used.
	//   "HEX" - Hex encoding.
	//   "BASE64" - Base 64 encoding.
	Encoding string `json:"encoding,omitempty"`
	// EncryptionInfo: Optional. Encryption information for UserData uploads. If
	// not set, it's assumed that uploaded identifying information is hashed but
	// not encrypted. For non `UserData` uploads, this field is ignored.
	EncryptionInfo *EncryptionInfo `json:"encryptionInfo,omitempty"`
	// TermsOfService: Optional. The terms of service that the user has
	// accepted/rejected.
	TermsOfService *TermsOfService `json:"termsOfService,omitempty"`
	// ValidateOnly: Optional. For testing purposes. If `true`, the request is
	// validated but not executed. Only errors are returned, not results.
	ValidateOnly bool `json:"validateOnly,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AudienceMembers") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AudienceMembers") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestAudienceMembersRequest) MarshalJSON() ([]byte, error) {
	type NoMethod IngestAudienceMembersRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestAudienceMembersResponse: Response from the
// IngestAudienceMembersRequest.
type IngestAudienceMembersResponse struct {
	// RequestId: The auto-generated ID of the request.
	RequestId string `json:"requestId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "RequestId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RequestId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestAudienceMembersResponse) MarshalJSON() ([]byte, error) {
	type NoMethod IngestAudienceMembersResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestAudienceMembersStatus: The status of the ingest audience members
// request.
type IngestAudienceMembersStatus struct {
	// MobileDataIngestionStatus: The status of the mobile data ingestion to the
	// destination.
	MobileDataIngestionStatus *IngestMobileDataStatus `json:"mobileDataIngestionStatus,omitempty"`
	// PairDataIngestionStatus: The status of the pair data ingestion to the
	// destination.
	PairDataIngestionStatus *IngestPairDataStatus `json:"pairDataIngestionStatus,omitempty"`
	// UserDataIngestionStatus: The status of the user data ingestion to the
	// destination.
	UserDataIngestionStatus *IngestUserDataStatus `json:"userDataIngestionStatus,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MobileDataIngestionStatus")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MobileDataIngestionStatus") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestAudienceMembersStatus) MarshalJSON() ([]byte, error) {
	type NoMethod IngestAudienceMembersStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestEventsRequest: Request to upload audience members to the provided
// destinations. Returns an IngestEventsResponse.
type IngestEventsRequest struct {
	// Consent: Optional. Request-level consent to apply to all users in the
	// request. User-level consent overrides request-level consent, and can be
	// specified in each Event.
	Consent *Consent `json:"consent,omitempty"`
	// Destinations: Required. The list of destinations to send the events to.
	Destinations []*Destination `json:"destinations,omitempty"`
	// Encoding: Optional. Required for UserData uploads. The encoding type of the
	// user identifiers. For hashed user identifiers, this is the encoding type of
	// the hashed string. For encrypted hashed user identifiers, this is the
	// encoding type of the outer encrypted string, but not necessarily the inner
	// hashed string, meaning the inner hashed string could be encoded in a
	// different way than the outer encrypted string. For non `UserData` uploads,
	// this field is ignored.
	//
	// Possible values:
	//   "ENCODING_UNSPECIFIED" - Unspecified Encoding type. Should never be used.
	//   "HEX" - Hex encoding.
	//   "BASE64" - Base 64 encoding.
	Encoding string `json:"encoding,omitempty"`
	// EncryptionInfo: Optional. Encryption information for UserData uploads. If
	// not set, it's assumed that uploaded identifying information is hashed but
	// not encrypted. For non `UserData` uploads, this field is ignored.
	EncryptionInfo *EncryptionInfo `json:"encryptionInfo,omitempty"`
	// Events: Required. The list of events to send to the specified destinations.
	// At most 2000 Event resources can be sent in a single request.
	Events []*Event `json:"events,omitempty"`
	// ValidateOnly: Optional. For testing purposes. If `true`, the request is
	// validated but not executed. Only errors are returned, not results.
	ValidateOnly bool `json:"validateOnly,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Consent") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Consent") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestEventsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod IngestEventsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestEventsResponse: Response from the IngestEventsRequest.
type IngestEventsResponse struct {
	// RequestId: The auto-generated ID of the request.
	RequestId string `json:"requestId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "RequestId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RequestId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestEventsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod IngestEventsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestEventsStatus: The status of the events ingestion to the destination.
type IngestEventsStatus struct {
	// RecordCount: The total count of events sent in the upload request. Includes
	// all events in the request, regardless of whether they were successfully
	// ingested or not.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "RecordCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RecordCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestEventsStatus) MarshalJSON() ([]byte, error) {
	type NoMethod IngestEventsStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestMobileDataStatus: The status of the mobile data ingestion to the
// destination containing stats related to the ingestion.
type IngestMobileDataStatus struct {
	// MobileIdCount: The total count of mobile ids sent in the upload request for
	// the destination. Includes all mobile ids in the request, regardless of
	// whether they were successfully ingested or not.
	MobileIdCount int64 `json:"mobileIdCount,omitempty,string"`
	// RecordCount: The total count of audience members sent in the upload request
	// for the destination. Includes all audience members in the request,
	// regardless of whether they were successfully ingested or not.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "MobileIdCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MobileIdCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestMobileDataStatus) MarshalJSON() ([]byte, error) {
	type NoMethod IngestMobileDataStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestPairDataStatus: The status of the pair data ingestion to the
// destination containing stats related to the ingestion.
type IngestPairDataStatus struct {
	// PairIdCount: The total count of pair ids sent in the upload request for the
	// destination. Includes all pair ids in the request, regardless of whether
	// they were successfully ingested or not.
	PairIdCount int64 `json:"pairIdCount,omitempty,string"`
	// RecordCount: The total count of audience members sent in the upload request
	// for the destination. Includes all audience members in the request,
	// regardless of whether they were successfully ingested or not.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "PairIdCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PairIdCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestPairDataStatus) MarshalJSON() ([]byte, error) {
	type NoMethod IngestPairDataStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IngestUserDataStatus: The status of the user data ingestion to the
// destination containing stats related to the ingestion.
type IngestUserDataStatus struct {
	// RecordCount: The total count of audience members sent in the upload request
	// for the destination. Includes all audience members in the request,
	// regardless of whether they were successfully ingested or not.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// UploadMatchRateRange: The match rate range of the upload.
	//
	// Possible values:
	//   "MATCH_RATE_RANGE_UNKNOWN" - The match rate range is unknown.
	//   "MATCH_RATE_RANGE_NOT_ELIGIBLE" - The match rate range is not eligible.
	//   "MATCH_RATE_RANGE_LESS_THAN_20" - The match rate range is less than 20%
	// (in the interval `[0, 20)`).
	//   "MATCH_RATE_RANGE_20_TO_30" - The match rate range is between 20% and 30%
	// (in the interval `[20, 31)`).
	//   "MATCH_RATE_RANGE_31_TO_40" - The match rate range is between 31% and 40%
	// (in the interval `[31, 41)`).
	//   "MATCH_RATE_RANGE_41_TO_50" - The match rate range is between 41% and 50%
	// (in the interval `[41, 51)`).
	//   "MATCH_RATE_RANGE_51_TO_60" - The match rate range is between 51% and 60%
	// (in the interval `[51, 61)`.
	//   "MATCH_RATE_RANGE_61_TO_70" - The match rate range is between 61% and 70%
	// (in the interval `[61, 71)`).
	//   "MATCH_RATE_RANGE_71_TO_80" - The match rate range is between 71% and 80%
	// (in the interval `[71, 81)`).
	//   "MATCH_RATE_RANGE_81_TO_90" - The match rate range is between 81% and 90%
	// (in the interval `[81, 91)`).
	//   "MATCH_RATE_RANGE_91_TO_100" - The match rate range is between 91% and
	// 100% (in the interval `[91, 100]`).
	UploadMatchRateRange string `json:"uploadMatchRateRange,omitempty"`
	// UserIdentifierCount: The total count of user identifiers sent in the upload
	// request for the destination. Includes all user identifiers in the request,
	// regardless of whether they were successfully ingested or not.
	UserIdentifierCount int64 `json:"userIdentifierCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "RecordCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RecordCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IngestUserDataStatus) MarshalJSON() ([]byte, error) {
	type NoMethod IngestUserDataStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Item: Represents an item in the cart associated with the event.
type Item struct {
	// AdditionalItemParameters: Optional. A bucket of any event parameters related
	// to an item
	// (https://developers.google.com/analytics/devguides/collection/protocol/ga4/reference/events)
	// to be included within the event that were not already specified using other
	// structured fields.
	AdditionalItemParameters []*ItemParameter `json:"additionalItemParameters,omitempty"`
	// ItemId: Optional. A unique identifier to reference the item.
	ItemId string `json:"itemId,omitempty"`
	// MerchantProductId: Optional. The product ID within the Merchant Center
	// account.
	MerchantProductId string `json:"merchantProductId,omitempty"`
	// Quantity: Optional. The number of this item associated with the event.
	Quantity int64 `json:"quantity,omitempty,string"`
	// UnitPrice: Optional. The unit price excluding tax, shipping, and any
	// transaction level discounts.
	UnitPrice float64 `json:"unitPrice,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalItemParameters")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalItemParameters") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Item) MarshalJSON() ([]byte, error) {
	type NoMethod Item
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *Item) UnmarshalJSON(data []byte) error {
	type NoMethod Item
	var s1 struct {
		UnitPrice gensupport.JSONFloat64 `json:"unitPrice"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.UnitPrice = float64(s1.UnitPrice)
	return nil
}

// ItemParameter: A bucket of any event parameters related to an item
// (https://developers.google.com/analytics/devguides/collection/protocol/ga4/reference/events)
// to be included within the event that were not already specified using other
// structured fields.
type ItemParameter struct {
	// ParameterName: Required. The name of the parameter to use.
	ParameterName string `json:"parameterName,omitempty"`
	// Value: Required. The string representation of the value of the parameter to
	// set.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ParameterName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ParameterName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ItemParameter) MarshalJSON() ([]byte, error) {
	type NoMethod ItemParameter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MobileData: Mobile IDs for the audience. At least one mobile ID is required.
type MobileData struct {
	// MobileIds: Required. The list of mobile device IDs (advertising ID/IDFA). At
	// most 10 `mobileIds` can be provided in a single AudienceMember.
	MobileIds []string `json:"mobileIds,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MobileIds") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MobileIds") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MobileData) MarshalJSON() ([]byte, error) {
	type NoMethod MobileData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PairData: PAIR (//support.google.com/admanager/answer/15067908) IDs for the
// audience. At least one PAIR ID is required.
type PairData struct {
	// PairIds: Required. Cleanroom-provided PII data, hashed with SHA256, and
	// encrypted with an EC commutative cipher using publisher key for the PAIR
	// ((//support.google.com/admanager/answer/15067908)) user list. At most 10
	// `pairIds` can be provided in a single AudienceMember.
	PairIds []string `json:"pairIds,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PairIds") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PairIds") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PairData) MarshalJSON() ([]byte, error) {
	type NoMethod PairData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductAccount: Represents a specific account.
type ProductAccount struct {
	// AccountId: Required. The ID of the account. For example, your Google Ads
	// account ID.
	AccountId string `json:"accountId,omitempty"`
	// AccountType: Optional. The type of the account. For example, `GOOGLE_ADS`.
	// Either `account_type` or the deprecated `product` is required. If both are
	// set, the values must match.
	//
	// Possible values:
	//   "ACCOUNT_TYPE_UNSPECIFIED" - Unspecified product. Should never be used.
	//   "GOOGLE_ADS" - Google Ads.
	//   "DISPLAY_VIDEO_PARTNER" - Display & Video 360 partner.
	//   "DISPLAY_VIDEO_ADVERTISER" - Display & Video 360 advertiser.
	//   "DATA_PARTNER" - Data Partner.
	//   "GOOGLE_ANALYTICS_PROPERTY" - Google Analytics.
	AccountType string `json:"accountType,omitempty"`
	// Product: Deprecated. Use `account_type` instead.
	//
	// Possible values:
	//   "PRODUCT_UNSPECIFIED" - Unspecified product. Should never be used.
	//   "GOOGLE_ADS" - Google Ads.
	//   "DISPLAY_VIDEO_PARTNER" - Display & Video 360 partner.
	//   "DISPLAY_VIDEO_ADVERTISER" - Display & Video 360 advertiser.
	//   "DATA_PARTNER" - Data Partner.
	Product string `json:"product,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccountId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductAccount) MarshalJSON() ([]byte, error) {
	type NoMethod ProductAccount
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RemoveAudienceMembersRequest: Request to remove users from an audience in
// the provided destinations. Returns a RemoveAudienceMembersResponse.
type RemoveAudienceMembersRequest struct {
	// AudienceMembers: Required. The list of users to remove.
	AudienceMembers []*AudienceMember `json:"audienceMembers,omitempty"`
	// Destinations: Required. The list of destinations to remove the users from.
	Destinations []*Destination `json:"destinations,omitempty"`
	// Encoding: Optional. Required for UserData uploads. The encoding type of the
	// user identifiers. Applies to only the outer encoding for encrypted user
	// identifiers. For non `UserData` uploads, this field is ignored.
	//
	// Possible values:
	//   "ENCODING_UNSPECIFIED" - Unspecified Encoding type. Should never be used.
	//   "HEX" - Hex encoding.
	//   "BASE64" - Base 64 encoding.
	Encoding string `json:"encoding,omitempty"`
	// EncryptionInfo: Optional. Encryption information for UserData uploads. If
	// not set, it's assumed that uploaded identifying information is hashed but
	// not encrypted. For non `UserData` uploads, this field is ignored.
	EncryptionInfo *EncryptionInfo `json:"encryptionInfo,omitempty"`
	// ValidateOnly: Optional. For testing purposes. If `true`, the request is
	// validated but not executed. Only errors are returned, not results.
	ValidateOnly bool `json:"validateOnly,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AudienceMembers") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AudienceMembers") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RemoveAudienceMembersRequest) MarshalJSON() ([]byte, error) {
	type NoMethod RemoveAudienceMembersRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RemoveAudienceMembersResponse: Response from the
// RemoveAudienceMembersRequest.
type RemoveAudienceMembersResponse struct {
	// RequestId: The auto-generated ID of the request.
	RequestId string `json:"requestId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "RequestId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RequestId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RemoveAudienceMembersResponse) MarshalJSON() ([]byte, error) {
	type NoMethod RemoveAudienceMembersResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RemoveAudienceMembersStatus: The status of the remove audience members
// request.
type RemoveAudienceMembersStatus struct {
	// MobileDataRemovalStatus: The status of the mobile data removal from the
	// destination.
	MobileDataRemovalStatus *RemoveMobileDataStatus `json:"mobileDataRemovalStatus,omitempty"`
	// PairDataRemovalStatus: The status of the pair data removal from the
	// destination.
	PairDataRemovalStatus *RemovePairDataStatus `json:"pairDataRemovalStatus,omitempty"`
	// UserDataRemovalStatus: The status of the user data removal from the
	// destination.
	UserDataRemovalStatus *RemoveUserDataStatus `json:"userDataRemovalStatus,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MobileDataRemovalStatus") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MobileDataRemovalStatus") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RemoveAudienceMembersStatus) MarshalJSON() ([]byte, error) {
	type NoMethod RemoveAudienceMembersStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RemoveMobileDataStatus: The status of the mobile data removal from the
// destination.
type RemoveMobileDataStatus struct {
	// MobileIdCount: The total count of mobile Ids sent in the removal request.
	// Includes all mobile ids in the request, regardless of whether they were
	// successfully removed or not.
	MobileIdCount int64 `json:"mobileIdCount,omitempty,string"`
	// RecordCount: The total count of audience members sent in the removal
	// request. Includes all audience members in the request, regardless of whether
	// they were successfully removed or not.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "MobileIdCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MobileIdCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RemoveMobileDataStatus) MarshalJSON() ([]byte, error) {
	type NoMethod RemoveMobileDataStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RemovePairDataStatus: The status of the pair data removal from the
// destination.
type RemovePairDataStatus struct {
	// PairIdCount: The total count of pair ids sent in the removal request.
	// Includes all pair ids in the request, regardless of whether they were
	// successfully removed or not.
	PairIdCount int64 `json:"pairIdCount,omitempty,string"`
	// RecordCount: The total count of audience members sent in the removal
	// request. Includes all audience members in the request, regardless of whether
	// they were successfully removed or not.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "PairIdCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PairIdCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RemovePairDataStatus) MarshalJSON() ([]byte, error) {
	type NoMethod RemovePairDataStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RemoveUserDataStatus: The status of the user data removal from the
// destination.
type RemoveUserDataStatus struct {
	// RecordCount: The total count of audience members sent in the removal
	// request. Includes all audience members in the request, regardless of whether
	// they were successfully removed or not.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// UserIdentifierCount: The total count of user identifiers sent in the removal
	// request. Includes all user identifiers in the request, regardless of whether
	// they were successfully removed or not.
	UserIdentifierCount int64 `json:"userIdentifierCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "RecordCount") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RecordCount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RemoveUserDataStatus) MarshalJSON() ([]byte, error) {
	type NoMethod RemoveUserDataStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RequestStatusPerDestination: A request status per destination.
type RequestStatusPerDestination struct {
	// AudienceMembersIngestionStatus: The status of the ingest audience members
	// request.
	AudienceMembersIngestionStatus *IngestAudienceMembersStatus `json:"audienceMembersIngestionStatus,omitempty"`
	// AudienceMembersRemovalStatus: The status of the remove audience members
	// request.
	AudienceMembersRemovalStatus *RemoveAudienceMembersStatus `json:"audienceMembersRemovalStatus,omitempty"`
	// Destination: A destination within a DM API request.
	Destination *Destination `json:"destination,omitempty"`
	// ErrorInfo: An error info error containing the error reason and error counts
	// related to the upload.
	ErrorInfo *ErrorInfo `json:"errorInfo,omitempty"`
	// EventsIngestionStatus: The status of the ingest events request.
	EventsIngestionStatus *IngestEventsStatus `json:"eventsIngestionStatus,omitempty"`
	// RequestStatus: The request status of the destination.
	//
	// Possible values:
	//   "REQUEST_STATUS_UNKNOWN" - The request status is unknown.
	//   "SUCCESS" - The request succeeded.
	//   "PROCESSING" - The request is processing.
	//   "FAILED" - The request failed.
	//   "PARTIAL_SUCCESS" - The request partially succeeded.
	RequestStatus string `json:"requestStatus,omitempty"`
	// WarningInfo: A warning info containing the warning reason and warning counts
	// related to the upload.
	WarningInfo *WarningInfo `json:"warningInfo,omitempty"`
	// ForceSendFields is a list of field names (e.g.
	// "AudienceMembersIngestionStatus") to unconditionally include in API
	// requests. By default, fields with empty or default values are omitted from
	// API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AudienceMembersIngestionStatus")
	// to include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RequestStatusPerDestination) MarshalJSON() ([]byte, error) {
	type NoMethod RequestStatusPerDestination
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RetrieveRequestStatusResponse: Response from the
// RetrieveRequestStatusRequest.
type RetrieveRequestStatusResponse struct {
	// RequestStatusPerDestination: A list of request statuses per destination. The
	// order of the statuses matches the order of the destinations in the original
	// request.
	RequestStatusPerDestination []*RequestStatusPerDestination `json:"requestStatusPerDestination,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g.
	// "RequestStatusPerDestination") to unconditionally include in API requests.
	// By default, fields with empty or default values are omitted from API
	// requests. See https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields
	// for more details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RequestStatusPerDestination") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RetrieveRequestStatusResponse) MarshalJSON() ([]byte, error) {
	type NoMethod RetrieveRequestStatusResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TermsOfService: The terms of service that the user has accepted/rejected.
type TermsOfService struct {
	// CustomerMatchTermsOfServiceStatus: Optional. The Customer Match terms of
	// service: https://support.google.com/adspolicy/answer/6299717. This must be
	// accepted when ingesting UserData or MobileData. This field is not required
	// for Partner Match User list.
	//
	// Possible values:
	//   "TERMS_OF_SERVICE_STATUS_UNSPECIFIED" - Not specified.
	//   "ACCEPTED" - Status indicating the caller has chosen to accept the terms
	// of service.
	//   "REJECTED" - Status indicating the caller has chosen to reject the terms
	// of service.
	CustomerMatchTermsOfServiceStatus string `json:"customerMatchTermsOfServiceStatus,omitempty"`
	// ForceSendFields is a list of field names (e.g.
	// "CustomerMatchTermsOfServiceStatus") to unconditionally include in API
	// requests. By default, fields with empty or default values are omitted from
	// API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g.
	// "CustomerMatchTermsOfServiceStatus") to include in API requests with the
	// JSON null value. By default, fields with empty values are omitted from API
	// requests. See https://pkg.go.dev/google.golang.org/api#hdr-NullFields for
	// more details.
	NullFields []string `json:"-"`
}

func (s TermsOfService) MarshalJSON() ([]byte, error) {
	type NoMethod TermsOfService
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UserData: Data that identifies the user. At least one identifier is
// required.
type UserData struct {
	// UserIdentifiers: Required. The identifiers for the user. It's possible to
	// provide multiple instances of the same type of data (for example, multiple
	// email addresses). To increase the likelihood of a match, provide as many
	// identifiers as possible. At most 10 `userIdentifiers` can be provided in a
	// single AudienceMember or Event.
	UserIdentifiers []*UserIdentifier `json:"userIdentifiers,omitempty"`
	// ForceSendFields is a list of field names (e.g. "UserIdentifiers") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "UserIdentifiers") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UserData) MarshalJSON() ([]byte, error) {
	type NoMethod UserData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UserIdentifier: A single identifier for the user.
type UserIdentifier struct {
	// Address: The known components of a user's address. Holds a grouping of
	// identifiers that are matched all at once.
	Address *AddressInfo `json:"address,omitempty"`
	// EmailAddress: Hashed email address using SHA-256 hash function after
	// normalization.
	EmailAddress string `json:"emailAddress,omitempty"`
	// PhoneNumber: Hashed phone number using SHA-256 hash function after
	// normalization (E164 standard).
	PhoneNumber string `json:"phoneNumber,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Address") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Address") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UserIdentifier) MarshalJSON() ([]byte, error) {
	type NoMethod UserIdentifier
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UserProperties: Advertiser-assessed information about the user at the time
// that the event happened. See
// https://support.google.com/google-ads/answer/14007601 for more details.
type UserProperties struct {
	// AdditionalUserProperties: Optional. A bucket of any additional user
	// properties
	// (https://developers.google.com/analytics/devguides/collection/protocol/ga4/user-properties)
	// for the user associated with this event.
	AdditionalUserProperties []*UserProperty `json:"additionalUserProperties,omitempty"`
	// CustomerType: Optional. Type of the customer associated with the event.
	//
	// Possible values:
	//   "CUSTOMER_TYPE_UNSPECIFIED" - Unspecified CustomerType. Should never be
	// used.
	//   "NEW" - The customer is new to the advertiser.
	//   "RETURNING" - The customer is returning to the advertiser.
	//   "REENGAGED" - The customer has re-engaged with the advertiser.
	CustomerType string `json:"customerType,omitempty"`
	// CustomerValueBucket: Optional. The advertiser-assessed value of the
	// customer.
	//
	// Possible values:
	//   "CUSTOMER_VALUE_BUCKET_UNSPECIFIED" - Unspecified CustomerValueBucket.
	// Should never be used.
	//   "LOW" - The customer is low value.
	//   "MEDIUM" - The customer is medium value.
	//   "HIGH" - The customer is high value.
	CustomerValueBucket string `json:"customerValueBucket,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalUserProperties")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalUserProperties") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UserProperties) MarshalJSON() ([]byte, error) {
	type NoMethod UserProperties
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UserProperty: A bucket of any additional user properties
// (https://developers.google.com/analytics/devguides/collection/protocol/ga4/user-properties)
// for the user associated with this event.
type UserProperty struct {
	// PropertyName: Required. The name of the user property to use.
	PropertyName string `json:"propertyName,omitempty"`
	// Value: Required. The string representation of the value of the user property
	// to use.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PropertyName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PropertyName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UserProperty) MarshalJSON() ([]byte, error) {
	type NoMethod UserProperty
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WarningCount: The warning count for a given warning reason.
type WarningCount struct {
	// Reason: The warning reason.
	//
	// Possible values:
	//   "PROCESSING_WARNING_REASON_UNSPECIFIED" - The processing warning reason is
	// unknown.
	//   "PROCESSING_WARNING_REASON_KEK_PERMISSION_DENIED" - The system did not
	// have the permissions needed to access the KEK.
	//   "PROCESSING_WARNING_REASON_DEK_DECRYPTION_ERROR" - The DEK failed to be
	// decrypted.
	//   "PROCESSING_WARNING_REASON_DECRYPTION_ERROR" - The event has a decryption
	// error.
	//   "PROCESSING_WARNING_REASON_WIP_AUTH_FAILED" - The WIP could not be used
	// because it was rejected by its attestation condition.
	//   "PROCESSING_WARNING_REASON_INVALID_WIP" - The WIP is formatted incorrectly
	// or the WIP does not exist.
	//   "PROCESSING_WARNING_REASON_INVALID_KEK" - The KEK cannot decrypt data
	// because it is the wrong KEK, or it does not exist.
	//   "PROCESSING_WARNING_REASON_USER_IDENTIFIER_DECRYPTION_ERROR" - Failed to
	// decrypt th UserIdentifier data using the DEK.
	//   "PROCESSING_WARNING_REASON_INTERNAL_ERROR" - Internal error.
	//   "PROCESSING_WARNING_REASON_AWS_AUTH_FAILED" - The system failed to
	// authenticate with AWS.
	Reason string `json:"reason,omitempty"`
	// RecordCount: The count of records that have a warning.
	RecordCount int64 `json:"recordCount,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Reason") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Reason") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WarningCount) MarshalJSON() ([]byte, error) {
	type NoMethod WarningCount
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WarningInfo: Warning counts for each type of warning.
type WarningInfo struct {
	// WarningCounts: A list of warnings and counts per warning reason.
	WarningCounts []*WarningCount `json:"warningCounts,omitempty"`
	// ForceSendFields is a list of field names (e.g. "WarningCounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "WarningCounts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WarningInfo) MarshalJSON() ([]byte, error) {
	type NoMethod WarningInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AudienceMembersIngestCall struct {
	s                            *Service
	ingestaudiencemembersrequest *IngestAudienceMembersRequest
	urlParams_                   gensupport.URLParams
	ctx_                         context.Context
	header_                      http.Header
}

// Ingest: Uploads a list of AudienceMember resources to the provided
// Destination.
func (r *AudienceMembersService) Ingest(ingestaudiencemembersrequest *IngestAudienceMembersRequest) *AudienceMembersIngestCall {
	c := &AudienceMembersIngestCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.ingestaudiencemembersrequest = ingestaudiencemembersrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AudienceMembersIngestCall) Fields(s ...googleapi.Field) *AudienceMembersIngestCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AudienceMembersIngestCall) Context(ctx context.Context) *AudienceMembersIngestCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AudienceMembersIngestCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AudienceMembersIngestCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.ingestaudiencemembersrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/audienceMembers:ingest")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "datamanager.audienceMembers.ingest", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "datamanager.audienceMembers.ingest" call.
// Any non-2xx status code is an error. Response headers are in either
// *IngestAudienceMembersResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AudienceMembersIngestCall) Do(opts ...googleapi.CallOption) (*IngestAudienceMembersResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &IngestAudienceMembersResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "datamanager.audienceMembers.ingest", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AudienceMembersRemoveCall struct {
	s                            *Service
	removeaudiencemembersrequest *RemoveAudienceMembersRequest
	urlParams_                   gensupport.URLParams
	ctx_                         context.Context
	header_                      http.Header
}

// Remove: Removes a list of AudienceMember resources from the provided
// Destination.
func (r *AudienceMembersService) Remove(removeaudiencemembersrequest *RemoveAudienceMembersRequest) *AudienceMembersRemoveCall {
	c := &AudienceMembersRemoveCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.removeaudiencemembersrequest = removeaudiencemembersrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AudienceMembersRemoveCall) Fields(s ...googleapi.Field) *AudienceMembersRemoveCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AudienceMembersRemoveCall) Context(ctx context.Context) *AudienceMembersRemoveCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AudienceMembersRemoveCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AudienceMembersRemoveCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.removeaudiencemembersrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/audienceMembers:remove")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "datamanager.audienceMembers.remove", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "datamanager.audienceMembers.remove" call.
// Any non-2xx status code is an error. Response headers are in either
// *RemoveAudienceMembersResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AudienceMembersRemoveCall) Do(opts ...googleapi.CallOption) (*RemoveAudienceMembersResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &RemoveAudienceMembersResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "datamanager.audienceMembers.remove", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type EventsIngestCall struct {
	s                   *Service
	ingesteventsrequest *IngestEventsRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// Ingest: Uploads a list of Event resources from the provided Destination.
func (r *EventsService) Ingest(ingesteventsrequest *IngestEventsRequest) *EventsIngestCall {
	c := &EventsIngestCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.ingesteventsrequest = ingesteventsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *EventsIngestCall) Fields(s ...googleapi.Field) *EventsIngestCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *EventsIngestCall) Context(ctx context.Context) *EventsIngestCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *EventsIngestCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *EventsIngestCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.ingesteventsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/events:ingest")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "datamanager.events.ingest", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "datamanager.events.ingest" call.
// Any non-2xx status code is an error. Response headers are in either
// *IngestEventsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *EventsIngestCall) Do(opts ...googleapi.CallOption) (*IngestEventsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &IngestEventsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "datamanager.events.ingest", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RequestStatusRetrieveCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Retrieve: Gets the status of a request given request id.
func (r *RequestStatusService) Retrieve() *RequestStatusRetrieveCall {
	c := &RequestStatusRetrieveCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// RequestId sets the optional parameter "requestId": Required. Required. The
// request ID of the Data Manager API request.
func (c *RequestStatusRetrieveCall) RequestId(requestId string) *RequestStatusRetrieveCall {
	c.urlParams_.Set("requestId", requestId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RequestStatusRetrieveCall) Fields(s ...googleapi.Field) *RequestStatusRetrieveCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RequestStatusRetrieveCall) IfNoneMatch(entityTag string) *RequestStatusRetrieveCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RequestStatusRetrieveCall) Context(ctx context.Context) *RequestStatusRetrieveCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RequestStatusRetrieveCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RequestStatusRetrieveCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/requestStatus:retrieve")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "datamanager.requestStatus.retrieve", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "datamanager.requestStatus.retrieve" call.
// Any non-2xx status code is an error. Response headers are in either
// *RetrieveRequestStatusResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *RequestStatusRetrieveCall) Do(opts ...googleapi.CallOption) (*RetrieveRequestStatusResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &RetrieveRequestStatusResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "datamanager.requestStatus.retrieve", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
