// Copyright 2026 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package merchantapi provides access to the Merchant API.
//
// For product documentation, see: https://developers.google.com/merchant/api
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/merchantapi/products/v1"
//	...
//	ctx := context.Background()
//	merchantapiService, err := merchantapi.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	merchantapiService, err := merchantapi.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package merchantapi // import "google.golang.org/api/merchantapi/products/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "merchantapi:products_v1"
const apiName = "merchantapi"
const apiVersion = "products_v1"
const basePath = "https://merchantapi.googleapis.com/"
const basePathTemplate = "https://merchantapi.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://merchantapi.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Manage your product listings and accounts for Google Shopping
	ContentScope = "https://www.googleapis.com/auth/content"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/content",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Accounts = NewAccountsService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.ProductInputs = NewAccountsProductInputsService(s)
	rs.Products = NewAccountsProductsService(s)
	return rs
}

type AccountsService struct {
	s *Service

	ProductInputs *AccountsProductInputsService

	Products *AccountsProductsService
}

func NewAccountsProductInputsService(s *Service) *AccountsProductInputsService {
	rs := &AccountsProductInputsService{s: s}
	return rs
}

type AccountsProductInputsService struct {
	s *Service
}

func NewAccountsProductsService(s *Service) *AccountsProductsService {
	rs := &AccountsProductsService{s: s}
	return rs
}

type AccountsProductsService struct {
	s *Service
}

// AutomatedDiscounts: Information regarding Automated Discounts.
type AutomatedDiscounts struct {
	// GadPrice: The current sale price for products with a price optimized using
	// Google Automated Discounts (GAD). Absent if the information about the
	// GAD_price of the product is not available.
	GadPrice *Price `json:"gadPrice,omitempty"`
	// PriorPrice: The price prior to the application of the first price reduction.
	// Absent if the information about the prior price of the product is not
	// available.
	PriorPrice *Price `json:"priorPrice,omitempty"`
	// PriorPriceProgressive: The price prior to the application of consecutive
	// price reductions. Absent if the information about the prior price of the
	// product is not available.
	PriorPriceProgressive *Price `json:"priorPriceProgressive,omitempty"`
	// ForceSendFields is a list of field names (e.g. "GadPrice") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GadPrice") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AutomatedDiscounts) MarshalJSON() ([]byte, error) {
	type NoMethod AutomatedDiscounts
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CarrierShipping: Carrier-based shipping configuration. Allows for setting
// shipping speed or shipping cost based on a carrier's provided info.
type CarrierShipping struct {
	// CarrierPrice: Selected carrier to calculate the shipping price from. Select
	// a carrier from the available carriers list
	// (https://support.google.com/merchants/answer/15449142#Supported), for
	// example `AUSTRALIA_POST_REGULAR`. Price will be calculated by this selected
	// carrier, the location expressed in originPostalCode, along with the user
	// location to determine the accurate shipping price. Carrier is represented by
	// a carrier service name or a carrier service ID. Cannot be set together with
	// flatPrice.
	//
	// Possible values:
	//   "CARRIER_PRICE_OPTION_UNSPECIFIED" - Carrier price option is unspecified.
	//   "AUSTRALIA_POST_REGULAR" - Australia Post Regular shipping service.
	//   "AUSTRALIA_POST_EXPRESS" - Australia Post Express shipping service.
	//   "AUSTRALIA_POST_REGULAR_S" - Australia Post Regular Small shipping
	// service.
	//   "AUSTRALIA_POST_REGULAR_M" - Australia Post Regular Medium shipping
	// service.
	//   "AUSTRALIA_POST_REGULAR_L" - Australia Post Regular Large shipping
	// service.
	//   "AUSTRALIA_POST_REGULAR_XL" - Australia Post Regular XL shipping service.
	//   "AUSTRALIA_POST_EXPRESS_S" - Australia Post Express Small shipping
	// service.
	//   "AUSTRALIA_POST_EXPRESS_M" - Australia Post Express Medium shipping
	// service.
	//   "AUSTRALIA_POST_EXPRESS_L" - Australia Post Express Large shipping
	// service.
	//   "AUSTRALIA_POST_EXPRESS_XL" - Australia Post Express XL shipping service.
	//   "TNT_ROAD_EXPRESS" - TNT Road Express shipping service.
	//   "TNT_OVERNIGHT_EXPRESS" - TNT Overnight Express shipping service.
	//   "TOLL_ROAD_DELIVERY" - Toll Road Delivery shipping service.
	//   "TOLL_OVERNIGHT_PRIORITY" - Toll Overnight Priority shipping service.
	//   "DHL_PAKET" - DHL Paket shipping service.
	//   "DHL_PACKCHEN" - DHL Packchen shipping service.
	//   "DPD_EXPRESS_12" - DPD Express 12 shipping service.
	//   "DPD_EXPRESS" - DPD Express shipping service.
	//   "DPD_CLASSIC_PARCEL" - DPD Classic Parcel shipping service.
	//   "HERMES_PACKCHEN" - Hermes Packchen shipping service.
	//   "HERMES_PAKETKLASSE_S" - Hermes Paketklasse S shipping service.
	//   "HERMES_PAKETKLASSE_M" - Hermes Paketklasse M shipping service.
	//   "HERMES_PAKETKLASSE_L" - Hermes Paketklasse L shipping service.
	//   "UPS_EXPRESS" - UPS Express shipping service.
	//   "UPS_EXPRESS_SAVER" - UPS Express Saver shipping service.
	//   "UPS_EXPRESS_STANDARD" - UPS Express Standard shipping service.
	//   "DHL_EXPRESS" - DHL Express shipping service.
	//   "DHL_EXPRESS_12" - DHL Express 12 shipping service.
	//   "DPD_NEXT_DAY" - DPD Next Day shipping service.
	//   "DPD_STANDARD_NEXT_DAY" - DPD Standard Next Day shipping service.
	//   "DPD_STANDARD_TWO_DAY" - DPD Standard Two Day shipping service.
	//   "RMG_1ST_CLASS_SMALL" - RMG 1st Class Small shipping service.
	//   "RMG_1ST_CLASS_MEDIUM" - RMG 1st Class Medium shipping service.
	//   "RMG_2ND_CLASS_SMALL" - RMG 2nd Class Small shipping service.
	//   "RMG_2ND_CLASS_MEDIUM" - RMG 2nd Class Medium shipping service.
	//   "TNT_EXPRESS" - TNT Express shipping service.
	//   "TNT_EXPRESS_10" - TNT Express 10 shipping service.
	//   "TNT_EXPRESS_12" - TNT Express 12 shipping service.
	//   "YODEL_B2C_48HR" - Yodel B2C 48HR shipping service.
	//   "YODEL_B2C_72HR" - Yodel B2C 72HR shipping service.
	//   "YODEL_B2C_PACKET" - Yodel B2C Packet shipping service.
	//   "FEDEX_GROUND" - FedEx Ground shipping service.
	//   "FEDEX_HOME_DELIVERY" - FedEx Home Delivery shipping service.
	//   "FEDEX_EXPRESS_SAVER" - FedEx Express Saver shipping service.
	//   "FEDEX_FIRST_OVERNIGHT" - FedEx First Overnight shipping service.
	//   "FEDEX_PRIORITY_OVERNIGHT" - FedEx Priority Overnight shipping service.
	//   "FEDEX_STANDARD_OVERNIGHT" - FedEx Standard Overnight shipping service.
	//   "FEDEX_2DAY" - FedEx 2Day shipping service.
	//   "UPS_STANDARD" - UPS Standard shipping service.
	//   "UPS_2ND_DAY_AIR" - UPS 2nd Day Air shipping service.
	//   "UPS_2ND_DAY_AM" - UPS 2nd Day AM shipping service.
	//   "UPS_3_DAY_SELECT" - UPS 3 Day Select shipping service.
	//   "UPS_GROUND" - UPS Ground shipping service.
	//   "UPS_NEXT_DAY_AIR" - UPS Next Day Air shipping service.
	//   "UPS_NEXT_DAY_AIR_EARLY_AM" - UPS Next Day Air Early AM shipping service.
	//   "UPS_NEXT_DAY_AIR_SAVER" - UPS Next Day Air Saver shipping service.
	//   "USPS_PRIORITY_MAIL_EXPRESS" - USPS Priority Mail Express shipping
	// service.
	//   "USPS_MEDIA_MAIL" - USPS Media Mail shipping service.
	//   "USPS_GROUND_ADVANTAGE_RETAIL" - USPS Ground Advantage Retail shipping
	// service.
	//   "USPS_PRIORITY_MAIL" - USPS Priority Mail shipping service.
	//   "USPS_GROUND_ADVANTAGE_COMMERCIAL" - USPS Ground Advantage Commercial
	// shipping service.
	CarrierPrice string `json:"carrierPrice,omitempty"`
	// CarrierPriceFlatAdjustment: A flat adjustment on the carrier price. Can be
	// either positive or negative. Cannot be zero. Requires `carrier_price` to be
	// present. Cannot be set together with flatPrice and
	// carrierPricePercentageAdjustment.
	CarrierPriceFlatAdjustment *Price `json:"carrierPriceFlatAdjustment,omitempty"`
	// CarrierPricePercentageAdjustment: A percentual adjustment on the carrier
	// price. Can be either positive or negative. Cannot be zero. Requires
	// `carrier_price` to be present. Cannot be set together with flatPrice and
	// carrierPriceFlatAdjustment.
	CarrierPricePercentageAdjustment float64 `json:"carrierPricePercentageAdjustment,omitempty"`
	// CarrierTransitTime: Selected carrier to calculate the shipping speed from.
	// Select a carrier from the available carriers list
	// (https://support.google.com/merchants/answer/15449142#Supported), for
	// example `AUSTRALIA_POST_REGULAR`. Speed will be calculated by this selected
	// carrier, the location expressed in originPostalCode, along with the user
	// location to determine the accurate delivery speed. Carrier is represented by
	// a carrier service name or a carrier service ID. Cannot be set together with
	// fixedMaxTransitTime or fixedMinTransitTime.
	//
	// Possible values:
	//   "CARRIER_TRANSIT_TIME_OPTION_UNSPECIFIED" - Carrier transit time option is
	// unspecified.
	//   "DHL_PAKET" - DHL Paket shipping service.
	//   "DHL_PACKCHEN" - DHL Packchen shipping service.
	//   "DHL_EXPRESSEASY" - DHL Express Easy shipping service.
	//   "DPD_EXPRESS" - DPD Express shipping service.
	//   "DPD_CLASSIC_PARCEL" - DPD Classic Parcel shipping service.
	//   "HERMES_HAUSTUR" - Hermes Haustur shipping service.
	//   "HERMES_PAKETSHOP" - Hermes Paketshop shipping service.
	//   "GLS_BUSINESS" - GLS Business shipping service.
	//   "GLS_EXPRESS" - GLS Express shipping service.
	//   "GLS_PRIVATE" - GLS Private shipping service.
	//   "COLISSIMO_DOMICILE" - Colissimo Domicile shipping service.
	//   "DHL_EXPRESS_12AM" - DHL Express 12 AM shipping service.
	//   "DHL_EXPRESS_9AM" - DHL Express 9 AM shipping service.
	//   "GEODIS_EXPRESS" - GEODIS Express shipping service.
	//   "GEODIS_PACK_30" - GEODIS Pack 30 shipping service.
	//   "GEODIS_SAME_DAY" - GEODIS Same Day shipping service.
	//   "GEODIS_TOP_24" - GEODIS Top 24 shipping service.
	//   "TNT_ESSENTIEL_24H" - TNT Essentiel 24H shipping service.
	//   "TNT_ESSENTIEL_FLEXIBILITE" - TNT Essentiel Flexibilite shipping service.
	//   "FEDEX_GROUND" - FedEx Ground shipping service.
	//   "FEDEX_HOME_DELIVERY" - FedEx Home Delivery shipping service.
	//   "FEDEX_EXPRESS_SAVER" - FedEx Express Saver shipping service.
	//   "FEDEX_FIRST_OVERNIGHT" - FedEx First Overnight shipping service.
	//   "FEDEX_PRIORITY_OVERNIGHT" - FedEx Priority Overnight shipping service.
	//   "FEDEX_STANDARD_OVERNIGHT" - FedEx Standard Overnight shipping service.
	//   "FEDEX_2DAY" - FedEx 2Day shipping service.
	//   "UPS_2ND_DAY_AIR" - UPS 2nd Day Air shipping service.
	//   "UPS_2ND_DAY_AM" - UPS 2nd Day AM shipping service.
	//   "UPS_3_DAY_SELECT" - UPS 3 Day Select shipping service.
	//   "UPS_GROUND" - UPS Ground shipping service.
	//   "UPS_NEXT_DAY_AIR" - UPS Next Day Air shipping service.
	//   "UPS_NEXT_DAY_AIR_EARLY_AM" - UPS Next Day Air Early AM shipping service.
	//   "UPS_NEXT_DAY_AIR_SAVER" - UPS Next Day Air Saver shipping service.
	//   "USPS_PRIORITY_MAIL_EXPRESS" - USPS Priority Mail Express shipping
	// service.
	//   "USPS_MEDIA_MAIL" - USPS Media Mail shipping service.
	//   "USPS_GROUND_ADVANTAGE_RETAIL" - USPS Ground Advantage Retail shipping
	// service.
	//   "USPS_PRIORITY_MAIL" - USPS Priority Mail shipping service.
	//   "USPS_GROUND_ADVANTAGE_COMMERCIAL" - USPS Ground Advantage Commercial
	// shipping service.
	//   "USPS_FIRST_CLASS_MAIL" - USPS First Class Mail shipping service.
	CarrierTransitTime string `json:"carrierTransitTime,omitempty"`
	// Country: The CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) of the
	// country to which an item will ship.
	Country string `json:"country,omitempty"`
	// FixedMaxTransitTime: Maximum transit time (inclusive) between when the order
	// has shipped and when it is delivered in business days. 0 means that the
	// order is delivered on the same day as it ships. Needs to be provided
	// together with maxHandlingTime. Cannot be set if carrierTransitTime is
	// present.
	FixedMaxTransitTime int64 `json:"fixedMaxTransitTime,omitempty,string"`
	// FixedMinTransitTime: Minimum transit time (inclusive) between when the order
	// has shipped and when it is delivered in business days. 0 means that the
	// order is delivered on the same day as it ships. fixedMinTransitTime can only
	// be set if fixedMaxTransitTime is set. Cannot be set if carrierTransitTime is
	// present.
	FixedMinTransitTime int64 `json:"fixedMinTransitTime,omitempty,string"`
	// FlatPrice: Fixed shipping price, represented as a number with currency.
	// Cannot be set together with carrierPrice or its adjustments
	// (carrierPriceFlatAdjustment, carrierPricePercentageAdjustment).
	FlatPrice *Price `json:"flatPrice,omitempty"`
	// MaxHandlingTime: Maximum handling time (inclusive) between when the order is
	// received and shipped in business days. 0 means that the order is shipped on
	// the same day as it is received if it happens before the cut-off time. Both
	// maxHandlingTime and fixedMaxTransitTime or carrierTransitTime are required
	// if providing shipping speeds.
	MaxHandlingTime int64 `json:"maxHandlingTime,omitempty,string"`
	// MinHandlingTime: Minimum handling time (inclusive) between when the order is
	// received and shipped in business days. 0 means that the order is shipped on
	// the same day as it is received if it happens before the cut-off time.
	// minHandlingTime can only be set if maxHandlingTime is also set.
	MinHandlingTime int64 `json:"minHandlingTime,omitempty,string"`
	// OriginPostalCode: The source location postal code from which this offer
	// ships. Represented only by a full-length postal code.
	OriginPostalCode string `json:"originPostalCode,omitempty"`
	// PostalCode: The postal code range that the shipping rate applies to,
	// represented by a postal code (eg. `94043`), a postal code prefix followed by
	// a * wildcard (eg. `94*`), a range between two postal codes (eg.
	// `94043-98033`) or two postal code prefixes of equal length (eg. `94*-98*`).
	PostalCode string `json:"postalCode,omitempty"`
	// Region: The geographic region to which a shipping rate applies. See region
	// (https://support.google.com/merchants/answer/6324484) for more information.
	Region string `json:"region,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CarrierPrice") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CarrierPrice") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CarrierShipping) MarshalJSON() ([]byte, error) {
	type NoMethod CarrierShipping
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *CarrierShipping) UnmarshalJSON(data []byte) error {
	type NoMethod CarrierShipping
	var s1 struct {
		CarrierPricePercentageAdjustment gensupport.JSONFloat64 `json:"carrierPricePercentageAdjustment"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.CarrierPricePercentageAdjustment = float64(s1.CarrierPricePercentageAdjustment)
	return nil
}

// CloudExportAdditionalProperties: Product property for the Cloud Retail API.
// For example, properties for a TV product could be "Screen-Resolution" or
// "Screen-Size".
type CloudExportAdditionalProperties struct {
	// BoolValue: Boolean value of the given property. For example for a TV
	// product, "True" or "False" if the screen is UHD.
	BoolValue bool `json:"boolValue,omitempty"`
	// FloatValue: Float values of the given property. For example for a TV product
	// 1.2345. Maximum repeatedness of this value is 400. Values are stored in an
	// arbitrary but consistent order.
	FloatValue []float64 `json:"floatValue,omitempty"`
	// IntValue: Integer values of the given property. For example, 1080 for a TV
	// product's Screen Resolution. Maximum repeatedness of this value is 400.
	// Values are stored in an arbitrary but consistent order.
	IntValue googleapi.Int64s `json:"intValue,omitempty"`
	// MaxValue: Maximum float value of the given property. For example for a TV
	// product 100.00.
	MaxValue float64 `json:"maxValue,omitempty"`
	// MinValue: Minimum float value of the given property. For example for a TV
	// product 1.00.
	MinValue float64 `json:"minValue,omitempty"`
	// PropertyName: Name of the given property. For example, "Screen-Resolution"
	// for a TV product. Maximum string size is 256 characters.
	PropertyName string `json:"propertyName,omitempty"`
	// TextValue: Text value of the given property. For example, "8K(UHD)" could be
	// a text value for a TV product. Maximum repeatedness of this value is 400.
	// Values are stored in an arbitrary but consistent order. Maximum string size
	// is 256 characters.
	TextValue []string `json:"textValue,omitempty"`
	// UnitCode: Unit of the given property. For example, "Pixels" for a TV
	// product. Maximum string size is 256B.
	UnitCode string `json:"unitCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BoolValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BoolValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CloudExportAdditionalProperties) MarshalJSON() ([]byte, error) {
	type NoMethod CloudExportAdditionalProperties
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *CloudExportAdditionalProperties) UnmarshalJSON(data []byte) error {
	type NoMethod CloudExportAdditionalProperties
	var s1 struct {
		FloatValue []gensupport.JSONFloat64 `json:"floatValue"`
		MaxValue   gensupport.JSONFloat64   `json:"maxValue"`
		MinValue   gensupport.JSONFloat64   `json:"minValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.FloatValue = make([]float64, len(s1.FloatValue))
	for i := range s1.FloatValue {
		s.FloatValue[i] = float64(s1.FloatValue[i])
	}
	s.MaxValue = float64(s1.MaxValue)
	s.MinValue = float64(s1.MinValue)
	return nil
}

// CustomAttribute: A message that represents custom attributes. Exactly one of
// `value` or `group_values` must not be empty.
type CustomAttribute struct {
	// GroupValues: Subattributes within this attribute group. If `group_values` is
	// not empty, `value` must be empty.
	GroupValues []*CustomAttribute `json:"groupValues,omitempty"`
	// Name: The name of the attribute.
	Name string `json:"name,omitempty"`
	// Value: The value of the attribute. If `value` is not empty, `group_values`
	// must be empty.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "GroupValues") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GroupValues") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomAttribute) MarshalJSON() ([]byte, error) {
	type NoMethod CustomAttribute
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DestinationStatus: The destination status of the product status.
type DestinationStatus struct {
	// ApprovedCountries: List of country codes (ISO 3166-1 alpha-2) where the
	// offer is approved.
	ApprovedCountries []string `json:"approvedCountries,omitempty"`
	// DisapprovedCountries: List of country codes (ISO 3166-1 alpha-2) where the
	// offer is disapproved.
	DisapprovedCountries []string `json:"disapprovedCountries,omitempty"`
	// PendingCountries: List of country codes (ISO 3166-1 alpha-2) where the offer
	// is pending approval.
	PendingCountries []string `json:"pendingCountries,omitempty"`
	// ReportingContext: The name of the reporting context.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_AFFILIATE" - [Youtube
	// Affiliate](https://support.google.com/youtube/answer/13376398).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - YouTube Checkout .
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApprovedCountries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApprovedCountries") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DestinationStatus) MarshalJSON() ([]byte, error) {
	type NoMethod DestinationStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// FreeShippingThreshold: Conditions to be met for a product to have free
// shipping.
type FreeShippingThreshold struct {
	// Country: The CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) of the
	// country to which an item will ship.
	Country string `json:"country,omitempty"`
	// PriceThreshold: The minimum product price for the shipping cost to become
	// free. Represented as a number.
	PriceThreshold *Price `json:"priceThreshold,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Country") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Country") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FreeShippingThreshold) MarshalJSON() ([]byte, error) {
	type NoMethod FreeShippingThreshold
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HandlingCutoffTime: Configuration for offer or offer-country level shipping
// handling cutoff time.
type HandlingCutoffTime struct {
	// Country: The CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) of the
	// country to which the handling cutoff time applies.
	Country string `json:"country,omitempty"`
	// CutoffTime: The handling cutoff time until which an order has to be placed
	// to be processed in the same day. This is a string in format of HHMM (e.g.
	// `1530`) for 3:30 PM. If not configured, the cutoff time will be defaulted to
	// 8AM PST.
	CutoffTime string `json:"cutoffTime,omitempty"`
	// CutoffTimezone: Timezone identifier
	// (https://developers.google.com/adwords/api/docs/appendix/codes-formats#timezone-ids)
	// For example 'Europe/Zurich'. If not set, the shipping destination timezone
	// will be used.
	CutoffTimezone string `json:"cutoffTimezone,omitempty"`
	// DisableDeliveryAfterCutoff: This field only applies to same-day delivery. If
	// true, prevents next-day delivery from being shown for this offer after the
	// cutoff time. This field only applies to same-day delivery offers, for
	// merchants who want to explicitly disable it.
	DisableDeliveryAfterCutoff bool `json:"disableDeliveryAfterCutoff,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Country") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Country") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HandlingCutoffTime) MarshalJSON() ([]byte, error) {
	type NoMethod HandlingCutoffTime
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Interval: Represents a time interval, encoded as a Timestamp start
// (inclusive) and a Timestamp end (exclusive). The start must be less than or
// equal to the end. When the start equals the end, the interval is empty
// (matches no time). When both start and end are unspecified, the interval
// matches any time.
type Interval struct {
	// EndTime: Optional. Exclusive end of the interval. If specified, a Timestamp
	// matching this interval will have to be before the end.
	EndTime string `json:"endTime,omitempty"`
	// StartTime: Optional. Inclusive start of the interval. If specified, a
	// Timestamp matching this interval will have to be the same or after the
	// start.
	StartTime string `json:"startTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "EndTime") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "EndTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Interval) MarshalJSON() ([]byte, error) {
	type NoMethod Interval
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ItemLevelIssue: The ItemLevelIssue of the product status.
type ItemLevelIssue struct {
	// ApplicableCountries: List of country codes (ISO 3166-1 alpha-2) where issue
	// applies to the offer.
	ApplicableCountries []string `json:"applicableCountries,omitempty"`
	// Attribute: The attribute's name, if the issue is caused by a single
	// attribute.
	Attribute string `json:"attribute,omitempty"`
	// Code: The error code of the issue.
	Code string `json:"code,omitempty"`
	// Description: A short issue description in English.
	Description string `json:"description,omitempty"`
	// Detail: A detailed issue description in English.
	Detail string `json:"detail,omitempty"`
	// Documentation: The URL of a web page to help with resolving this issue.
	Documentation string `json:"documentation,omitempty"`
	// ReportingContext: The reporting context the issue applies to.
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_AFFILIATE" - [Youtube
	// Affiliate](https://support.google.com/youtube/answer/13376398).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - YouTube Checkout .
	ReportingContext string `json:"reportingContext,omitempty"`
	// Resolution: Whether the issue can be resolved by the business.
	Resolution string `json:"resolution,omitempty"`
	// Severity: How this issue affects serving of the offer.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Not specified.
	//   "NOT_IMPACTED" - This issue represents a warning and does not have a
	// direct affect on the product.
	//   "DEMOTED" - The product is demoted and most likely have limited
	// performance in search results
	//   "DISAPPROVED" - Issue disapproves the product.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApplicableCountries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApplicableCountries") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ItemLevelIssue) MarshalJSON() ([]byte, error) {
	type NoMethod ItemLevelIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListProductsResponse: Response message for the ListProducts method.
type ListProductsResponse struct {
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Products: The processed products from the specified account. These are your
	// processed products after applying rules and supplemental data sources.
	Products []*Product `json:"products,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListProductsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListProductsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LoyaltyPoints: A message that represents loyalty points.
type LoyaltyPoints struct {
	// Name: Name of loyalty points program. It is recommended to limit the name to
	// 12 full-width characters or 24 Roman characters.
	Name string `json:"name,omitempty"`
	// PointsValue: The retailer's loyalty points in absolute value.
	PointsValue int64 `json:"pointsValue,omitempty,string"`
	// Ratio: The ratio of a point when converted to currency. Google assumes
	// currency based on Merchant Center settings. If ratio is left out, it
	// defaults to 1.0.
	Ratio float64 `json:"ratio,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Name") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Name") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LoyaltyPoints) MarshalJSON() ([]byte, error) {
	type NoMethod LoyaltyPoints
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *LoyaltyPoints) UnmarshalJSON(data []byte) error {
	type NoMethod LoyaltyPoints
	var s1 struct {
		Ratio gensupport.JSONFloat64 `json:"ratio"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Ratio = float64(s1.Ratio)
	return nil
}

// LoyaltyProgram: LINT.IfChange(LoyaltyProgram) A message that represents
// loyalty program.
type LoyaltyProgram struct {
	// CashbackForFutureUse: The cashback that can be used for future purchases.
	CashbackForFutureUse *Price `json:"cashbackForFutureUse,omitempty"`
	// LoyaltyPoints: The amount of loyalty points earned on a purchase.
	LoyaltyPoints int64 `json:"loyaltyPoints,omitempty,string"`
	// MemberPriceEffectiveDate: A date range during which the item is eligible for
	// member price. If not specified, the member price is always applicable. The
	// date range is represented by a pair of ISO 8601 dates separated by a space,
	// comma, or slash.
	MemberPriceEffectiveDate *Interval `json:"memberPriceEffectiveDate,omitempty"`
	// Price: The price for members of the given tier, that is, the instant
	// discount price. Must be smaller or equal to the regular price.
	Price *Price `json:"price,omitempty"`
	// ProgramLabel: The label of the loyalty program. This is an internal label
	// that uniquely identifies the relationship between a business entity and a
	// loyalty program entity. The label must be provided so that the system can
	// associate the assets below (for example, price and points) with a business.
	// The corresponding program must be linked to the Merchant Center account.
	ProgramLabel string `json:"programLabel,omitempty"`
	// ShippingLabel: The label of the shipping benefit. If the field has value,
	// this offer has loyalty shipping benefit. If the field value isn't provided,
	// the item is not eligible for loyalty shipping for the given loyalty tier.
	ShippingLabel string `json:"shippingLabel,omitempty"`
	// TierLabel: The label of the tier within the loyalty program. Must match one
	// of the labels within the program.
	TierLabel string `json:"tierLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CashbackForFutureUse") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CashbackForFutureUse") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LoyaltyProgram) MarshalJSON() ([]byte, error) {
	type NoMethod LoyaltyProgram
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Price: The price represented as a number and currency.
type Price struct {
	// AmountMicros: The price represented as a number in micros (1 million micros
	// is an equivalent to one's currency standard unit, for example, 1 USD =
	// 1000000 micros).
	AmountMicros int64 `json:"amountMicros,omitempty,string"`
	// CurrencyCode: The currency of the price using three-letter acronyms
	// according to ISO 4217 (http://en.wikipedia.org/wiki/ISO_4217).
	CurrencyCode string `json:"currencyCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmountMicros") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmountMicros") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Price) MarshalJSON() ([]byte, error) {
	type NoMethod Price
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Product: The processed product, built from multiple product inputs after
// applying rules and supplemental data sources. This processed product matches
// what is shown in your Merchant Center account. Each product is built from
// exactly one primary data source product input, and multiple supplemental
// data source inputs. After inserting, updating, or deleting a product input,
// it may take several minutes before the updated processed product can be
// retrieved. All fields in the processed product and its sub-messages match
// the name of their corresponding attribute in the Product data specification
// (https://support.google.com/merchants/answer/7052112) with some exceptions.
type Product struct {
	// AutomatedDiscounts: Output only. The automated discounts information for the
	// product.
	AutomatedDiscounts *AutomatedDiscounts `json:"automatedDiscounts,omitempty"`
	// ContentLanguage: Output only. The two-letter ISO 639-1
	// (http://en.wikipedia.org/wiki/ISO_639-1) language code for the product.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// CustomAttributes: Output only. A list of custom (merchant-provided)
	// attributes. It can also be used to submit any attribute of the data
	// specification in its generic form (for example, `{ "name": "size type",
	// "value": "regular" }`). This is useful for submitting attributes not
	// explicitly exposed by the API, such as additional attributes used for Buy on
	// Google.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// DataSource: Output only. The primary data source of the product.
	DataSource string `json:"dataSource,omitempty"`
	// FeedLabel: Output only. The feed label lets you categorize and identify your
	// products. The maximum allowed characters is 20 and the supported characters
	// are`A-Z`, `0-9`, hyphen and underscore. The feed label must not include any
	// spaces. For more information, see Using feed labels
	// (//support.google.com/merchants/answer/14994087)
	FeedLabel string `json:"feedLabel,omitempty"`
	// LegacyLocal: Output only. Determines whether the product is **only**
	// targeting local destinations and whether the product name should be
	// distinguished with a `local~` prefix. For example,
	// `accounts/123/products/local~en~US~sku123`.
	LegacyLocal bool `json:"legacyLocal,omitempty"`
	// Name: The name of the product. Format:
	// `accounts/{account}/products/{product}` where the last section `product`
	// consists of: `content_language~feed_label~offer_id` example for product name
	// is `accounts/123/products/en~US~sku123`. A legacy local product name would
	// be `accounts/123/products/local~en~US~sku123`. Note: For calls to the v1beta
	// version, the `product` section consists of:
	// `channel~content_language~feed_label~offer_id`, for example:
	// `accounts/123/products/online~en~US~sku123`.
	Name string `json:"name,omitempty"`
	// OfferId: Output only. Your unique identifier for the product. This is the
	// same for the product input and processed product. Leading and trailing
	// whitespaces are stripped and multiple whitespaces are replaced by a single
	// whitespace upon submission. See the product data specification
	// (https://support.google.com/merchants/answer/188494#id) for details.
	OfferId string `json:"offerId,omitempty"`
	// ProductAttributes: Output only. A list of strongly-typed product attributes.
	ProductAttributes *ProductAttributes `json:"productAttributes,omitempty"`
	// ProductStatus: Output only. The status of a product, data validation issues,
	// that is, information about a product computed asynchronously.
	ProductStatus *ProductStatus `json:"productStatus,omitempty"`
	// VersionNumber: Output only. Represents the existing version (freshness) of
	// the product, which can be used to preserve the right order when multiple
	// updates are done at the same time. If set, the insertion is prevented when
	// version number is lower than the current version number of the existing
	// product. Re-insertion (for example, product refresh after 30 days) can be
	// performed with the current `version_number`. Only supported for insertions
	// into primary data sources. If the operation is prevented, the aborted
	// exception will be thrown.
	VersionNumber int64 `json:"versionNumber,omitempty,string"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AutomatedDiscounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AutomatedDiscounts") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Product) MarshalJSON() ([]byte, error) {
	type NoMethod Product
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductAttributes: Product attributes.
type ProductAttributes struct {
	// AdditionalImageLinks: Additional URLs of images of the item.
	AdditionalImageLinks []string `json:"additionalImageLinks,omitempty"`
	// AdsGrouping: Used to group items in an arbitrary way. Only for CPA%,
	// discouraged otherwise. For more information, see Display ads attribute
	// (https://support.google.com/merchants/answer/6069387).
	AdsGrouping string `json:"adsGrouping,omitempty"`
	// AdsLabels: Similar to ads_grouping, but only works on CPC.
	AdsLabels []string `json:"adsLabels,omitempty"`
	// AdsRedirect: Allows advertisers to override the item URL when the product is
	// shown within the context of Product ads.
	AdsRedirect string `json:"adsRedirect,omitempty"`
	// Adult: Set to true if the item is targeted towards adults.
	Adult bool `json:"adult,omitempty"`
	// AgeGroup: Target age group
	// (https://support.google.com/merchants/answer/6324463) of the item.
	//
	// Possible values:
	//   "AGE_GROUP_UNSPECIFIED" - Age group is not specified.
	//   "ADULT" - Teens or older.
	//   "KIDS" - 5-13 years old.
	//   "TODDLER" - 1-5 years old.
	//   "INFANT" - 3-12 months old.
	//   "NEWBORN" - 0-3 months old.
	AgeGroup string `json:"ageGroup,omitempty"`
	// AutoPricingMinPrice: A safeguard in the [automated discounts]
	// (https://support.google.com/merchants/answer/10295759) and "Dynamic
	// Promotions" (https://support.google.com/merchants/answer/13949249) projects,
	// ensuring that discounts on business offers do not fall below this value,
	// thereby preserving the offer's value and profitability.
	AutoPricingMinPrice *Price `json:"autoPricingMinPrice,omitempty"`
	// Availability: Availability
	// (https://support.google.com/merchants/answer/6324448) status of the item.
	//
	// Possible values:
	//   "AVAILABILITY_UNSPECIFIED" - Availability is not specified.
	//   "IN_STOCK" - In stock.
	//   "OUT_OF_STOCK" - Out of stock.
	//   "PREORDER" - Pre-order.
	//   "LIMITED_AVAILABILITY" - Limited availability.
	//   "BACKORDER" - Backorder.
	Availability string `json:"availability,omitempty"`
	// AvailabilityDate: The day a pre-ordered product becomes available for
	// delivery, in ISO 8601 (http://en.wikipedia.org/wiki/ISO_8601) format.
	AvailabilityDate string `json:"availabilityDate,omitempty"`
	// Brand: Brand (https://support.google.com/merchants/answer/6324351) of the
	// item. For example, "Google".
	Brand string `json:"brand,omitempty"`
	// CanonicalLink: URL for the canonical version of your item's landing page.
	CanonicalLink string `json:"canonicalLink,omitempty"`
	// CarrierShipping: Rules for carrier-based shipping.
	CarrierShipping []*CarrierShipping `json:"carrierShipping,omitempty"`
	// Certifications: Product Certifications, for example for energy efficiency
	// labeling of products recorded in the EU EPREL
	// (https://eprel.ec.europa.eu/screen/home) database. See the Help Center
	// (https://support.google.com/merchants/answer/13528839) article for more
	// information.
	Certifications []*ProductCertification `json:"certifications,omitempty"`
	// CloudExportAdditionalProperties: Extra fields to export to the Cloud Retail
	// program.
	CloudExportAdditionalProperties []*CloudExportAdditionalProperties `json:"cloudExportAdditionalProperties,omitempty"`
	// Color: Color (https://support.google.com/merchants/answer/6324487) of the
	// item. For example, "red".
	Color string `json:"color,omitempty"`
	// Condition: Condition (https://support.google.com/merchants/answer/6324469)
	// or state of the item.
	//
	// Possible values:
	//   "CONDITION_UNSPECIFIED" - Default value. This value is unused.
	//   "NEW" - Brand new, original, unopened packaging.
	//   "USED" - Previously used, original packaging opened or missing.
	//   "REFURBISHED" - Professionally restored to working order, comes with a
	// warranty, may or may not have the original packaging.
	Condition string `json:"condition,omitempty"`
	// CostOfGoodsSold: Cost of goods sold. Used for gross profit reporting.
	CostOfGoodsSold *Price `json:"costOfGoodsSold,omitempty"`
	// CustomLabel0: Custom label 0
	// (https://support.google.com/merchants/answer/6324473) for custom grouping of
	// items in a Shopping campaign.
	CustomLabel0 string `json:"customLabel0,omitempty"`
	// CustomLabel1: Custom label 1
	// (https://support.google.com/merchants/answer/6324473) for custom grouping of
	// items in a Shopping campaign.
	CustomLabel1 string `json:"customLabel1,omitempty"`
	// CustomLabel2: Custom label 2
	// (https://support.google.com/merchants/answer/6324473) for custom grouping of
	// items in a Shopping campaign.
	CustomLabel2 string `json:"customLabel2,omitempty"`
	// CustomLabel3: Custom label 3
	// (https://support.google.com/merchants/answer/6324473) for custom grouping of
	// items in a Shopping campaign.
	CustomLabel3 string `json:"customLabel3,omitempty"`
	// CustomLabel4: Custom label 4
	// (https://support.google.com/merchants/answer/6324473) for custom grouping of
	// items in a Shopping campaign.
	CustomLabel4 string `json:"customLabel4,omitempty"`
	// Description: Description of the item.
	Description string `json:"description,omitempty"`
	// DisclosureDate: The date time when an offer becomes visible in search
	// results across Google’s YouTube surfaces, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format. See Disclosure date
	// (https://support.google.com/merchants/answer/13034208) for more information.
	DisclosureDate string `json:"disclosureDate,omitempty"`
	// DisplayAdsId: An identifier for an item for dynamic remarketing campaigns.
	DisplayAdsId string `json:"displayAdsId,omitempty"`
	// DisplayAdsLink: URL directly to your item's landing page for dynamic
	// remarketing campaigns.
	DisplayAdsLink string `json:"displayAdsLink,omitempty"`
	// DisplayAdsSimilarIds: Advertiser-specified recommendations. For more
	// information, see Display ads attribute specification
	// (https://support.google.com/merchants/answer/6069387).
	DisplayAdsSimilarIds []string `json:"displayAdsSimilarIds,omitempty"`
	// DisplayAdsTitle: Title of an item for dynamic remarketing campaigns.
	DisplayAdsTitle string `json:"displayAdsTitle,omitempty"`
	// DisplayAdsValue: Offer margin for dynamic remarketing campaigns. For more
	// information, see Display ads attribute
	// (https://support.google.com/merchants/answer/6069387).
	DisplayAdsValue float64 `json:"displayAdsValue,omitempty"`
	// EnergyEfficiencyClass: The energy efficiency class
	// (https://support.google.com/merchants/answer/7562785) as defined in EU
	// directive 2010/30/EU.
	//
	// Possible values:
	//   "ENERGY_EFFICIENCY_CLASS_UNSPECIFIED" - The energy efficiency class is
	// unspecified.
	//   "APPP" - A+++.
	//   "APP" - A++.
	//   "AP" - A+.
	//   "A" - A.
	//   "B" - B.
	//   "C" - C.
	//   "D" - D.
	//   "E" - E.
	//   "F" - F.
	//   "G" - G.
	EnergyEfficiencyClass string `json:"energyEfficiencyClass,omitempty"`
	// ExcludedDestinations: The list of destinations to exclude for this target
	// (corresponds to unchecked check boxes in Merchant Center). For more
	// information, see Excluded destination
	// (https://support.google.com/merchants/answer/6324486). Note: We recommend
	// setting destinations on datasources level for most use cases. Use this field
	// within products to only setup exceptions.
	//
	// Possible values:
	//   "DESTINATION_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/google-ads/answer/2454022).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3057972).
	//   "FREE_LISTINGS" - [Free
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/12362804).
	//   "YOUTUBE_SHOPPING_CHECKOUT" - Youtube shopping checkout.
	//   "YOUTUBE_AFFILIATE" - [Youtube
	// Affiliate](https://support.google.com/youtube/answer/13376398).
	//   "FREE_VEHICLE_LISTINGS" - [Free vehicle
	// listings](https://support.google.com/merchants/answer/11189169).
	//   "VEHICLE_ADS" - [Vehicle
	// ads](https://support.google.com/merchants/answer/11189169).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	ExcludedDestinations []string `json:"excludedDestinations,omitempty"`
	// ExpirationDate: Date on which the item should expire, as specified upon
	// insertion, in ISO 8601 (http://en.wikipedia.org/wiki/ISO_8601) format. The
	// actual expiration date is exposed in `productstatuses` as
	// googleExpirationDate (https://support.google.com/merchants/answer/6324499)
	// and might be earlier if `expirationDate` is too far in the future.
	ExpirationDate string `json:"expirationDate,omitempty"`
	// ExternalSellerId: Required for multi-seller accounts. Use this attribute if
	// you're a marketplace uploading products for various sellers to your
	// multi-seller account.
	ExternalSellerId string `json:"externalSellerId,omitempty"`
	// FreeShippingThreshold: Conditions to be met for a product to have free
	// shipping.
	FreeShippingThreshold []*FreeShippingThreshold `json:"freeShippingThreshold,omitempty"`
	// Gender: Target gender (https://support.google.com/merchants/answer/6324479)
	// of the item.
	//
	// Possible values:
	//   "GENDER_UNSPECIFIED" - Gender is not specified.
	//   "MALE" - Male.
	//   "FEMALE" - Female.
	//   "UNISEX" - Unisex.
	Gender string `json:"gender,omitempty"`
	// GoogleProductCategory: Google's category of the item (see Google product
	// taxonomy (https://support.google.com/merchants/answer/1705911)). When
	// querying products, this field will contain the user provided value. There is
	// currently no way to get back the auto assigned google product categories
	// through the API.
	GoogleProductCategory string `json:"googleProductCategory,omitempty"`
	// Gtins: Global Trade Item Numbers (GTIN
	// (https://support.google.com/merchants/answer/188494#gtin)) of the item. You
	// can provide up to 10 GTINs.
	Gtins []string `json:"gtins,omitempty"`
	// HandlingCutoffTimes: The handling cutoff times for shipping.
	HandlingCutoffTimes []*HandlingCutoffTime `json:"handlingCutoffTimes,omitempty"`
	// IdentifierExists: Set this value to false when the item does not have unique
	// product identifiers appropriate to its category, such as GTIN, MPN, and
	// brand. Defaults to true, if not provided.
	IdentifierExists bool `json:"identifierExists,omitempty"`
	// ImageLink: URL of an image of the item.
	ImageLink string `json:"imageLink,omitempty"`
	// IncludedDestinations: The list of destinations to include for this target
	// (corresponds to checked check boxes in Merchant Center). Default
	// destinations are always included unless provided in `excludedDestinations`.
	// For more information, see Included destination
	// (https://support.google.com/merchants/answer/7501026). Note: We recommend
	// setting destinations on datasources level for most use cases. Use this field
	// within products to only setup exceptions.
	//
	// Possible values:
	//   "DESTINATION_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/google-ads/answer/2454022).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3057972).
	//   "FREE_LISTINGS" - [Free
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/12362804).
	//   "YOUTUBE_SHOPPING_CHECKOUT" - Youtube shopping checkout.
	//   "YOUTUBE_AFFILIATE" - [Youtube
	// Affiliate](https://support.google.com/youtube/answer/13376398).
	//   "FREE_VEHICLE_LISTINGS" - [Free vehicle
	// listings](https://support.google.com/merchants/answer/11189169).
	//   "VEHICLE_ADS" - [Vehicle
	// ads](https://support.google.com/merchants/answer/11189169).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	IncludedDestinations []string `json:"includedDestinations,omitempty"`
	// Installment: Number and amount of installments to pay for an item.
	Installment *ProductInstallment `json:"installment,omitempty"`
	// IsBundle: Whether the item is a business-defined sub-API. A [sub-API]
	// (https://support.google.com/merchants/answer/6324449) is a custom grouping
	// of different products sold by a business for a single price.
	IsBundle bool `json:"isBundle,omitempty"`
	// ItemGroupId: Shared identifier for all variants of the same product.
	ItemGroupId string `json:"itemGroupId,omitempty"`
	// LifestyleImageLinks: Additional URLs of lifestyle images of the item, used
	// to explicitly identify images that showcase your item in a real-world
	// context. See the Help Center article
	// (https://support.google.com/merchants/answer/9103186) for more information.
	LifestyleImageLinks []string `json:"lifestyleImageLinks,omitempty"`
	// Link: URL directly linking to your item's page on your online store.
	Link string `json:"link,omitempty"`
	// LinkTemplate: Link template
	// (https://support.google.com/merchants/answer/13871172) for business hosted
	// local storefront.
	LinkTemplate string `json:"linkTemplate,omitempty"`
	// LoyaltyPoints: Loyalty points that users receive after purchasing the item.
	// Japan only.
	LoyaltyPoints *LoyaltyPoints `json:"loyaltyPoints,omitempty"`
	// LoyaltyPrograms: A list of loyalty program information that is used to
	// surface loyalty benefits (for example, better pricing, points, etc) to the
	// user of this item.
	LoyaltyPrograms []*LoyaltyProgram `json:"loyaltyPrograms,omitempty"`
	// Material: The material (https://support.google.com/merchants/answer/6324410)
	// of which the item is made. For example, "Leather" or "Cotton".
	Material string `json:"material,omitempty"`
	// MaxEnergyEfficiencyClass: The energy efficiency class
	// (https://support.google.com/merchants/answer/7562785) as defined in EU
	// directive 2010/30/EU.
	//
	// Possible values:
	//   "ENERGY_EFFICIENCY_CLASS_UNSPECIFIED" - The energy efficiency class is
	// unspecified.
	//   "APPP" - A+++.
	//   "APP" - A++.
	//   "AP" - A+.
	//   "A" - A.
	//   "B" - B.
	//   "C" - C.
	//   "D" - D.
	//   "E" - E.
	//   "F" - F.
	//   "G" - G.
	MaxEnergyEfficiencyClass string `json:"maxEnergyEfficiencyClass,omitempty"`
	// MaxHandlingTime: Maximal product handling time (in business days).
	MaxHandlingTime int64 `json:"maxHandlingTime,omitempty,string"`
	// MaximumRetailPrice: Maximum retail price (MRP) of the item. Applicable to
	// India only.
	MaximumRetailPrice *Price `json:"maximumRetailPrice,omitempty"`
	// MinEnergyEfficiencyClass: The energy efficiency class
	// (https://support.google.com/merchants/answer/7562785) as defined in EU
	// directive 2010/30/EU.
	//
	// Possible values:
	//   "ENERGY_EFFICIENCY_CLASS_UNSPECIFIED" - The energy efficiency class is
	// unspecified.
	//   "APPP" - A+++.
	//   "APP" - A++.
	//   "AP" - A+.
	//   "A" - A.
	//   "B" - B.
	//   "C" - C.
	//   "D" - D.
	//   "E" - E.
	//   "F" - F.
	//   "G" - G.
	MinEnergyEfficiencyClass string `json:"minEnergyEfficiencyClass,omitempty"`
	// MinHandlingTime: Minimal product handling time (in business days).
	MinHandlingTime int64 `json:"minHandlingTime,omitempty,string"`
	// MobileLink: URL for the mobile-optimized version of your item's landing
	// page.
	MobileLink string `json:"mobileLink,omitempty"`
	// MobileLinkTemplate: Link template
	// (https://support.google.com/merchants/answer/13870216) for business hosted
	// local storefront optimized for mobile devices.
	MobileLinkTemplate string `json:"mobileLinkTemplate,omitempty"`
	// Mpn: Manufacturer Part Number (MPN
	// (https://support.google.com/merchants/answer/188494#mpn)) of the item.
	Mpn string `json:"mpn,omitempty"`
	// Multipack: The number of identical products in a business-defined multipack.
	Multipack int64 `json:"multipack,omitempty,string"`
	// Pattern: The item's pattern
	// (https://support.google.com/merchants/answer/6324483). For example, polka
	// dots.
	Pattern string `json:"pattern,omitempty"`
	// Pause: Publication of this item will be temporarily paused
	// (https://support.google.com/merchants/answer/11909930).
	//
	// Possible values:
	//   "PAUSE_UNSPECIFIED" - The pause is unspecified.
	//   "ADS" - You’re currently pausing your product for all ads locations
	// (including Shopping Ads, Display Ads, and local inventory ads).
	//   "ALL" - You’re currently pausing your product for all Shopping locations
	// (including Shopping Ads, Display Ads, local inventory ads, Buy on Google,
	// and free listings).
	Pause string `json:"pause,omitempty"`
	// PickupMethod: The pickup
	// (https://support.google.com/merchants/answer/14634021) option for the item.
	//
	// Possible values:
	//   "PICKUP_METHOD_UNSPECIFIED" - Pickup method is not specified.
	//   "NOT_SUPPORTED" - The item is not available for store pickup.
	//   "BUY" - The entire transaction occurs online.
	//   "RESERVE" - The item is reserved online and the transaction occurs
	// in-store.
	//   "SHIP_TO_STORE" - The item is purchased online and shipped to a local
	// store for the customer to pick up.
	PickupMethod string `json:"pickupMethod,omitempty"`
	// PickupSla: Item store pickup timeline. For more information, see Pickup SLA
	// (https://support.google.com/merchants/answer/14635400).
	//
	// Possible values:
	//   "PICKUP_SLA_UNSPECIFIED" - Pickup SLA is not specified.
	//   "SAME_DAY" - Indicates that the product is available for pickup the same
	// day that the order is placed, subject to cut off times.
	//   "NEXT_DAY" - Indicates that the product is available for pickup the
	// following day that the order is placed.
	//   "TWO_DAY" - Indicates that the product will be shipped to a store for a
	// customer to pick up in 2 days.
	//   "THREE_DAY" - Indicates that the product will be shipped to a store for a
	// customer to pick up in 3 days.
	//   "FOUR_DAY" - Indicates that the product will be shipped to a store for a
	// customer to pick up in 4 days
	//   "FIVE_DAY" - Indicates that the product will be shipped to a store for a
	// customer to pick up in 5 days.
	//   "SIX_DAY" - Indicates that the product will be shipped to a store for a
	// customer to pick up in 6 days.
	//   "MULTI_WEEK" - Indicates that the product will be shipped to a store for a
	// customer to pick up in one week or more.
	PickupSla string `json:"pickupSla,omitempty"`
	// Price: Price of the item.
	Price *Price `json:"price,omitempty"`
	// ProductDetails: Technical specification or additional product details.
	ProductDetails []*ProductDetail `json:"productDetails,omitempty"`
	// ProductHeight: The height of the product in the units provided. The value
	// must be between 0 (exclusive) and 3000 (inclusive).
	ProductHeight *ProductDimension `json:"productHeight,omitempty"`
	// ProductHighlights: Bullet points describing the most relevant product
	// highlights (https://support.google.com/merchants/answer/9216100).
	ProductHighlights []string `json:"productHighlights,omitempty"`
	// ProductLength: The length of the product in the units provided. The value
	// must be between 0 (exclusive) and 3000 (inclusive).
	ProductLength *ProductDimension `json:"productLength,omitempty"`
	// ProductTypes: Categories of the item (formatted as in product data
	// specification
	// (https://support.google.com/merchants/answer/7052112#product_category)).
	ProductTypes []string `json:"productTypes,omitempty"`
	// ProductWeight: The weight of the product in the units provided. The value
	// must be between 0 (exclusive) and 2000 (inclusive).
	ProductWeight *ProductWeight `json:"productWeight,omitempty"`
	// ProductWidth: The width of the product in the units provided. The value must
	// be between 0 (exclusive) and 3000 (inclusive).
	ProductWidth *ProductDimension `json:"productWidth,omitempty"`
	// PromotionIds: The unique ID of a promotion.
	PromotionIds []string `json:"promotionIds,omitempty"`
	// SalePrice: Advertised sale price of the item.
	SalePrice *Price `json:"salePrice,omitempty"`
	// SalePriceEffectiveDate: Date range during which the item is on sale, see
	// product data specification
	// (https://support.google.com/merchants/answer/7052112#price_and_availability).
	SalePriceEffectiveDate *Interval `json:"salePriceEffectiveDate,omitempty"`
	// SellOnGoogleQuantity: The quantity of the product that is available for
	// selling on Google. Supported only for online products.
	SellOnGoogleQuantity int64 `json:"sellOnGoogleQuantity,omitempty,string"`
	// Shipping: Shipping rules.
	Shipping []*Shipping `json:"shipping,omitempty"`
	// ShippingHandlingBusinessDays: The business days during which orders can be
	// handled. If not provided, Monday to Friday business days will be assumed.
	ShippingHandlingBusinessDays []*ShippingBusinessDaysConfig `json:"shippingHandlingBusinessDays,omitempty"`
	// ShippingHeight: Height of the item for shipping.
	ShippingHeight *ShippingDimension `json:"shippingHeight,omitempty"`
	// ShippingLabel: The shipping label of the product, used to group product in
	// account-level shipping rules.
	ShippingLabel string `json:"shippingLabel,omitempty"`
	// ShippingLength: Length of the item for shipping.
	ShippingLength *ShippingDimension `json:"shippingLength,omitempty"`
	// ShippingTransitBusinessDays: The business days during which orders are in
	// transit. If not provided, Monday to Friday business days will be assumed.
	ShippingTransitBusinessDays []*ShippingBusinessDaysConfig `json:"shippingTransitBusinessDays,omitempty"`
	// ShippingWeight: Weight of the item for shipping.
	ShippingWeight *ShippingWeight `json:"shippingWeight,omitempty"`
	// ShippingWidth: Width of the item for shipping.
	ShippingWidth *ShippingDimension `json:"shippingWidth,omitempty"`
	// ShoppingAdsExcludedCountries: List of country codes (ISO 3166-1 alpha-2)
	// (https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2) to exclude the offer from
	// Shopping Ads destination. Countries from this list are removed from
	// countries configured in data source settings.
	ShoppingAdsExcludedCountries []string `json:"shoppingAdsExcludedCountries,omitempty"`
	// Size: Size of the item. Only one value is allowed. For variants with
	// different sizes, insert a separate product for each size with the same
	// `itemGroupId` value, see Size
	// (https://support.google.com/merchants/answer/6324492).
	Size string `json:"size,omitempty"`
	// SizeSystem: System in which the size is specified. Recommended for apparel
	// items. For more information, see Size system
	// (https://support.google.com/merchants/answer/6324502).
	//
	// Possible values:
	//   "SIZE_SYSTEM_UNSPECIFIED" - Unspecified size system.
	//   "AU" - AU.
	//   "BR" - BR.
	//   "CN" - CN.
	//   "DE" - DE.
	//   "EU" - EU.
	//   "FR" - FR.
	//   "IT" - IT.
	//   "JP" - JP.
	//   "MEX" - MEX.
	//   "UK" - UK.
	//   "US" - US.
	SizeSystem string `json:"sizeSystem,omitempty"`
	// SizeTypes: The cut of the item. It can be used to represent combined size
	// types for apparel items. Maximum two of size types can be provided, see Size
	// type (https://support.google.com/merchants/answer/6324497).
	//
	// Possible values:
	//   "SIZE_TYPE_UNSPECIFIED" - The size type is not specified.
	//   "REGULAR" - Regular size.
	//   "PETITE" - Petite size.
	//   "MATERNITY" - Maternity size.
	//   "BIG" - Big size.
	//   "TALL" - Tall size.
	//   "PLUS" - Plus size.
	SizeTypes []string `json:"sizeTypes,omitempty"`
	// StructuredDescription: Structured description, for algorithmically
	// (AI)-generated descriptions.
	StructuredDescription *StructuredDescription `json:"structuredDescription,omitempty"`
	// StructuredTitle: Structured title, for algorithmically (AI)-generated
	// titles.
	StructuredTitle *StructuredTitle `json:"structuredTitle,omitempty"`
	// SubscriptionCost: Number of periods (weeks, months or years) and amount of
	// payment per period for an item with an associated subscription contract.
	SubscriptionCost *SubscriptionCost `json:"subscriptionCost,omitempty"`
	// SustainabilityIncentives: The list of sustainability incentive programs.
	SustainabilityIncentives []*ProductSustainabilityIncentive `json:"sustainabilityIncentives,omitempty"`
	// Title: Title of the item.
	Title string `json:"title,omitempty"`
	// TransitTimeLabel: The transit time label of the product, used to group
	// product in account-level transit time tables.
	TransitTimeLabel string `json:"transitTimeLabel,omitempty"`
	// UnitPricingBaseMeasure: The preference of the denominator of the unit price.
	UnitPricingBaseMeasure *UnitPricingBaseMeasure `json:"unitPricingBaseMeasure,omitempty"`
	// UnitPricingMeasure: The measure and dimension of an item.
	UnitPricingMeasure *UnitPricingMeasure `json:"unitPricingMeasure,omitempty"`
	// VirtualModelLink: URL of the 3D image of the item. See the Help Center
	// article (https://support.google.com/merchants/answer/13674896) for more
	// information.
	VirtualModelLink string `json:"virtualModelLink,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalImageLinks") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalImageLinks") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductAttributes) MarshalJSON() ([]byte, error) {
	type NoMethod ProductAttributes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductAttributes) UnmarshalJSON(data []byte) error {
	type NoMethod ProductAttributes
	var s1 struct {
		DisplayAdsValue gensupport.JSONFloat64 `json:"displayAdsValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.DisplayAdsValue = float64(s1.DisplayAdsValue)
	return nil
}

// ProductCertification: Product certification
// (https://support.google.com/merchants/answer/13528839), initially introduced
// for EU energy efficiency labeling compliance using the EU EPREL database.
type ProductCertification struct {
	// CertificationAuthority: The certification authority.
	//
	// Possible values:
	//   "CERTIFICATION_AUTHORITY_UNSPECIFIED" - Certification authority is not
	// specified.
	//   "ADEME" - For the French CO2 emissions class for vehicles.
	//   "BMWK" - For the German CO2 emissions classes for vehicles.
	//   "EPA" - Environment Protection Agency.
	//   "EC" - European Commission for energy labels in the EU.
	CertificationAuthority string `json:"certificationAuthority,omitempty"`
	// CertificationCode: The certification code. Maximum length is 2000
	// characters.
	CertificationCode string `json:"certificationCode,omitempty"`
	// CertificationName: The name of the certification.
	//
	// Possible values:
	//   "CERTIFICATION_NAME_UNSPECIFIED" - Certification name is not specified.
	//   "ENERGY_STAR" - Energy Star.
	//   "ENERGY_STAR_MOST_EFFICIENT" - Energy Star Most Efficient.
	//   "EPREL" - Represents energy efficiency certifications in the EU European
	// Registry for Energy Labeling (EPREL) database.
	//   "EU_ECOLABEL" - EU Ecolabel.
	//   "VEHICLE_ENERGY_EFFICIENCY" - The overall CO2 class of a vehicle
	//   "VEHICLE_ENERGY_EFFICIENCY_DISCHARGED_BATTERY" - For the CO2 class of a
	// vehicle with a discharged battery.
	CertificationName string `json:"certificationName,omitempty"`
	// CertificationValue: The certification value (also known as class, level or
	// grade), for example "A+", "C", "gold". Maximum length is 2000 characters.
	CertificationValue string `json:"certificationValue,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CertificationAuthority") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CertificationAuthority") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductCertification) MarshalJSON() ([]byte, error) {
	type NoMethod ProductCertification
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductChange: The change that happened to the product including old value,
// new value, country code as the region code and reporting context.
type ProductChange struct {
	// NewValue: The new value of the changed resource or attribute. If empty, it
	// means that the product was deleted. Will have one of these values :
	// (`approved`, `pending`, `disapproved`, ``)
	NewValue string `json:"newValue,omitempty"`
	// OldValue: The old value of the changed resource or attribute. If empty, it
	// means that the product was created. Will have one of these values :
	// (`approved`, `pending`, `disapproved`, ``)
	OldValue string `json:"oldValue,omitempty"`
	// RegionCode: Countries that have the change (if applicable). Represented in
	// the ISO 3166 format.
	RegionCode string `json:"regionCode,omitempty"`
	// ReportingContext: Reporting contexts that have the change (if applicable).
	// Currently this field supports only (`SHOPPING_ADS`, `LOCAL_INVENTORY_ADS`,
	// `YOUTUBE_SHOPPING`, `YOUTUBE_CHECKOUT`, `YOUTUBE_AFFILIATE`) from the enum
	// value ReportingContextEnum
	// (/merchant/api/reference/rest/Shared.Types/ReportingContextEnum)
	//
	// Possible values:
	//   "REPORTING_CONTEXT_ENUM_UNSPECIFIED" - Not specified.
	//   "SHOPPING_ADS" - [Shopping
	// ads](https://support.google.com/merchants/answer/6149970).
	//   "DISCOVERY_ADS" - Deprecated: Use `DEMAND_GEN_ADS` instead. [Discovery and
	// Demand Gen ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS" - [Demand Gen
	// ads](https://support.google.com/merchants/answer/13389785).
	//   "DEMAND_GEN_ADS_DISCOVER_SURFACE" - [Demand Gen ads on Discover
	// surface](https://support.google.com/merchants/answer/13389785).
	//   "VIDEO_ADS" - [Video
	// ads](https://support.google.com/google-ads/answer/6340491).
	//   "DISPLAY_ADS" - [Display
	// ads](https://support.google.com/merchants/answer/6069387).
	//   "LOCAL_INVENTORY_ADS" - [Local inventory
	// ads](https://support.google.com/merchants/answer/3271956).
	//   "VEHICLE_INVENTORY_ADS" - [Vehicle inventory
	// ads](https://support.google.com/merchants/answer/11544533).
	//   "FREE_LISTINGS" - [Free product
	// listings](https://support.google.com/merchants/answer/9199328).
	//   "FREE_LOCAL_LISTINGS" - [Free local product
	// listings](https://support.google.com/merchants/answer/9825611).
	//   "FREE_LOCAL_VEHICLE_LISTINGS" - [Free local vehicle
	// listings](https://support.google.com/merchants/answer/11544533).
	//   "YOUTUBE_AFFILIATE" - [Youtube
	// Affiliate](https://support.google.com/youtube/answer/13376398).
	//   "YOUTUBE_SHOPPING" - [YouTube
	// Shopping](https://support.google.com/merchants/answer/13478370).
	//   "CLOUD_RETAIL" - [Cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "LOCAL_CLOUD_RETAIL" - [Local cloud
	// retail](https://cloud.google.com/solutions/retail).
	//   "PRODUCT_REVIEWS" - [Product
	// Reviews](https://support.google.com/merchants/answer/14620732).
	//   "MERCHANT_REVIEWS" - [Merchant
	// Reviews](https://developers.google.com/merchant-review-feeds).
	//   "YOUTUBE_CHECKOUT" - YouTube Checkout .
	ReportingContext string `json:"reportingContext,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NewValue") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NewValue") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductChange) MarshalJSON() ([]byte, error) {
	type NoMethod ProductChange
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductDetail: The product details.
type ProductDetail struct {
	// AttributeName: The name of the product detail.
	AttributeName string `json:"attributeName,omitempty"`
	// AttributeValue: The value of the product detail.
	AttributeValue string `json:"attributeValue,omitempty"`
	// SectionName: The section header used to group a set of product details.
	SectionName string `json:"sectionName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AttributeName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AttributeName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductDetail) MarshalJSON() ([]byte, error) {
	type NoMethod ProductDetail
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductDimension: The dimension of the product.
type ProductDimension struct {
	// Unit: Required. The dimension units. Acceptable values are: * "in" *
	// "cm"
	Unit string `json:"unit,omitempty"`
	// Value: Required. The dimension value represented as a number. The value can
	// have a maximum precision of four decimal places.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductDimension) MarshalJSON() ([]byte, error) {
	type NoMethod ProductDimension
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductDimension) UnmarshalJSON(data []byte) error {
	type NoMethod ProductDimension
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// ProductInput: This resource represents input data you submit for a product,
// not the processed product that you see in Merchant Center, in Shopping ads,
// or across Google surfaces. Product inputs, rules and supplemental data
// source data are combined to create the processed Product. For more
// information, see Manage products (/merchant/api/guides/products/overview).
// Required product input attributes to pass data validation checks are
// primarily defined in the Products Data Specification
// (https://support.google.com/merchants/answer/188494). The following
// attributes are required: feedLabel, contentLanguage and offerId. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the processed product can be retrieved. All fields in the
// product input and its sub-messages match the English name of their
// corresponding attribute in the Products Data Specification
// (https://support.google.com/merchants/answer/188494) with some exceptions
// (https://support.google.com/merchants/answer/7052112). The following
// reference documentation lists the field names in the **camelCase** casing
// style while the Products Data Specification lists the names in the
// **snake_case** casing style.
type ProductInput struct {
	// ContentLanguage: Required. Immutable. The two-letter ISO 639-1
	// (http://en.wikipedia.org/wiki/ISO_639-1) language code for the product.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// CustomAttributes: Optional. A list of custom (merchant-provided) attributes.
	// It can also be used for submitting any attribute of the data specification
	// in its generic form (for example, `{ "name": "size type", "value": "regular"
	// }`). This is useful for submitting attributes not explicitly exposed by the
	// API. Maximum allowed number of characters for each custom attribute is 10240
	// (represents sum of characters for name and value). Maximum 2500 custom
	// attributes can be set per product, with total size of 102.4kB. Underscores
	// in custom attribute names are replaced by spaces upon insertion.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// FeedLabel: Required. Immutable. The feed label that lets you categorize and
	// identify your products. The maximum allowed characters are 20, and the
	// supported characters are `A-Z`, `0-9`, hyphen, and underscore. The feed
	// label must not include any spaces. For more information, see Using feed
	// labels (//support.google.com/merchants/answer/14994087).
	FeedLabel string `json:"feedLabel,omitempty"`
	// LegacyLocal: Immutable. Determines whether the product is **only** targeting
	// local destinations and whether the product name should be distinguished with
	// a `local~` prefix. For example,
	// `accounts/123/productInputs/local~en~US~sku123`. If a product that is not
	// `legacy_local` is already targeting local destinations, creating a
	// `legacy_local` product with an otherwise matching name will fail.
	LegacyLocal bool `json:"legacyLocal,omitempty"`
	// Name: Identifier. The name of the product. Format:
	// `accounts/{account}/productInputs/{productinput}` The {productinput} segment
	// is a unique identifier for the product. This identifier must be unique
	// within a merchant account and generally follows the structure:
	// `content_language~feed_label~offer_id`. Example: `en~US~sku123` For legacy
	// local products, the structure is:
	// `local~content_language~feed_label~offer_id`. Example: `local~en~US~sku123`
	// The format of the {productinput} segment in the URL is automatically
	// detected by the server, supporting two options: 1. **Encoded Format**: The
	// `{productinput}` segment is an unpadded base64url encoded string (RFC 4648
	// Section 5). The decoded string must result in the
	// `content_language~feed_label~offer_id` structure. This encoding MUST be used
	// if any part of the product identifier (like `offer_id`) contains characters
	// such as `/`, `%`, or `~`. * Example: To represent the product ID
	// `en~US~sku/123`, the `{productinput}` segment must be the base64url encoding
	// of this string, which is `ZW5-VVMtc2t1LzEyMw`. The full resource name for
	// the product would be `accounts/123/productinputs/ZW5-VVMtc2t1LzEyMw`. 2.
	// **Plain Format**: The `{productinput}` segment is the tilde-separated string
	// `content_language~feed_label~offer_id`. This format is suitable only when
	// `content_language`, `feed_label`, and `offer_id` do not contain
	// URL-problematic characters like `/`, `%`, or `~`. We recommend using the
	// **Encoded Format** for all product IDs to ensure correct parsing, especially
	// those containing special characters. The presence of tilde (`~`) characters
	// in the `{productinput}` segment is used to differentiate between the two
	// formats. Note: For calls to the v1beta version, the plain format is
	// `channel~content_language~feed_label~offer_id`, for example:
	// `accounts/123/productinputs/online~en~US~sku123`.
	Name string `json:"name,omitempty"`
	// OfferId: Required. Immutable. Your unique identifier for the product. This
	// is the same for the product input and processed product. Leading and
	// trailing whitespaces are stripped and multiple whitespaces are replaced by a
	// single whitespace upon submission. See the products data specification
	// (https://support.google.com/merchants/answer/188494#id) for details.
	OfferId string `json:"offerId,omitempty"`
	// Product: Output only. The name of the processed product. Format:
	// `accounts/{account}/products/{product}`
	Product string `json:"product,omitempty"`
	// ProductAttributes: Optional. A list of strongly-typed product attributes.
	ProductAttributes *ProductAttributes `json:"productAttributes,omitempty"`
	// VersionNumber: Optional. Immutable. Represents the existing version
	// (freshness) of the product, which can be used to preserve the right order
	// when multiple updates are done at the same time. If set, the insertion is
	// prevented when version number is lower than the current version number of
	// the existing product. Re-insertion (for example, product refresh after 30
	// days) can be performed with the current `version_number`. Only supported for
	// insertions into primary data sources. Do not set this field for updates. Do
	// not set this field for insertions into supplemental data sources. If the
	// operation is prevented, the aborted exception will be thrown.
	VersionNumber int64 `json:"versionNumber,omitempty,string"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ContentLanguage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ContentLanguage") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductInput) MarshalJSON() ([]byte, error) {
	type NoMethod ProductInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductInstallment: A message that represents installment.
type ProductInstallment struct {
	// Amount: The amount the buyer has to pay per month.
	Amount *Price `json:"amount,omitempty"`
	// CreditType: Type of installment payments.
	//
	// Possible values:
	//   "CREDIT_TYPE_UNSPECIFIED" - Default value. This value is unused.
	//   "FINANCE" - Finance.
	//   "LEASE" - Lease.
	CreditType string `json:"creditType,omitempty"`
	// Downpayment: The up-front down payment amount the buyer has to pay.
	Downpayment *Price `json:"downpayment,omitempty"`
	// Months: The number of installments the buyer has to pay.
	Months int64 `json:"months,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Amount") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Amount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductInstallment) MarshalJSON() ([]byte, error) {
	type NoMethod ProductInstallment
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStatus: The status of a product, data validation issues, that is,
// information about a product computed asynchronously.
type ProductStatus struct {
	// CreationDate: Date on which the item has been created, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	CreationDate string `json:"creationDate,omitempty"`
	// DestinationStatuses: The intended destinations for the product.
	DestinationStatuses []*DestinationStatus `json:"destinationStatuses,omitempty"`
	// GoogleExpirationDate: Date on which the item expires, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	GoogleExpirationDate string `json:"googleExpirationDate,omitempty"`
	// ItemLevelIssues: A list of all issues associated with the product.
	ItemLevelIssues []*ItemLevelIssue `json:"itemLevelIssues,omitempty"`
	// LastUpdateDate: Date on which the item has been last updated, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	LastUpdateDate string `json:"lastUpdateDate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CreationDate") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreationDate") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatus) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductStatusChangeMessage: The message that the merchant will receive to
// notify about product status change event
type ProductStatusChangeMessage struct {
	// Account: The target account that owns the entity that changed. Format :
	// `accounts/{merchant_id}`
	Account string `json:"account,omitempty"`
	// Attribute: The attribute in the resource that changed, in this case it will
	// be always `Status`.
	//
	// Possible values:
	//   "ATTRIBUTE_UNSPECIFIED" - Unspecified attribute
	//   "STATUS" - Status of the changed entity
	Attribute string `json:"attribute,omitempty"`
	// Changes: A message to describe the change that happened to the product
	Changes []*ProductChange `json:"changes,omitempty"`
	// EventTime: The time at which the event was generated. If you want to order
	// the notification messages you receive you should rely on this field not on
	// the order of receiving the notifications.
	EventTime string `json:"eventTime,omitempty"`
	// ExpirationTime: Optional. The product expiration time. This field will not
	// be set if the notification is sent for a product deletion event.
	ExpirationTime string `json:"expirationTime,omitempty"`
	// ManagingAccount: The account that manages the merchant's account. can be the
	// same as merchant id if it is standalone account. Format :
	// `accounts/{service_provider_id}`
	ManagingAccount string `json:"managingAccount,omitempty"`
	// Resource: The product name. Format: `accounts/{account}/products/{product}`
	Resource string `json:"resource,omitempty"`
	// ResourceId: The product id.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceType: The resource that changed, in this case it will always be
	// `Product`.
	//
	// Possible values:
	//   "RESOURCE_UNSPECIFIED" - Unspecified resource
	//   "PRODUCT" - Resource type : product
	ResourceType string `json:"resourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Account") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Account") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductStatusChangeMessage) MarshalJSON() ([]byte, error) {
	type NoMethod ProductStatusChangeMessage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductSustainabilityIncentive: Information regarding sustainability-related
// incentive programs such as rebates or tax relief.
type ProductSustainabilityIncentive struct {
	// Amount: The fixed amount of the incentive.
	Amount *Price `json:"amount,omitempty"`
	// Percentage: The percentage of the sale price that the incentive is applied
	// to.
	Percentage float64 `json:"percentage,omitempty"`
	// Type: Sustainability incentive program.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Unspecified or unknown sustainability incentive type.
	//   "EV_TAX_CREDIT" - Program offering tax liability reductions for electric
	// vehicles and, in some countries, plug-in hybrids. These reductions can be
	// based on a specific amount or a percentage of the sale price.
	//   "EV_PRICE_DISCOUNT" - A subsidy program, often called an environmental
	// bonus, provides a purchase grant for electric vehicles and, in some
	// countries, plug-in hybrids. The grant amount may be a fixed sum or a
	// percentage of the sale price.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Amount") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Amount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductSustainabilityIncentive) MarshalJSON() ([]byte, error) {
	type NoMethod ProductSustainabilityIncentive
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductSustainabilityIncentive) UnmarshalJSON(data []byte) error {
	type NoMethod ProductSustainabilityIncentive
	var s1 struct {
		Percentage gensupport.JSONFloat64 `json:"percentage"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Percentage = float64(s1.Percentage)
	return nil
}

// ProductWeight: The weight of the product.
type ProductWeight struct {
	// Unit: Required. The weight unit. Acceptable values are: * "g" * "kg" *
	// "oz" * "lb"
	Unit string `json:"unit,omitempty"`
	// Value: Required. The weight represented as a number. The weight can have a
	// maximum precision of four decimal places.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductWeight) MarshalJSON() ([]byte, error) {
	type NoMethod ProductWeight
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductWeight) UnmarshalJSON(data []byte) error {
	type NoMethod ProductWeight
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// Shipping: The Shipping of the product.
type Shipping struct {
	// Country: The CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) of the
	// country to which an item will ship.
	Country string `json:"country,omitempty"`
	// HandlingCutoffTime: The handling cutoff time until which an order has to be
	// placed to be processed in the same day. This is a string in format of HHMM
	// (e.g. `1530`) for 3:30 PM. If not configured, the cutoff time will be
	// defaulted to 8AM PST and `handling_cutoff_timezone` will be ignored.
	HandlingCutoffTime string `json:"handlingCutoffTime,omitempty"`
	// HandlingCutoffTimezone: Timezone identifier
	// (https://developers.google.com/adwords/api/docs/appendix/codes-formats#timezone-ids)
	// For example `Europe/Zurich`. This field only applies if
	// `handling_cutoff_time` is set. If `handling_cutoff_time` is set but this
	// field is not set, the shipping destination timezone will be used. If both
	// fields are not set, the handling cutoff time will default to 8AM PST.
	HandlingCutoffTimezone string `json:"handlingCutoffTimezone,omitempty"`
	// LocationGroupName: The location where the shipping is applicable,
	// represented by a location group name.
	LocationGroupName string `json:"locationGroupName,omitempty"`
	// LocationId: The numeric ID of a location that the shipping rate applies to
	// as defined in the AdWords API
	// (https://developers.google.com/adwords/api/docs/appendix/geotargeting).
	LocationId int64 `json:"locationId,omitempty,string"`
	// MaxHandlingTime: Maximum handling time (inclusive) between when the order is
	// received and shipped in business days. 0 means that the order is shipped on
	// the same day as it is received if it happens before the cut-off time. Both
	// maxHandlingTime and maxTransitTime are required if providing shipping
	// speeds. minHandlingTime is optional if maxHandlingTime is present.
	MaxHandlingTime int64 `json:"maxHandlingTime,omitempty,string"`
	// MaxTransitTime: Maximum transit time (inclusive) between when the order has
	// shipped and when it is delivered in business days. 0 means that the order is
	// delivered on the same day as it ships. Both maxHandlingTime and
	// maxTransitTime are required if providing shipping speeds. minTransitTime is
	// optional if maxTransitTime is present.
	MaxTransitTime int64 `json:"maxTransitTime,omitempty,string"`
	// MinHandlingTime: Minimum handling time (inclusive) between when the order is
	// received and shipped in business days. 0 means that the order is shipped on
	// the same day as it is received if it happens before the cut-off time.
	// minHandlingTime can only be present together with maxHandlingTime; but it is
	// not required if maxHandlingTime is present.
	MinHandlingTime int64 `json:"minHandlingTime,omitempty,string"`
	// MinTransitTime: Minimum transit time (inclusive) between when the order has
	// shipped and when it is delivered in business days. 0 means that the order is
	// delivered on the same day as it ships. minTransitTime can only be present
	// together with maxTransitTime; but it is not required if maxTransitTime is
	// present.
	MinTransitTime int64 `json:"minTransitTime,omitempty,string"`
	// PostalCode: The postal code range that the shipping rate applies to,
	// represented by a postal code, a postal code prefix followed by a * wildcard,
	// a range between two postal codes or two postal code prefixes of equal
	// length.
	PostalCode string `json:"postalCode,omitempty"`
	// Price: Fixed shipping price, represented as a number.
	Price *Price `json:"price,omitempty"`
	// Region: The geographic region to which a shipping rate applies. See region
	// (https://support.google.com/merchants/answer/6324484) for more information.
	Region string `json:"region,omitempty"`
	// Service: A free-form description of the service class or delivery speed.
	Service string `json:"service,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Country") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Country") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Shipping) MarshalJSON() ([]byte, error) {
	type NoMethod Shipping
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ShippingBusinessDaysConfig: The business days during which orders are on
// their path to fulfillment. If not provided, Monday to Friday business days
// will be assumed.
type ShippingBusinessDaysConfig struct {
	// BusinessDays: Effective days of the week considered for the delivery time
	// calculation. May not be empty. The more business days included the faster
	// the delivery. Can be set through individual days (e.g. `MTWRF`), or day
	// ranges (e.g. `Mon-Fri`). For more information about accepted formats, see
	// Shipping handling business days
	// (https://support.google.com/merchants/answer/16072859).
	BusinessDays string `json:"businessDays,omitempty"`
	// Country: The CLDR territory code
	// (http://www.unicode.org/repos/cldr/tags/latest/common/main/en.xml) of the
	// country to which an item will ship.
	Country string `json:"country,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BusinessDays") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BusinessDays") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ShippingBusinessDaysConfig) MarshalJSON() ([]byte, error) {
	type NoMethod ShippingBusinessDaysConfig
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ShippingDimension: The ShippingDimension of the product.
type ShippingDimension struct {
	// Unit: The unit of value.
	Unit string `json:"unit,omitempty"`
	// Value: The dimension of the product used to calculate the shipping cost of
	// the item.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ShippingDimension) MarshalJSON() ([]byte, error) {
	type NoMethod ShippingDimension
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ShippingDimension) UnmarshalJSON(data []byte) error {
	type NoMethod ShippingDimension
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// ShippingWeight: The ShippingWeight of the product.
type ShippingWeight struct {
	// Unit: The unit of value.
	Unit string `json:"unit,omitempty"`
	// Value: The weight of the product used to calculate the shipping cost of the
	// item.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ShippingWeight) MarshalJSON() ([]byte, error) {
	type NoMethod ShippingWeight
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ShippingWeight) UnmarshalJSON(data []byte) error {
	type NoMethod ShippingWeight
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// StructuredDescription: Structured description, for algorithmically
// (AI)-generated descriptions.
type StructuredDescription struct {
	// Content: The description text Maximum length is 5000 characters
	Content string `json:"content,omitempty"`
	// DigitalSourceType: The digital source type. Following IPTC
	// (https://cv.iptc.org/newscodes/digitalsourcetype).
	//
	// Possible values:
	//   "DIGITAL_SOURCE_TYPE_UNSPECIFIED" - Digital source type is unspecified.
	//   "TRAINED_ALGORITHMIC_MEDIA" - Text created algorithmically using a model
	// derived from sampled content.
	//   "DEFAULT" - Text NOT created algorithmically using a model derived from
	// sampled content (the default)
	DigitalSourceType string `json:"digitalSourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Content") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Content") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StructuredDescription) MarshalJSON() ([]byte, error) {
	type NoMethod StructuredDescription
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StructuredTitle: Structured title, for algorithmically (AI)-generated
// titles.
type StructuredTitle struct {
	// Content: The title text Maximum length is 150 characters
	Content string `json:"content,omitempty"`
	// DigitalSourceType: The digital source type. Following IPTC
	// (https://cv.iptc.org/newscodes/digitalsourcetype).
	//
	// Possible values:
	//   "DIGITAL_SOURCE_TYPE_UNSPECIFIED" - Digital source type is unspecified.
	//   "TRAINED_ALGORITHMIC_MEDIA" - Text created algorithmically using a model
	// derived from sampled content.
	//   "DEFAULT" - Text NOT created algorithmically using a model derived from
	// sampled content (the default)
	DigitalSourceType string `json:"digitalSourceType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Content") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Content") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StructuredTitle) MarshalJSON() ([]byte, error) {
	type NoMethod StructuredTitle
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SubscriptionCost: The SubscriptionCost of the product.
type SubscriptionCost struct {
	// Amount: The amount the buyer has to pay per subscription period.
	Amount *Price `json:"amount,omitempty"`
	// Period: The type of subscription period. Supported values are: * "month" *
	// "year" * "week"
	//
	// Possible values:
	//   "SUBSCRIPTION_PERIOD_UNSPECIFIED" - Indicates that the subscription period
	// is unspecified.
	//   "MONTH" - Indicates that the subscription period is month.
	//   "YEAR" - Indicates that the subscription period is year.
	//   "WEEK" - Indicates that the subscription period is week.
	Period string `json:"period,omitempty"`
	// PeriodLength: The number of subscription periods the buyer has to pay.
	PeriodLength int64 `json:"periodLength,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Amount") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Amount") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SubscriptionCost) MarshalJSON() ([]byte, error) {
	type NoMethod SubscriptionCost
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UnitPricingBaseMeasure: The UnitPricingBaseMeasure of the product.
type UnitPricingBaseMeasure struct {
	// Unit: The unit of the denominator.
	Unit string `json:"unit,omitempty"`
	// Value: The denominator of the unit price.
	Value int64 `json:"value,omitempty,string"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UnitPricingBaseMeasure) MarshalJSON() ([]byte, error) {
	type NoMethod UnitPricingBaseMeasure
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UnitPricingMeasure: The UnitPricingMeasure of the product.
type UnitPricingMeasure struct {
	// Unit: The unit of the measure.
	Unit string `json:"unit,omitempty"`
	// Value: The measure of an item.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UnitPricingMeasure) MarshalJSON() ([]byte, error) {
	type NoMethod UnitPricingMeasure
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *UnitPricingMeasure) UnmarshalJSON(data []byte) error {
	type NoMethod UnitPricingMeasure
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

type AccountsProductInputsDeleteCall struct {
	s          *Service
	nameid     string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a product input from your Merchant Center account. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the processed product can be retrieved.
//
//   - name: The name of the product input to delete. Format:
//     `accounts/{account}/productInputs/{productInput}` The {productInput}
//     segment is a unique identifier for the product. This identifier must be
//     unique within a merchant account and generally follows the structure:
//     `content_language~feed_label~offer_id`. Example: `en~US~sku123` For legacy
//     local products, the structure is:
//     `local~content_language~feed_label~offer_id`. Example:
//     `local~en~US~sku123` The format of the {productInput} segment in the URL
//     is automatically detected by the server, supporting two options: 1.
//     **Encoded Format**: The `{productInput}` segment is an unpadded base64url
//     encoded string (RFC 4648 Section 5). The decoded string must result in the
//     `content_language~feed_label~offer_id` structure. This encoding MUST be
//     used if any part of the product identifier (like `offer_id`) contains
//     characters such as `/`, `%`, or `~`. * Example: To represent the product
//     ID `en~US~sku/123`, the `{productInput}` segment must be the base64url
//     encoding of this string, which is `ZW5-VVMtc2t1LzEyMw`. The full resource
//     name for the product would be
//     `accounts/123/productInputs/ZW5-VVMtc2t1LzEyMw`. 2. **Plain Format**: The
//     `{productInput}` segment is the tilde-separated string
//     `content_language~feed_label~offer_id`. This format is suitable only when
//     `content_language`, `feed_label`, and `offer_id` do not contain
//     URL-problematic characters like `/`, `%`, or `~`. We recommend using the
//     **Encoded Format** for all product IDs to ensure correct parsing,
//     especially those containing special characters. The presence of tilde
//     (`~`) characters in the `{productInput}` segment is used to differentiate
//     between the two formats. Note: For calls to the v1beta version, the plain
//     format is `channel~content_language~feed_label~offer_id`, for example:
//     `accounts/123/productinputs/online~en~US~sku123`.
func (r *AccountsProductInputsService) Delete(nameid string) *AccountsProductInputsDeleteCall {
	c := &AccountsProductInputsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.nameid = nameid
	return c
}

// DataSource sets the optional parameter "dataSource": Required. The primary
// or supplemental data source from which the product input should be deleted.
// Format: `accounts/{account}/dataSources/{datasource}`. For example,
// `accounts/123456/dataSources/104628`.
func (c *AccountsProductInputsDeleteCall) DataSource(dataSource string) *AccountsProductInputsDeleteCall {
	c.urlParams_.Set("dataSource", dataSource)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductInputsDeleteCall) Fields(s ...googleapi.Field) *AccountsProductInputsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductInputsDeleteCall) Context(ctx context.Context) *AccountsProductInputsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductInputsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductInputsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.nameid,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productInputs.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductInputsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProductInputsInsertCall struct {
	s            *Service
	parent       string
	productinput *ProductInput
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Insert: Uploads a product input to your Merchant Center account
// (/merchant/api/guides/products/add-manage#add_a_product). You must have a
// products data source
// (/merchant/api/guides/data-sources/api-sources#create-primary-data-source)
// to be able to insert a product. The unique identifier of the data source is
// passed as a query parameter in the request URL. If a product input with the
// same contentLanguage, offerId, and dataSource already exists, then the
// product input inserted by this method replaces that entry. After inserting,
// updating, or deleting a product input, it may take several minutes before
// the processed product can be retrieved.
//
//   - parent: The account where this product will be inserted. Format:
//     `accounts/{account}`.
func (r *AccountsProductInputsService) Insert(parent string, productinput *ProductInput) *AccountsProductInputsInsertCall {
	c := &AccountsProductInputsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.productinput = productinput
	return c
}

// DataSource sets the optional parameter "dataSource": Required. The primary
// or supplemental product data source name. If the product already exists and
// data source provided is different, then the product will be moved to a new
// data source. For more information, see Create a primary data source
// (/merchant/api/guides/data-sources/api-sources#create-primary-data-source).
// Only API data sources are supported. Format:
// `accounts/{account}/dataSources/{datasource}`. For example,
// `accounts/123456/dataSources/104628`.
func (c *AccountsProductInputsInsertCall) DataSource(dataSource string) *AccountsProductInputsInsertCall {
	c.urlParams_.Set("dataSource", dataSource)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductInputsInsertCall) Fields(s ...googleapi.Field) *AccountsProductInputsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductInputsInsertCall) Context(ctx context.Context) *AccountsProductInputsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductInputsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductInputsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.productinput)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1/{+parent}/productInputs:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productInputs.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *ProductInput.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductInputsInsertCall) Do(opts ...googleapi.CallOption) (*ProductInput, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ProductInput{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProductInputsPatchCall struct {
	s            *Service
	nameid       string
	productinput *ProductInput
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Patch: Updates the existing product input in your Merchant Center account.
// The name of the product input to update is taken from the `name` field
// within the `ProductInput` resource. After inserting, updating, or deleting a
// product input, it may take several minutes before the processed product can
// be retrieved.
//
//   - name: Identifier. The name of the product. Format:
//     `accounts/{account}/productInputs/{productinput}` The {productinput}
//     segment is a unique identifier for the product. This identifier must be
//     unique within a merchant account and generally follows the structure:
//     `content_language~feed_label~offer_id`. Example: `en~US~sku123` For legacy
//     local products, the structure is:
//     `local~content_language~feed_label~offer_id`. Example:
//     `local~en~US~sku123` The format of the {productinput} segment in the URL
//     is automatically detected by the server, supporting two options: 1.
//     **Encoded Format**: The `{productinput}` segment is an unpadded base64url
//     encoded string (RFC 4648 Section 5). The decoded string must result in the
//     `content_language~feed_label~offer_id` structure. This encoding MUST be
//     used if any part of the product identifier (like `offer_id`) contains
//     characters such as `/`, `%`, or `~`. * Example: To represent the product
//     ID `en~US~sku/123`, the `{productinput}` segment must be the base64url
//     encoding of this string, which is `ZW5-VVMtc2t1LzEyMw`. The full resource
//     name for the product would be
//     `accounts/123/productinputs/ZW5-VVMtc2t1LzEyMw`. 2. **Plain Format**: The
//     `{productinput}` segment is the tilde-separated string
//     `content_language~feed_label~offer_id`. This format is suitable only when
//     `content_language`, `feed_label`, and `offer_id` do not contain
//     URL-problematic characters like `/`, `%`, or `~`. We recommend using the
//     **Encoded Format** for all product IDs to ensure correct parsing,
//     especially those containing special characters. The presence of tilde
//     (`~`) characters in the `{productinput}` segment is used to differentiate
//     between the two formats. Note: For calls to the v1beta version, the plain
//     format is `channel~content_language~feed_label~offer_id`, for example:
//     `accounts/123/productinputs/online~en~US~sku123`.
func (r *AccountsProductInputsService) Patch(nameid string, productinput *ProductInput) *AccountsProductInputsPatchCall {
	c := &AccountsProductInputsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.nameid = nameid
	c.productinput = productinput
	return c
}

// DataSource sets the optional parameter "dataSource": Required. The primary
// or supplemental product data source where `data_source` name identifies the
// product input to be updated. Only API data sources are supported. Format:
// `accounts/{account}/dataSources/{datasource}`. For example,
// `accounts/123456/dataSources/104628`.
func (c *AccountsProductInputsPatchCall) DataSource(dataSource string) *AccountsProductInputsPatchCall {
	c.urlParams_.Set("dataSource", dataSource)
	return c
}

// UpdateMask sets the optional parameter "updateMask": The list of product
// attributes to be updated. If the update mask is omitted, then it is treated
// as implied field mask equivalent to all fields that are populated (have a
// non-empty value). Attributes specified in the update mask without a value
// specified in the body will be deleted from the product. Update mask can only
// be specified for top level fields in attributes and custom attributes. To
// specify the update mask for custom attributes you need to add the
// `custom_attribute.` prefix. Providing special "*" value for full product
// replacement is not supported.
func (c *AccountsProductInputsPatchCall) UpdateMask(updateMask string) *AccountsProductInputsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductInputsPatchCall) Fields(s ...googleapi.Field) *AccountsProductInputsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductInputsPatchCall) Context(ctx context.Context) *AccountsProductInputsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductInputsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductInputsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.productinput)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.nameid,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.productInputs.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *ProductInput.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductInputsPatchCall) Do(opts ...googleapi.CallOption) (*ProductInput, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ProductInput{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.productInputs.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProductsGetCall struct {
	s            *Service
	nameid       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves the processed product from your Merchant Center account.
// After inserting, updating, or deleting a product input, it may take several
// minutes before the updated final product can be retrieved.
//
//   - name: The name of the product. Format:
//     `accounts/{account}/products/{product}` The `{product}` segment is a
//     unique identifier for the product. This identifier must be unique within a
//     merchant account and generally follows the structure:
//     `content_language~feed_label~offer_id`. Example: `en~US~sku123` For legacy
//     local products, the structure is:
//     `local~content_language~feed_label~offer_id`. Example:
//     `local~en~US~sku123` The format of the `{product}` segment in the URL is
//     automatically detected by the server, supporting two options: 1. **Encoded
//     Format**: The `{product}` segment is an unpadded base64url encoded string
//     (RFC 4648 Section 5). The decoded string must result in the
//     `content_language~feed_label~offer_id` structure. This encoding MUST be
//     used if any part of the product identifier (like `offer_id`) contains
//     characters such as `/`, `%`, or `~`. * Example: To represent the product
//     ID `en~US~sku/123`, the `{product}` segment must be the base64url encoding
//     of this string, which is `ZW5-VVMtc2t1LzEyMw`. The full resource name for
//     the product would be `accounts/123/products/ZW5-VVMtc2t1LzEyMw`. 2.
//     **Plain Format**: The `{product}` segment is the tilde-separated string
//     `content_language~feed_label~offer_id`. This format is suitable only when
//     `content_language`, `feed_label`, and `offer_id` do not contain
//     URL-problematic characters like `/`, `%`, or `~`. We recommend using the
//     **Encoded Format** for all product IDs to ensure correct parsing,
//     especially those containing special characters. The presence of tilde
//     (`~`) characters in the `{product}` segment is used to differentiate
//     between the two formats. Note: For calls to the v1beta version, the plain
//     format is `channel~content_language~feed_label~offer_id`, for example:
//     `accounts/123/products/online~en~US~sku123`.
func (r *AccountsProductsService) Get(nameid string) *AccountsProductsGetCall {
	c := &AccountsProductsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.nameid = nameid
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductsGetCall) Fields(s ...googleapi.Field) *AccountsProductsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProductsGetCall) IfNoneMatch(entityTag string) *AccountsProductsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductsGetCall) Context(ctx context.Context) *AccountsProductsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.nameid,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.products.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Product.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsProductsGetCall) Do(opts ...googleapi.CallOption) (*Product, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Product{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AccountsProductsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the processed products in your Merchant Center account. The
// response might contain fewer items than specified by `pageSize`. Rely on
// `pageToken` to determine if there are more items to be requested. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the updated processed product can be retrieved.
//
//   - parent: The account to list processed products for. Format:
//     `accounts/{account}`.
func (r *AccountsProductsService) List(parent string) *AccountsProductsListCall {
	c := &AccountsProductsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// products to return. The service may return fewer than this value. The
// maximum value is 1000; values above 1000 will be coerced to 1000. If
// unspecified, the default page size of 25 products will be returned.
func (c *AccountsProductsListCall) PageSize(pageSize int64) *AccountsProductsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListProducts` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListProducts` must
// match the call that provided the page token.
func (c *AccountsProductsListCall) PageToken(pageToken string) *AccountsProductsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsProductsListCall) Fields(s ...googleapi.Field) *AccountsProductsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsProductsListCall) IfNoneMatch(entityTag string) *AccountsProductsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsProductsListCall) Context(ctx context.Context) *AccountsProductsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsProductsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsProductsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "products/v1/{+parent}/products")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "merchantapi.accounts.products.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListProductsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsProductsListCall) Do(opts ...googleapi.CallOption) (*ListProductsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListProductsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "merchantapi.accounts.products.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsProductsListCall) Pages(ctx context.Context, f func(*ListProductsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
