// Copyright (C) 2019 Storj Labs, Inc.
// See LICENSE for copying information.

package macaroon_test

import (
	"testing"

	assert "github.com/stretchr/testify/require"

	"storj.io/common/macaroon"
)

func TestNilMacaroon(t *testing.T) {
	mac, err := macaroon.NewUnrestricted(nil)
	assert.NoError(t, err)
	assert.NotNil(t, mac)
	data := mac.Serialize()
	assert.NotNil(t, data)
	assert.NotEmpty(t, data)
	mac2, err := macaroon.ParseMacaroon(data)
	assert.NoError(t, err)
	assert.NotNil(t, mac2)
	assert.Equal(t, mac, mac2)

	t.Run("Successful add Caveat", func(t *testing.T) {
		mac, err = mac.AddFirstPartyCaveat([]byte("cav1"))
		assert.NotNil(t, mac)
		assert.NoError(t, err)
		assert.Equal(t, len(mac.Caveats()), 1)
	})

	t.Run("Successful serialization", func(t *testing.T) {
		data := mac.Serialize()
		assert.NotNil(t, data)
		assert.NotEmpty(t, data)

		mac2, err := macaroon.ParseMacaroon(data)
		assert.NotNil(t, mac2)
		assert.NoError(t, err)
		assert.Equal(t, mac, mac2)
	})
}

func TestMacaroon(t *testing.T) {
	secret, err := macaroon.NewSecret()
	assert.NoError(t, err)
	assert.NotNil(t, secret)
	assert.Equal(t, len(secret), 32)

	mac, err := macaroon.NewUnrestricted(secret)
	assert.NoError(t, err)
	assert.NotNil(t, mac)

	nonce := mac.Head()
	assert.NotNil(t, nonce)
	assert.Equal(t, len(nonce), 32)

	t.Run("Successful add Caveat", func(t *testing.T) {
		mac, err = mac.AddFirstPartyCaveat([]byte("cav1"))
		assert.NotNil(t, mac)
		assert.NoError(t, err)
		assert.Equal(t, len(mac.Caveats()), 1)
	})

	t.Run("Successful serialization", func(t *testing.T) {
		data := mac.Serialize()
		assert.NotNil(t, data)
		assert.NotEmpty(t, data)

		mac2, err := macaroon.ParseMacaroon(data)
		assert.NotNil(t, mac2)
		assert.NoError(t, err)
		assert.Equal(t, mac, mac2)
	})

	t.Run("Successful Unpack", func(t *testing.T) {
		ok := mac.Validate(secret)
		assert.True(t, ok)

		expectedTails := mac.Tails(secret)
		ok, tails := mac.ValidateAndTails(secret)
		assert.True(t, ok)
		assert.EqualValues(t, expectedTails, tails)

		c := mac.Caveats()
		assert.NotNil(t, c)
		assert.NotEmpty(t, c)
	})
}

func FuzzParseMacaroon(f *testing.F) {
	f.Add([]byte{})
	f.Add([]byte{0x2, 0x2, 0x20, 0xfb, 0x22, 0xe5, 0x50, 0x30, 0x5, 0xca, 0x60, 0x5, 0xc5, 0x4a, 0x5d, 0x5, 0x1c, 0x4c, 0xa0, 0x95, 0x58, 0x45, 0xfe, 0x77, 0x44, 0xd0, 0x11, 0xdd, 0x69, 0x9, 0xa1, 0x46, 0x5, 0x23, 0x6e, 0x0, 0x0, 0x6, 0x20, 0x5b, 0x50, 0x2a, 0xcd, 0xc3, 0x64, 0x69, 0xca, 0xeb, 0xbe, 0xf6, 0xa3, 0x6, 0x74, 0x8f, 0x9c, 0xc3, 0xd, 0x47, 0xfd, 0xd9, 0xd1, 0xd9, 0xb9, 0xd, 0x8d, 0x18, 0xe9, 0xf9, 0x5a, 0x6f, 0x7})
	f.Add([]byte{0x2, 0x2, 0x20, 0x10, 0xaf, 0x1a, 0xc3, 0xd9, 0xc9, 0x73, 0x46, 0x3b, 0x14, 0xab, 0x42, 0x1, 0x45, 0x67, 0x7, 0xb4, 0x8f, 0xdb, 0x67, 0x4b, 0x56, 0xb9, 0xdc, 0x41, 0x3c, 0x11, 0x6, 0x3c, 0xfc, 0xa8, 0xb9, 0x0, 0x2, 0x4, 0x63, 0x61, 0x76, 0x31, 0x0, 0x0, 0x6, 0x20, 0x9f, 0x83, 0x86, 0xe1, 0x24, 0xeb, 0xae, 0xcf, 0xb8, 0x64, 0xf1, 0x6e, 0x76, 0x40, 0x94, 0xd0, 0xee, 0x9e, 0xad, 0x83, 0x7e, 0x9d, 0x32, 0xb8, 0xc1, 0xf8, 0x4f, 0xbd, 0xa4, 0x3f, 0x97, 0x7a})
	f.Add([]byte{0x2, 0x2, 0x20, 0xf7, 0x22, 0x5b, 0xb8, 0x92, 0x8c, 0x62, 0x41, 0x44, 0x2c, 0x15, 0x64, 0xb7, 0xa5, 0xde, 0x1a, 0x9d, 0x78, 0xf6, 0x4b, 0x23, 0xe9, 0xd9, 0x6a, 0xdf, 0x8e, 0x31, 0x67, 0x9c, 0xd0, 0x41, 0xd8, 0x0, 0x2, 0x4, 0x63, 0x61, 0x76, 0x31, 0x0, 0x0, 0x6, 0x20, 0xca, 0xa3, 0x1a, 0xca, 0xc6, 0xcc, 0x46, 0x5e, 0x1d, 0x2a, 0xbe, 0x15, 0xf, 0x1a, 0x5a, 0x7a, 0x44, 0xaa, 0xa8, 0x8b, 0x29, 0xed, 0x74, 0x42, 0x67, 0x79, 0x62, 0xef, 0x82, 0x82, 0x81, 0xad})
	f.Fuzz(func(t *testing.T, data []byte) {
		_, _ = macaroon.ParseMacaroon(data)
	})
}
