# Configuration

The `sqlc` tool is configured via a `sqlc.(yaml|yml)` or `sqlc.json` file. This
file must be in the directory where the `sqlc` command is run.

## Version 2

```yaml
version: "2"
cloud:
  project: "<PROJECT_ID>"
sql:
- schema: "postgresql/schema.sql"
  queries: "postgresql/query.sql"
  engine: "postgresql"
  gen:
    go: 
      package: "authors"
      out: "postgresql"
  database:
    managed: true
  rules:
    - sqlc/db-prepare
- schema: "mysql/schema.sql"
  queries: "mysql/query.sql"
  engine: "mysql"
  gen:
    go:
      package: "authors"
      out: "mysql"
```

### sql

Each mapping in the `sql` collection has the following keys:

- `name`:
  - An human-friendly identifier for this query set. Optional.
- `engine`:
  - One of `postgresql`, `mysql` or `sqlite`.
- `schema`:
  - Directory of SQL migrations or path to single SQL file; or a list of paths.
- `queries`:
  - Directory of SQL queries or path to single SQL file; or a list of paths.
- `codegen`:
  - A collection of mappings to configure code generators. See [codegen](#codegen) for the supported keys.
- `gen`:
  - A mapping to configure built-in code generators. See [gen](#gen) for the supported keys.
- `database`:
  - A mapping to configure database connections. See [database](#database) for the supported keys.
- `rules`:
  - A collection of rule names to run via `sqlc vet`. See [rules](#rules) for configuration options.
- `analyzer`:
  - A mapping to configure query analysis. See [analyzer](#analyzer) for the supported keys.
- `strict_function_checks`
  - If true, return an error if a called SQL function does not exist. Defaults to `false`.
- `strict_order_by`
  - If true, return an error if a order by column is ambiguous. Defaults to `true`.

### codegen

The `codegen` mapping supports the following keys:

- `out`:
  - Output directory for generated code.
- `plugin`:
  - The name of the plugin. Must be defined in the `plugins` collection.
- `options`:
  - A mapping of plugin-specific options.

```yaml
version: '2'
plugins:
- name: py
  wasm:
    url: https://github.com/sqlc-dev/sqlc-gen-python/releases/download/v0.16.0-alpha/sqlc-gen-python.wasm
    sha256: 428476c7408fd4c032da4ec74e8a7344f4fa75e0f98a5a3302f238283b9b95f2
sql:
- schema: "schema.sql"
  queries: "query.sql"
  engine: postgresql
  codegen:
  - out: src/authors
    plugin: py
    options:
      package: authors
      emit_sync_querier: true
      emit_async_querier: true
      query_parameter_limit: 5
```

### database

The `database` mapping supports the following keys:

- `managed`:
  - If true, connect to a [managed database](../howto/managed-databases.md). Defaults to `false`.
- `uri`:
  - Database connection URI

The `uri` string can contain references to environment variables using the `${...}`
syntax. In the following example, the connection string will have the value of
the `PG_PASSWORD` environment variable set as its password.

```yaml
version: '2'
sql:
- schema: schema.sql
  queries: query.sql
  engine: postgresql
  database:
    uri: postgresql://postgres:${PG_PASSWORD}@localhost:5432/authors
  gen:
    go:
      package: authors
      out: postgresql
```

### analyzer

The `analyzer` mapping supports the following keys:

- `database`:
  -  If false, do not use the configured database for query analysis. Defaults to `true`.
  
### gen

The `gen` mapping supports the following keys:

#### go

- `package`:
  - The package name to use for the generated code. Defaults to `out` basename.
- `out`:
  - Output directory for generated code.
- `sql_package`:
  - Either `pgx/v4`, `pgx/v5` or `database/sql`. Defaults to `database/sql`.
- `sql_driver`:
  - Either `github.com/jackc/pgx/v4`, `github.com/jackc/pgx/v5`, `github.com/lib/pq` or `github.com/go-sql-driver/mysql`. No defaults. Required if query annotation `:copyfrom` is used.
- `emit_db_tags`:
  - If true, add DB tags to generated structs. Defaults to `false`.
- `emit_prepared_queries`:
  - If true, include support for prepared queries. Defaults to `false`.
- `emit_interface`:
  - If true, output a `Querier` interface in the generated package. Defaults to `false`.
- `emit_exact_table_names`:
  - If true, struct names will mirror table names. Otherwise, sqlc attempts to singularize plural table names. Defaults to `false`.
- `emit_empty_slices`:
  - If true, slices returned by `:many` queries will be empty instead of `nil`. Defaults to `false`.
- `emit_exported_queries`:
  - If true, autogenerated SQL statement can be exported to be accessed by another package.
- `emit_json_tags`:
  - If true, add JSON tags to generated structs. Defaults to `false`.
- `emit_result_struct_pointers`:
  - If true, query results are returned as pointers to structs. Queries returning multiple results are returned as slices of pointers. Defaults to `false`.
- `emit_params_struct_pointers`:
  - If true, parameters are passed as pointers to structs. Defaults to `false`.
- `emit_methods_with_db_argument`:
  - If true, generated methods will accept a DBTX argument instead of storing a DBTX on the `*Queries` struct. Defaults to `false`.
- `emit_pointers_for_null_types`:
  - If true, generated types for nullable columns are emitted as pointers (ie. `*string`) instead of `database/sql` null types (ie. `NullString`). Currently only supported for PostgreSQL if `sql_package` is `pgx/v4` or `pgx/v5`, and for SQLite. Defaults to `false`.
- `emit_enum_valid_method`:
  - If true, generate a Valid method on enum types,
    indicating whether a string is a valid enum value.
- `emit_all_enum_values`:
  - If true, emit a function per enum type
    that returns all valid enum values.
- `emit_sql_as_comment`:
  - If true, emits the SQL statement as a code-block comment above the generated function, appending to any existing comments. Defaults to `false`.
- `build_tags`:
  - If set, add a `//go:build <build_tags>` directive at the beginning of each generated Go file.
- `initialisms`:
  - An array of [initialisms](https://google.github.io/styleguide/go/decisions.html#initialisms) to upper-case. For example, `app_id` becomes `AppID`. Defaults to `["id"]`.
- `json_tags_id_uppercase`:
  - If true, "Id" in json tags will be uppercase. If false, will be camelcase. Defaults to `false`
- `json_tags_case_style`:
  - `camel` for camelCase, `pascal` for PascalCase, `snake` for snake_case or `none` to use the column name in the DB. Defaults to `none`.
- `omit_unused_structs`:
  - If `true`, sqlc won't generate table and enum structs that aren't used in queries for a given package. Defaults to `false`.
- `output_batch_file_name`:
  - Customize the name of the batch file. Defaults to `batch.go`.
- `output_db_file_name`:
  - Customize the name of the db file. Defaults to `db.go`.
- `output_models_file_name`:
  - Customize the name of the models file. Defaults to `models.go`.
- `output_querier_file_name`:
  - Customize the name of the querier file. Defaults to `querier.go`.
- `output_copyfrom_file_name`:
  - Customize the name of the copyfrom file. Defaults to `copyfrom.go`.
- `output_files_suffix`:
  - If specified the suffix will be added to the name of the generated files.
- `query_parameter_limit`:
  - The number of positional arguments that will be generated for Go functions. To always emit a parameter struct, set this to `0`. Defaults to `1`.
- `rename`:
  - Customize the name of generated struct fields. See [Renaming fields](../howto/rename.md) for usage information.
- `overrides`:
  - It is a collection of definitions that dictates which types are used to map a database types.

##### overrides

See [Overriding types](../howto/overrides.md) for an in-depth guide to using type overrides. Each mapping of the `overrides` collection has the following keys:

- `db_type`:
  - The PostgreSQL or MySQL type to override. Find the full list of supported types in [postgresql_type.go](https://github.com/sqlc-dev/sqlc/blob/main/internal/codegen/golang/postgresql_type.go#L12) or [mysql_type.go](https://github.com/sqlc-dev/sqlc/blob/main/internal/codegen/golang/mysql_type.go#L12). Note that for Postgres you must use the pg_catalog prefixed names where available. Can't be used if the `column` key is defined.
- `column`:
  - In case the type overriding should be done on specific a column of a table instead of a type. `column` should be of the form `table.column` but you can be even more specific by specifying `schema.table.column` or `catalog.schema.table.column`. Can't be used if the `db_type` key is defined.
- `go_type`:
  - A fully qualified name to a Go type to use in the generated code.
- `go_struct_tag`:
  - A reflect-style struct tag to use in the generated code, e.g. `a:"b" x:"y,z"`.
    If you want general json/db tags for all fields, use `emit_db_tags` and/or `emit_json_tags` instead.
- `nullable`:
  - If `true`, use this type when a column is nullable. Defaults to `false`.

For more complicated import paths, the `go_type` can also be an object with the following keys:

- `import`:
  - The import path for the package where the type is defined.
- `package`:
  - The package name where the type is defined. This should only be necessary when your import path doesn't end with the desired package name.
- `type`:
  - The type name itself, without any package prefix.
- `pointer`:
  - If set to `true`, generated code will use pointers to the type rather than the type itself.
- `slice`:
  - If set to `true`, generated code will use a slice of the type rather than the type itself.

#### kotlin

> Removed in v1.17.0 and replaced by the [sqlc-gen-kotlin](https://github.com/sqlc-dev/sqlc-gen-kotlin) plugin. Follow the [migration guide](../guides/migrating-to-sqlc-gen-kotlin) to switch.

- `package`:
  - The package name to use for the generated code.
- `out`:
  - Output directory for generated code.
- `emit_exact_table_names`:
  - If true, use the exact table name for generated models. Otherwise, guess a singular form. Defaults to `false`.

#### python

> Removed in v1.17.0 and replaced by the [sqlc-gen-python](https://github.com/sqlc-dev/sqlc-gen-python) plugin. Follow the [migration guide](../guides/migrating-to-sqlc-gen-python) to switch.

- `package`:
  - The package name to use for the generated code.
- `out`:
  - Output directory for generated code.
- `emit_exact_table_names`:
  - If true, use the exact table name for generated models. Otherwise, guess a singular form. Defaults to `false`.
- `emit_sync_querier`:
  - If true, generate a class with synchronous methods. Defaults to `false`.
- `emit_async_querier`:
  - If true, generate a class with asynchronous methods. Defaults to `false`.
- `emit_pydantic_models`:
  - If true, generate classes that inherit from `pydantic.BaseModel`. Otherwise, define classes using the `dataclass` decorator. Defaults to `false`.

#### json

- `out`:
  - Output directory for the generated JSON.
- `filename`:
  - Filename for the generated JSON document. Defaults to `codegen_request.json`.
- `indent`:
  - Indent string to use in the JSON document. Defaults to `  `.

### plugins

Each mapping in the `plugins` collection has the following keys:

- `name`:
  - The name of this plugin. Required
- `env`
  - A list of environment variables to pass to the plugin. By default, no environment variables are passed.
- `process`: A mapping with a single `cmd` key
  - `cmd`:
    - The executable to call when using this plugin
  - `format`:
    - The format expected. Supports `json` and `protobuf` formats. Defaults to `protobuf`.
- `wasm`: A mapping with a two keys `url` and `sha256`
  - `url`:
    - The URL to fetch the WASM file. Supports the `https://` or `file://` schemes.
  - `sha256`
    - The SHA256 checksum for the downloaded file.
   
```yaml
version: "2"
plugins:
- name: "py"
  wasm: 
    url: "https://github.com/sqlc-dev/sqlc-gen-python/releases/download/v0.16.0-alpha/sqlc-gen-python.wasm"
    sha256: "428476c7408fd4c032da4ec74e8a7344f4fa75e0f98a5a3302f238283b9b95f2"
- name: "js"
  env:
  - PATH
  process: 
    cmd: "sqlc-gen-json"
```

### rules

Each mapping in the `rules` collection has the following keys:

- `name`:
  - The name of this rule. Required
- `rule`:
  - A [Common Expression Language (CEL)](https://github.com/google/cel-spec) expression. Required.
- `message`:
  - An optional message shown when this rule evaluates to `true`.

See the [vet](../howto/vet.md) documentation for a list of built-in rules and
help writing custom rules.
   
```yaml
version: "2"
sql:
  - schema: "query.sql"
    queries: "query.sql"
    engine: "postgresql"
    gen:
      go:
        package: "authors"
        out: "db"
    rules:
      - no-pg
      - no-delete
      - only-one-param
      - no-exec
rules:
  - name: no-pg
    message: "invalid engine: postgresql"
    rule: |
      config.engine == "postgresql"
  - name: no-delete
    message: "don't use delete statements"
    rule: |
      query.sql.contains("DELETE")
  - name: only-one-param
    message: "too many parameters"
    rule: |
      query.params.size() > 1
  - name: no-exec
    message: "don't use exec"
    rule: |
      query.cmd == "exec"
```
  
### Global overrides

Sometimes, the same configuration must be done across various specifications of
code generation.  Then a global definition for type overriding and field
renaming can be done using the `overrides` mapping the following manner:

```yaml
version: "2"
overrides:
  go:
    rename:
      id: "Identifier"
    overrides:
      - db_type: "timestamptz"
        nullable: true
        engine: "postgresql"
        go_type:
          import: "gopkg.in/guregu/null.v4"
          package: "null"
          type: "Time"
sql:
- schema: "postgresql/schema.sql"
  queries: "postgresql/query.sql"
  engine: "postgresql"
  gen:
    go: 
      package: "authors"
      out: "postgresql"
- schema: "mysql/schema.sql"
  queries: "mysql/query.sql"
  engine: "mysql"
  gen:
    go:
      package: "authors"
      out: "mysql"
```

With the previous configuration, whenever a struct field is generated from a
table column that is called `id`, it will generated as `Identifier`.

Also, whenever there is a nullable `timestamp with time zone` column in a
Postgres table, it will be generated as `null.Time`.  Note that the mapping for
global type overrides has a field called `engine` that is absent in the regular
type overrides. This field is only used when there are multiple definitions
using multiple engines. Otherwise, the value of the `engine` key
defaults to the engine that is currently being used.

Currently, type overrides and field renaming, both global and regular, are only
fully supported in Go.

## Version 1

```yaml
version: "1"
packages:
  - name: "db"
    path: "internal/db"
    queries: "./sql/query/"
    schema: "./sql/schema/"
    engine: "postgresql"
    emit_db_tags: false
    emit_prepared_queries: true
    emit_interface: false
    emit_exact_table_names: false
    emit_empty_slices: false
    emit_exported_queries: false
    emit_json_tags: true
    emit_result_struct_pointers: false
    emit_params_struct_pointers: false
    emit_methods_with_db_argument: false
    emit_pointers_for_null_types: false
    emit_enum_valid_method: false
    emit_all_enum_values: false
    build_tags: "some_tag"
    json_tags_case_style: "camel"
    omit_unused_structs: false
    output_batch_file_name: "batch.go"
    output_db_file_name: "db.go"
    output_models_file_name: "models.go"
    output_querier_file_name: "querier.go"
    output_copyfrom_file_name: "copyfrom.go"
    query_parameter_limit: 1
```

### packages

Each mapping in the `packages` collection has the following keys:

- `name`:
  - The package name to use for the generated code. Defaults to `path` basename.
- `path`:
  - Output directory for generated code.
- `queries`:
  - Directory of SQL queries or path to single SQL file; or a list of paths.
- `schema`:
  - Directory of SQL migrations or path to single SQL file; or a list of paths.
- `engine`:
  - Either `postgresql` or `mysql`. Defaults to `postgresql`.
- `sql_package`:
  - Either `pgx/v4`, `pgx/v5` or `database/sql`. Defaults to `database/sql`.
- `emit_db_tags`:
  - If true, add DB tags to generated structs. Defaults to `false`.
- `emit_prepared_queries`:
  - If true, include support for prepared queries. Defaults to `false`.
- `emit_interface`:
  - If true, output a `Querier` interface in the generated package. Defaults to `false`.
- `emit_exact_table_names`:
  - If true, struct names will mirror table names. Otherwise, sqlc attempts to singularize plural table names. Defaults to `false`.
- `emit_empty_slices`:
  - If true, slices returned by `:many` queries will be empty instead of `nil`. Defaults to `false`.
- `emit_exported_queries`:
  - If true, autogenerated SQL statement can be exported to be accessed by another package.
- `emit_json_tags`:
  - If true, add JSON tags to generated structs. Defaults to `false`.
- `emit_result_struct_pointers`:
  - If true, query results are returned as pointers to structs. Queries returning multiple results are returned as slices of pointers. Defaults to `false`.
- `emit_params_struct_pointers`:
  - If true, parameters are passed as pointers to structs. Defaults to `false`.
- `emit_methods_with_db_argument`:
  - If true, generated methods will accept a DBTX argument instead of storing a DBTX on the `*Queries` struct. Defaults to `false`.
- `emit_pointers_for_null_types`:
  - If true and `sql_package` is set to `pgx/v4` or `pgx/v5`, generated types for nullable columns are emitted as pointers (ie. `*string`) instead of `database/sql` null types (ie. `NullString`). Defaults to `false`.
- `emit_enum_valid_method`:
  - If true, generate a Valid method on enum types,
    indicating whether a string is a valid enum value.
- `emit_all_enum_values`:
  - If true, emit a function per enum type
    that returns all valid enum values.
- `build_tags`:
  - If set, add a `//go:build <build_tags>` directive at the beginning of each generated Go file.
- `json_tags_case_style`:
  - `camel` for camelCase, `pascal` for PascalCase, `snake` for snake_case or `none` to use the column name in the DB. Defaults to `none`.
- `omit_unused_structs`:
  - If `true`, sqlc won't generate table and enum structs that aren't used in queries for a given package. Defaults to `false`.
- `output_batch_file_name`:
  - Customize the name of the batch file. Defaults to `batch.go`.
- `output_db_file_name`:
  - Customize the name of the db file. Defaults to `db.go`.
- `output_models_file_name`:
  - Customize the name of the models file. Defaults to `models.go`.
- `output_querier_file_name`:
  - Customize the name of the querier file. Defaults to `querier.go`.
- `output_copyfrom_file_name`:
  - Customize the name of the copyfrom file. Defaults to `copyfrom.go`.
- `output_files_suffix`:
  - If specified the suffix will be added to the name of the generated files.
- `query_parameter_limit`:
  - Positional arguments that will be generated in Go functions (`>= 0`). To always emit a parameter struct, you would need to set it to `0`. Defaults to `1`.

### overrides

The default mapping of PostgreSQL/MySQL types to Go types only uses packages outside
the standard library when it must.

For example, the `uuid` PostgreSQL type is mapped to `github.com/google/uuid`.
If a different Go package for UUIDs is required, specify the package in the
`overrides` array. In this case, I'm going to use the `github.com/gofrs/uuid`
instead.

```yaml
version: "1"
packages: [...]
overrides:
  - go_type: "github.com/gofrs/uuid.UUID"
    db_type: "uuid"
```

Each override document has the following keys:

- `db_type`:
  - The PostgreSQL or MySQL type to override. Find the full list of supported types in [postgresql_type.go](https://github.com/sqlc-dev/sqlc/blob/main/internal/codegen/golang/postgresql_type.go#L12) or [mysql_type.go](https://github.com/sqlc-dev/sqlc/blob/main/internal/codegen/golang/mysql_type.go#L12). Note that for Postgres you must use the pg_catalog prefixed names where available.
- `go_type`:
  - A fully qualified name to a Go type to use in the generated code.
- `go_struct_tag`:
  - A reflect-style struct tag to use in the generated code, e.g. `a:"b" x:"y,z"`.
    If you want general json/db tags for all fields, use `emit_db_tags` and/or `emit_json_tags` instead.
- `nullable`:
  - If true, use this type when a column is nullable. Defaults to `false`.

Note that a single `db_type` override configuration applies to either nullable or non-nullable
columns, but not both. If you want a single `go_type` to override in both cases, you'll
need to specify two overrides.

For more complicated import paths, the `go_type` can also be an object.

```yaml
version: "1"
packages: [...]
overrides:
  - db_type: "uuid"
    go_type:
      import: "a/b/v2"
      package: "b"
      type: "MyType"
```

#### Per-Column Type Overrides

Sometimes you would like to override the Go type used in model or query generation for
a specific field of a table and not on a type basis as described in the previous section.

This may be configured by specifying the `column` property in the override definition. `column`
should be of the form `table.column` but you can be even more specific by specifying `schema.table.column`
or `catalog.schema.table.column`.

```yaml
version: "1"
packages: [...]
overrides:
  - column: "authors.id"
    go_type: "github.com/segmentio/ksuid.KSUID"
```

#### Package Level Overrides

Overrides can be configured globally, as demonstrated in the previous sections, or they can be configured per-package which
scopes the override behavior to just a single package:

```yaml
version: "1"
packages:
  - overrides: [...]
```

### rename

Struct field names are generated from column names using a simple algorithm:
split the column name on underscores and capitalize the first letter of each
part.

```
account     -> Account
spotify_url -> SpotifyUrl
app_id      -> AppID
```

If you're not happy with a field's generated name, use the `rename` mapping
to pick a new name. The keys are column names and the values are the struct
field name to use.

```yaml
version: "1"
packages: [...]
rename:
  spotify_url: "SpotifyURL"
```
